"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkNumberLengthForType = checkNumberLengthForType;
exports["default"] = checkNumberLength;
var _mergeArrays = _interopRequireDefault(require("./mergeArrays.js"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { "default": e }; }
function checkNumberLength(nationalNumber, metadata) {
  return checkNumberLengthForType(nationalNumber, undefined, metadata);
}

// Checks whether a number is possible for the country based on its length.
// Should only be called for the "new" metadata which has "possible lengths".
function checkNumberLengthForType(nationalNumber, type, metadata) {
  var type_info = metadata.type(type);

  // There should always be "<possiblePengths/>" set for every type element.
  // This is declared in the XML schema.
  // For size efficiency, where a sub-description (e.g. fixed-line)
  // has the same "<possiblePengths/>" as the "general description", this is missing,
  // so we fall back to the "general description". Where no numbers of the type
  // exist at all, there is one possible length (-1) which is guaranteed
  // not to match the length of any real phone number.
  var possible_lengths = type_info && type_info.possibleLengths() || metadata.possibleLengths();
  // let local_lengths    = type_info && type.possibleLengthsLocal() || metadata.possibleLengthsLocal()

  // Metadata before version `1.0.18` didn't contain `possible_lengths`.
  if (!possible_lengths) {
    return 'IS_POSSIBLE';
  }
  if (type === 'FIXED_LINE_OR_MOBILE') {
    // No such country in metadata.
    /* istanbul ignore next */
    if (!metadata.type('FIXED_LINE')) {
      // The rare case has been encountered where no fixedLine data is available
      // (true for some non-geographic entities), so we just check mobile.
      return checkNumberLengthForType(nationalNumber, 'MOBILE', metadata);
    }
    var mobile_type = metadata.type('MOBILE');
    if (mobile_type) {
      // Merge the mobile data in if there was any. "Concat" creates a new
      // array, it doesn't edit possible_lengths in place, so we don't need a copy.
      // Note that when adding the possible lengths from mobile, we have
      // to again check they aren't empty since if they are this indicates
      // they are the same as the general desc and should be obtained from there.
      possible_lengths = (0, _mergeArrays["default"])(possible_lengths, mobile_type.possibleLengths());
      // The current list is sorted; we need to merge in the new list and
      // re-sort (duplicates are okay). Sorting isn't so expensive because
      // the lists are very small.

      // if (local_lengths) {
      // 	local_lengths = mergeArrays(local_lengths, mobile_type.possibleLengthsLocal())
      // } else {
      // 	local_lengths = mobile_type.possibleLengthsLocal()
      // }
    }
  }
  // If the type doesn't exist then return 'INVALID_LENGTH'.
  else if (type && !type_info) {
    return 'INVALID_LENGTH';
  }
  var actual_length = nationalNumber.length;

  // In `libphonenumber-js` all "local-only" formats are dropped for simplicity.
  // // This is safe because there is never an overlap beween the possible lengths
  // // and the local-only lengths; this is checked at build time.
  // if (local_lengths && local_lengths.indexOf(nationalNumber.length) >= 0)
  // {
  // 	return 'IS_POSSIBLE_LOCAL_ONLY'
  // }

  var minimum_length = possible_lengths[0];
  if (minimum_length === actual_length) {
    return 'IS_POSSIBLE';
  }
  if (minimum_length > actual_length) {
    return 'TOO_SHORT';
  }
  if (possible_lengths[possible_lengths.length - 1] < actual_length) {
    return 'TOO_LONG';
  }

  // We skip the first element since we've already checked it.
  return possible_lengths.indexOf(actual_length, 1) >= 0 ? 'IS_POSSIBLE' : 'INVALID_LENGTH';
}
//# sourceMappingURL=checkNumberLength.js.map