// import React, { useState, useEffect } from 'react';
// import { Bar, Pie } from 'react-chartjs-2';
// import { Chart as ChartJS, CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend, ArcElement, ChartOptions } from 'chart.js';
// import { ArrowDownCircle, ArrowRight, ArrowUpCircle, PiggyBank, UserCheck, Users, Wallet } from 'lucide-react';

// ChartJS.register(CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend, ArcElement);

// type User = {
//   id: string;
//   name: string;
//   email: string;
//   number: string;
//   total_deposit_usdt: number;
//   total_withdrawal_usdt: number;
// };

// interface DepositTransaction {
//   uid: number | string;
//   date: string;
//   txtid: string;
//   amount: number;
//   coin: string;
// }

// interface LiveBet {
//   game: string;
//   uid: number;
//   amount: number;
//   coin: string;
//   name: string;
// }

// interface WithdrawalTransaction {
//   uid: number | string;
//   date: string;
//   wallet: string;
//   amount: number;
//   coin: string;
//   name: string | null;
// }

// const AdminDashboard = () => {
//   const [userCount, setUserCount] = useState(0);
//   const [totalDeposits, setTotalDeposits] = useState(0);
//   const [totalWithdrawals, setTotalWithdrawals] = useState(0);
//   const [users, setUsers] = useState<User[]>([]);
//   const [todayDeposits, setTodayDeposits] = useState(0);
//   const [todayWithdrawals, setTodayWithdrawals] = useState(0);
//   const [liveBets, setLiveBets] = useState<LiveBet[]>([]);
//   const [showModal, setShowModal] = useState(false);
//   const [transactions, setTransactions] = useState<Array<DepositTransaction | WithdrawalTransaction>>([]);

//   const [chartData1, setChartData] = useState({
//     labels: [],
//     datasets: [{
//       data: [],
//       backgroundColor: 'indigo',
//     }]
//   });

//   const [pieData, setPieData] = useState({
//     labels: ['Crash', 'Mines', 'Other Games'],
//     datasets: [{
//       data: [0, 0, 0],
//       backgroundColor: ['#4338ca', '#818cf8', '#c7d2fe'],
//     }]
//   });

//   useEffect(() => {
//     const fetchUserData = async () => {
//       try {
//         const userResponse = await fetch('https://api.air1bet.com/users');
//         const userData = await userResponse.json();
//         setUsers(userData);
//         setUserCount(userData.length);

//         const totalDepositsResponse = await fetch('https://api.air1bet.com/total-deposits');
//         const totalDepositsData = await totalDepositsResponse.json();
//         setTotalDeposits(totalDepositsData.total_deposits_usdt);

//         const totalWithdrawalsResponse = await fetch('https://api.air1bet.com/total-withdrawals');
//         const totalWithdrawalsData = await totalWithdrawalsResponse.json();
//         setTotalWithdrawals(totalWithdrawalsData.total_withdrawals_usdt);

//       } catch (error) {
//         console.error('Failed to fetch user data:', error);
//       }
//     };

//     const fetchTodayDepositsAndWithdrawals = async () => {
//       try {
//         const today = new Date().toISOString().split('T')[0];

//         const depositResponse = await fetch('https://api.air1bet.com/today-deposits');
//         const depositData = await depositResponse.json();

//         const totalDepositAmountToday = depositData.reduce((total: any, deposit: { date: string; amount: any; }) => {
//           const depositDate = deposit.date.split('T')[0];
//           return depositDate === today ? total + deposit.amount : total;
//         }, 0);
//         setTodayDeposits(totalDepositAmountToday);

//         const withdrawResponse = await fetch('https://api.air1bet.com/today-withdrawals');
//         const withdrawData = await withdrawResponse.json();

//         const totalWithdrawalAmountToday = withdrawData.reduce((total: any, withdraw: { date: string; amount: any; }) => {
//           const withdrawDate = withdraw.date.split('T')[0];
//           return withdrawDate === today ? total + withdraw.amount : total;
//         }, 0);
//         setTodayWithdrawals(totalWithdrawalAmountToday);

//       } catch (error) {
//         console.error('Failed to fetch deposit or withdrawal data:', error);
//       }
//     };

//     fetchTodayDepositsAndWithdrawals();
//     fetchUserData();
//   }, []);

//   const fetchTodayDeposits = async () => {
//     const response = await fetch('https://api.air1bet.com/today-deposits');
//     const data: DepositTransaction[] = await response.json();
//     const today = new Date().toISOString().split('T')[0];
//     const todayDeposits = data.filter(deposit => new Date(deposit.date).toISOString().split('T')[0] === today);
//     setTransactions(todayDeposits);
//   };

//   const fetchTodayWithdrawals = async () => {
//     const response = await fetch('https://api.air1bet.com/today-withdrawals');
//     const data: WithdrawalTransaction[] = await response.json();
//     const today = new Date().toISOString().split('T')[0];
//     const todayWithdrawals = data.filter(withdrawal => new Date(withdrawal.date).toISOString().split('T')[0] === today);
//     setTransactions(todayWithdrawals);
//   };

//   const handleShowTransactions = async (type: 'deposit' | 'withdrawal') => {
//     setShowModal(true);
//     if (type === 'deposit') {
//       await fetchTodayDeposits();
//     } else {
//       await fetchTodayWithdrawals();
//     }
//   };

//   const fetchLiveBets = async () => {
//     try {
//       const response = await fetch('https://api.air1bet.com/live-bets');
//       const data: LiveBet[] = await response.json();
//       setLiveBets(data);
//     } catch (error) {
//       console.error('Failed to fetch live bets:', error);
//     }
//   };

//   useEffect(() => {
//     fetchLiveBets();
//     const interval = setInterval(fetchLiveBets, 10000);
//     return () => clearInterval(interval);
//   }, []);

//   useEffect(() => {
//     // Update chart data when liveBets changes

//     // setChartData({
//     //   labels: Object.keys(gameData),
//     //   datasets: [{
//     //     data: Object.values(gameData),
//     //     backgroundColor: 'indigo',
//     //   }]
//     // });

//     // Update pie chart data
//     const crashBets = liveBets.filter(bet => bet.game === 'crash').reduce((sum, bet) => sum + bet.amount, 0);
//     const mineBets = liveBets.filter(bet => bet.game === 'mine').reduce((sum, bet) => sum + bet.amount, 0);
//     const otherBets = liveBets.filter(bet => bet.game !== 'crash' && bet.game !== 'mine').reduce((sum, bet) => sum + bet.amount, 0);

//     setPieData({
//       labels: ['Crash', 'Mines', 'Other Games'],
//       datasets: [{
//         data: [crashBets, mineBets, otherBets],
//         backgroundColor: ['#4338ca', '#818cf8', '#c7d2fe'],
//       }]
//     });
//   }, [liveBets]);


//   const gameData = {
//     Mine: 400000,
//     Crash: 600000,
//     Plinko: 1480000,
//     High_Low: 700000,
//     Hash_Dice: 100000,
//     SingleKeno: 20000,
//     Hilo: 10000
//   };

//   const chartData = {
//     labels: Object.keys(gameData),
//     datasets: [{
//       data: Object.values(gameData),
//       backgroundColor: '#646ECD', // Indigo color
//       barThickness: 40,
//     }]
//   };

//   const options: ChartOptions<'bar'> = {
//     responsive: true,
//     maintainAspectRatio: false,
//     plugins: {
//       legend: {
//         display: false,
//       },
//       title: {
//         display: true,
//         // text: 'Live Bets by Game',
//         font: {
//           size: 18,
//           weight: 'bold' as const,

//         },
//         padding: {
//           top: 10,
//           bottom: 30,
//         },
//       },
//     },
//     scales: {
//       x: {
//         grid: {
//           display: false,
//         },
//         ticks: {
//           color: 'white',  // This line changes the x-axis label color to white
//         },
//         border: {
//           color: 'white'  // This changes the x-axis line color to white
//         }
//       },
//       y: {
//         beginAtZero: true,
//         grid: {
//           display: false,
//         },
//         ticks: {
//           color: 'white',  // This line changes the text color to white
//           callback: function (value) {
//             return (value as number).toLocaleString();
//           },
//           stepSize: 200000,
//           // max: 1600000,
//         },
//         border: {
//           color: 'white'  // This changes the x-axis line color to white
//         }
//       },
//     },
//   };
//   return (
//     <div className='ml-0 lg:ml-[256px] h-screen '>
//       <h1 className="text-3xl font-bold mb-8 text-white">Dashboard</h1>
//       <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-4">
//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <Users className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-green-400 px-2 py-1 rounded flex gap-1 items-center">+12% <ArrowUpCircle size={15}/></span>
      
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">{userCount}</p>
//           <p className="text-white text-sm mt-1">Total Users</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>

//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <PiggyBank className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-green-400 px-2 py-1 rounded flex gap-1 items-center">+12% <ArrowUpCircle size={15}/></span>
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">${totalDeposits.toFixed(2)}</p>
//           <p className="text-white text-sm mt-1">Total Deposits</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>

//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <Wallet className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-red-400 px-2 py-1 rounded flex gap-1 items-center">-28% <ArrowDownCircle size={15}/></span>
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">${totalWithdrawals.toFixed(2)}</p>
//           <p className="text-white text-sm mt-1">Total Withdrawals</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>

//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <UserCheck className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-green-400 px-2 py-1 rounded flex gap-1 items-center">+84% <ArrowUpCircle size={15}/></span>
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">{liveBets.length}</p>
//           <p className="text-white text-sm mt-1">Live Users</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>

//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <ArrowUpCircle className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-green-400 px-2 py-1 rounded flex gap-1 items-center">+0% <ArrowUpCircle size={15}/></span>
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">${todayDeposits.toFixed(2)}</p>
//           <p className="text-white text-sm mt-1">Today's Deposit</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>

//         <div className="bg-[#646ECD] p-4 rounded-lg shadow-md">
//           <div className="flex items-center justify-between">
//             <ArrowDownCircle className="text-white h-6 w-6" />
//             <span className="text-white text-xs font-semibold bg-green-400 px-2 py-1 rounded flex gap-1 items-center">+0% <ArrowUpCircle size={15}/></span>
//           </div>
//           <p className="text-white text-2xl font-bold mt-2">${todayWithdrawals.toFixed(2)}</p>
//           <p className="text-white text-sm mt-1">Today's Withdrawal</p>
//           <div className="mt-2 text-white text-xs flex items-center">
//             <span>Since last month</span>
//             <ArrowRight className="h-4 w-4 ml-1" />
//           </div>
//         </div>
//       </div>

     
//     </div>
//   );
// };


import React, { useState, useEffect } from 'react';
import { Bar, Pie, Line } from 'react-chartjs-2';
import { 
  Chart as ChartJS, 
  CategoryScale, 
  LinearScale, 
  BarElement, 
  Title, 
  Tooltip, 
  Legend, 
  ArcElement,
  PointElement,
  LineElement,
  ChartOptions,
  FontSpec
} from 'chart.js';
import { 
  ArrowDownCircle, 
  ArrowRight, 
  ArrowUpCircle, 
  PiggyBank, 
  UserCheck, 
  Users, 
  Wallet,
  DollarSign,
  TrendingUp,
  Layers,
  Activity
} from 'lucide-react';

// Register ChartJS components
ChartJS.register(
  CategoryScale, 
  LinearScale, 
  BarElement, 
  Title, 
  Tooltip, 
  Legend, 
  ArcElement,
  PointElement,
  LineElement
);

// TypeScript interfaces
interface Breakdown {
  crypto: string;
  fiat: string;
  upi?: string;
  payIn?: string;
}

interface TransactionBreakdown {
  crypto: { amount: string; count: number };
  fiat: { amount: string; count: number };
  upi?: { amount: string; count: number };
  payIn?: { amount: string; count: number };
}

interface Deposits {
  total: string;
  previousTotal: string;
  breakdown: Breakdown;
  today: {
    amount: string;
    count: number;
    breakdown: TransactionBreakdown;
  };
}

interface Withdrawals {
  total: string;
  previousTotal: string;
  breakdown: Partial<Breakdown>;
  today: {
    amount: string;
    count: number;
    breakdown: Partial<TransactionBreakdown>;
  };
}

interface Financials {
  netBalance: string;
  previousNetBalance: string;
  todayNetBalance: string;
}

interface DashboardData {
  totalUsers: number;
  previousTotalUsers: number;
  newUsersToday: number;
  deposits: Deposits;
  withdrawals: Withdrawals;
  financials: Financials;
  timestamp: string;
  date: string;
}

interface RegistrationTrendItem {
  date: string;
  count: number;
}

interface CountryItem {
  country: string;
  count: number;
}

interface UserStats {
  totalUsers: number;
  previousTotalUsers: number;
  newUsersToday: number;
  activeUsers30d: number;
  previousActiveUsers30d: number;
  verifiedUsers: number;
  previousVerifiedUsers: number;
  topCountries: CountryItem[];
  registrationTrend: RegistrationTrendItem[];
  timestamp: string;
}

const AdminDashboard: React.FC = () => {
  // State for dashboard data
  const [dashboardData, setDashboardData] = useState<DashboardData>({
    totalUsers: 0,
    previousTotalUsers: 0,
    newUsersToday: 0,
    deposits: {
      total: "0",
      previousTotal: "0",
      breakdown: {
        crypto: "0",
        fiat: "0",
        upi: "0",
        payIn: "0"
      },
      today: {
        amount: "0",
        count: 0,
        breakdown: {
          crypto: { amount: "0", count: 0 },
          fiat: { amount: "0", count: 0 },
          upi: { amount: "0", count: 0 },
          payIn: { amount: "0", count: 0 }
        }
      }
    },
    withdrawals: {
      total: "0",
      previousTotal: "0",
      breakdown: {
        crypto: "0",
        fiat: "0"
      },
      today: {
        amount: "0",
        count: 0,
        breakdown: {
          crypto: { amount: "0", count: 0 },
          fiat: { amount: "0", count: 0 }
        }
      }
    },
    financials: {
      netBalance: "0",
      previousNetBalance: "0",
      todayNetBalance: "0"
    },
    timestamp: "",
    date: ""
  });

  // State for user statistics
  const [userStats, setUserStats] = useState<UserStats>({
    totalUsers: 0,
    previousTotalUsers: 0,
    newUsersToday: 0,
    activeUsers30d: 0,
    previousActiveUsers30d: 0,
    verifiedUsers: 0,
    previousVerifiedUsers: 0,
    topCountries: [],
    registrationTrend: [],
    timestamp: ""
  });

  // Fetch dashboard data
  useEffect(() => {
    const fetchDashboardData = async () => {
      try {
        const response = await fetch('https://api.air1bet.com/api/admin/dashboard', {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`
          }
        });
        const data: DashboardData = await response.json();
        setDashboardData(data);
      } catch (error) {
        console.error('Failed to fetch dashboard data:', error);
      }
    };

    const fetchUserStats = async () => {
      try {
        const response = await fetch('https://api.air1bet.com/api/admin/user-stats', {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`
          }
        });
        const data: UserStats = await response.json();
        setUserStats(data);
      } catch (error) {
        console.error('Failed to fetch user statistics:', error);
      }
    };

    fetchDashboardData();
    fetchUserStats();

    // Refresh data every 5 minutes
    const interval = setInterval(() => {
      fetchDashboardData();
      fetchUserStats();
    }, 300000);

    return () => clearInterval(interval);
  }, []);

  // Format currency
  const formatCurrency = (value: string | undefined): string => {
    const numericValue = parseFloat(value || "0");
    return numericValue.toLocaleString('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2
    });
  };

  // Format number
  const formatNumber = (value: number): string => {
    return value.toLocaleString('en-US');
  };

  // Calculate percentage change
  const calculatePercentageChange = (current: number, previous: number): number => {
    if (previous === 0) return current > 0 ? 100 : 0;
    return ((current - previous) / previous) * 100;
  };

  // Get percentage change indicator
  const getChangeIndicator = (current: number, previous: number): JSX.Element => {
    const percentage = calculatePercentageChange(current, previous);
    const isPositive = percentage >= 0;

    return (
      <span
        className={`text-white text-xs font-semibold ${
          isPositive ? 'bg-green-500' : 'bg-red-500'
        } px-2 py-1 rounded flex gap-1 items-center`}
      >
        {isPositive ? `+${percentage.toFixed(1)}%` : `${percentage.toFixed(1)}%`}
        {isPositive ? <ArrowUpCircle size={15} /> : <ArrowDownCircle size={15} />}
      </span>
    );
  };

  // Prepare data for deposit pie chart
  const depositPieData = {
    labels: ['Crypto', 'Fiat'],
    datasets: [
      {
        data: [
          parseFloat(dashboardData.deposits.breakdown.crypto || "0"),
          parseFloat(dashboardData.deposits.breakdown.fiat || "0"),
      
        ],
        backgroundColor: ['#4338ca', '#818cf8'],
        borderWidth: 0,
      },
    ],
  };

  // Prepare data for withdrawal pie chart
  const withdrawalPieData = {
    labels: ['Crypto', 'Fiat'],
    datasets: [
      {
        data: [
          parseFloat(dashboardData.withdrawals.breakdown.crypto || "0"),
          parseFloat(dashboardData.withdrawals.breakdown.fiat || "0")
        ],
        backgroundColor: ['#dc2626', '#fecaca'],
        borderWidth: 0,
      },
    ],
  };

  // Prepare data for user registration trend chart
  const registrationTrendData = {
    labels: userStats.registrationTrend.map(item => new Date(item.date).toLocaleDateString()),
    datasets: [
      {
        label: 'New Registrations',
        data: userStats.registrationTrend.map(item => item.count),
        fill: false,
        backgroundColor: '#4f46e5',
        borderColor: '#4f46e5',
        tension: 0.1
      },
    ],
  };

  // Prepare data for country distribution chart
  const countryDistributionData = {
    labels: userStats.topCountries.map(item => item.country),
    datasets: [
      {
        label: 'Users by Country',
        data: userStats.topCountries.map(item => item.count),
        backgroundColor: '#646ECD',
        barThickness: 30,
      },
    ],
  };

  // Chart options
  const pieOptions: ChartOptions<'pie'> = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: {
        position: 'right' as const,
        labels: {
          color: 'white',
          font: {
            size: 12
          }
        }
      }
    }
  };

  const lineOptions: ChartOptions<'line'> = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: {
        display: false
      },
      title: {
        display: true,
        text: 'User Registration Trend (Last 30 Days)',
        color: 'white',
        font: {
          size: 16,
          weight: 'bold' as FontSpec['weight']
        }
      }
    },
    scales: {
      x: {
        grid: {
          display: false
        },
        ticks: {
          color: 'white'
        },
        border: {
          color: 'white'
        }
      },
      y: {
        beginAtZero: true,
        grid: {
          color: 'rgba(255, 255, 255, 0.1)'
        },
        ticks: {
          color: 'white'
        },
        border: {
          color: 'white'
        }
      }
    }
  };

  const barOptions: ChartOptions<'bar'> = {
    responsive: true,
    maintainAspectRatio: false,
    indexAxis: 'y' as const,
    plugins: {
      legend: {
        display: false
      },
      title: {
        display: true,
        text: 'Top 10 Countries',
        color: 'white',
        font: {
          size: 16,
          weight: 'bold' as FontSpec['weight']
        }
      }
    },
    scales: {
      x: {
        grid: {
          display: false
        },
        ticks: {
          color: 'white'
        },
        border: {
          color: 'white'
        }
      },
      y: {
        grid: {
          display: false
        },
        ticks: {
          color: 'white'
        },
        border: {
          color: 'white'
        }
      }
    }
  };

  return (
    <div className="min-h-screen bg-gray-900 text-white p-6">
      <div className="ml-0 lg:ml-[256px]">
        <div className="flex justify-between items-center mb-8">
          <h1 className="text-3xl font-bold">Admin Dashboard</h1>
          <div className="text-sm text-gray-400">
            Last updated: {new Date(dashboardData.timestamp).toLocaleTimeString()}
          </div>
        </div>
        
        {/* Main Statistics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mb-8">
          {/* Total Users */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <Users className="text-blue-400 h-6 w-6" />
              {getChangeIndicator(dashboardData.totalUsers, dashboardData.previousTotalUsers)}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatNumber(dashboardData.totalUsers)}</p>
            <p className="text-gray-400 text-sm mt-1">Total Users</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>New today: {formatNumber(dashboardData.newUsersToday)}</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Total Deposits */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <PiggyBank className="text-green-400 h-6 w-6" />
              {getChangeIndicator(parseFloat(dashboardData.deposits.total), parseFloat(dashboardData.deposits.previousTotal))}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatCurrency(dashboardData.deposits.total)}</p>
            <p className="text-gray-400 text-sm mt-1">Total Deposits</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Today: {formatCurrency(dashboardData.deposits.today.amount)}</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Total Withdrawals */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <Wallet className="text-red-400 h-6 w-6" />
              {getChangeIndicator(parseFloat(dashboardData.withdrawals.total), parseFloat(dashboardData.withdrawals.previousTotal))}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatCurrency(dashboardData.withdrawals.total)}</p>
            <p className="text-gray-400 text-sm mt-1">Total Withdrawals</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Today: {formatCurrency(dashboardData.withdrawals.today.amount)}</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Net Balance */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <DollarSign className="text-purple-400 h-6 w-6" />
              {getChangeIndicator(parseFloat(dashboardData.financials.netBalance), parseFloat(dashboardData.financials.previousNetBalance))}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatCurrency(dashboardData.financials.netBalance)}</p>
            <p className="text-gray-400 text-sm mt-1">Net Balance</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Today: {formatCurrency(dashboardData.financials.todayNetBalance)}</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>
        </div>

        {/* Additional Stats */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mb-8">
          {/* Active Users */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <UserCheck className="text-indigo-400 h-6 w-6" />
              {getChangeIndicator(userStats.activeUsers30d, userStats.previousActiveUsers30d)}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatNumber(userStats.activeUsers30d)}</p>
            <p className="text-gray-400 text-sm mt-1">Active Users (30d)</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Verification rate: {userStats.totalUsers > 0 ? ((userStats.verifiedUsers / userStats.totalUsers) * 100).toFixed(1) : '0'}%</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Today's Deposits */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <ArrowUpCircle className="text-green-400 h-6 w-6" />
              <span className="text-white text-xs font-semibold bg-gray-500 px-2 py-1 rounded flex gap-1 items-center">
                N/A
              </span>
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatCurrency(dashboardData.deposits.today.amount)}</p>
            <p className="text-gray-400 text-sm mt-1">Today's Deposits</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Count: {dashboardData.deposits.today.count} transactions</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Today's Withdrawals */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <ArrowDownCircle className="text-red-400 h-6 w-6" />
              <span className="text-white text-xs font-semibold bg-gray-500 px-2 py-1 rounded flex gap-1 items-center">
                N/A
              </span>
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatCurrency(dashboardData.withdrawals.today.amount)}</p>
            <p className="text-gray-400 text-sm mt-1">Today's Withdrawals</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>Count: {dashboardData.withdrawals.today.count} transactions</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>

          {/* Verified Users */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="flex items-center justify-between">
              <Layers className="text-yellow-400 h-6 w-6" />
              {getChangeIndicator(userStats.verifiedUsers, userStats.previousVerifiedUsers)}
            </div>
            <p className="text-white text-3xl font-bold mt-2">{formatNumber(userStats.verifiedUsers)}</p>
            <p className="text-gray-400 text-sm mt-1">Verified Users</p>
            <div className="mt-2 text-gray-400 text-xs flex items-center">
              <span>KYC completion rate: {userStats.totalUsers > 0 ? ((userStats.verifiedUsers / userStats.totalUsers) * 100).toFixed(1) : '0'}%</span>
              <ArrowRight className="h-4 w-4 ml-1" />
            </div>
          </div>
        </div>

        {/* Charts Section */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
          {/* Deposit Distribution */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <h2 className="text-xl font-semibold mb-4">Deposit Distribution</h2>
            <div className="h-64">
              <Pie data={depositPieData} options={pieOptions} />
            </div>
            <div className="grid grid-cols-2 gap-4 mt-4">
              <div className="bg-gray-700 p-3 rounded-lg">
                <p className="text-gray-400 text-xs">Crypto</p>
                <p className="text-white font-semibold">{formatCurrency(dashboardData.deposits.breakdown.crypto)}</p>
              </div>
              <div className="bg-gray-700 p-3 rounded-lg">
                <p className="text-gray-400 text-xs">Fiat</p>
                <p className="text-white font-semibold">{formatCurrency(dashboardData.deposits.breakdown.fiat)}</p>
              </div>
             
            </div>
          </div>

          {/* Withdrawal Distribution */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <h2 className="text-xl font-semibold mb-4">Withdrawal Distribution</h2>
            <div className="h-64">
              <Pie data={withdrawalPieData} options={pieOptions} />
            </div>
            <div className="grid grid-cols-2 gap-4 mt-4">
              <div className="bg-gray-700 p-3 rounded-lg">
                <p className="text-gray-400 text-xs">Crypto</p>
                <p className="text-white font-semibold">{formatCurrency(dashboardData.withdrawals.breakdown.crypto)}</p>
              </div>
              <div className="bg-gray-700 p-3 rounded-lg">
                <p className="text-gray-400 text-xs">Fiat</p>
                <p className="text-white font-semibold">{formatCurrency(dashboardData.withdrawals.breakdown.fiat)}</p>
              </div>
            </div>
          </div>
        </div>

        {/* User Registration Trend and Country Distribution */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
          {/* User Registration Trend */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="h-64">
              <Line data={registrationTrendData} options={lineOptions} />
            </div>
          </div>

          {/* Country Distribution */}
          <div className="bg-gray-800 p-6 rounded-lg shadow-lg">
            <div className="h-64">
              <Bar data={countryDistributionData} options={barOptions} />
            </div>
          </div>
        </div>

        {/* Today's Activity */}
        <div className="bg-gray-800 p-6 rounded-lg shadow-lg mb-8">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold">Today's Activity Summary</h2>
            <Activity className="text-blue-400 h-5 w-5" />
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Deposits */}
            <div>
              <h3 className="text-lg font-medium mb-2 text-green-400">Deposits</h3>
              <div className="bg-gray-700 p-4 rounded-lg">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-gray-400 text-xs">Crypto</p>
                    <p className="text-white font-semibold">{formatCurrency(dashboardData.deposits.today.breakdown.crypto.amount)}</p>
                    <p className="text-gray-400 text-xs mt-1">{dashboardData.deposits.today.breakdown.crypto.count} transactions</p>
                  </div>
                  <div>
                    <p className="text-gray-400 text-xs">Fiat</p>
                    <p className="text-white font-semibold">{formatCurrency(dashboardData.deposits.today.breakdown.fiat.amount)}</p>
                    <p className="text-gray-400 text-xs mt-1">{dashboardData.deposits.today.breakdown.fiat.count} transactions</p>
                  </div>
                
                </div>
                <div className="mt-4 pt-3 border-t border-gray-600">
                  <div className="flex justify-between">
                    <p className="text-gray-300">Total</p>
                    <p className="text-green-400 font-semibold">{formatCurrency(dashboardData.deposits.today.amount)}</p>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Withdrawals */}
            <div>
              <h3 className="text-lg font-medium mb-2 text-red-400">Withdrawals</h3>
              <div className="bg-gray-700 p-4 rounded-lg">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-gray-400 text-xs">Crypto</p>
                    <p className="text-white font-semibold">{formatCurrency(dashboardData.withdrawals.today.breakdown.crypto?.amount)}</p>
                    <p className="text-gray-400 text-xs mt-1">{dashboardData.withdrawals.today.breakdown.crypto?.count || 0} transactions</p>
                  </div>
                  <div>
                    <p className="text-gray-400 text-xs">Fiat</p>
                    <p className="text-white font-semibold">{formatCurrency(dashboardData.withdrawals.today.breakdown.fiat?.amount)}</p>
                    <p className="text-gray-400 text-xs mt-1">{dashboardData.withdrawals.today.breakdown.fiat?.count || 0} transactions</p>
                  </div>
                </div>
                <div className="mt-4 pt-3 border-t border-gray-600">
                  <div className="flex justify-between">
                    <p className="text-gray-300">Total</p>
                    <p className="text-red-400 font-semibold">{formatCurrency(dashboardData.withdrawals.today.amount)}</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        
        {/* Net Balance Section */}
        <div className="bg-gray-800 p-6 rounded-lg shadow-lg mb-8">
          <h2 className="text-xl font-semibold mb-4">Financial Summary</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="bg-gray-700 p-4 rounded-lg flex justify-between items-center">
              <div>
                <p className="text-gray-400 text-sm">All-time Net Balance</p>
                <p className={`text-2xl font-bold ${parseFloat(dashboardData.financials.netBalance) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                  {formatCurrency(dashboardData.financials.netBalance)}
                </p>
              </div>
              <TrendingUp className={`h-8 w-8 ${parseFloat(dashboardData.financials.netBalance) >= 0 ? 'text-green-400' : 'text-red-400'}`} />
            </div>
            
            <div className="bg-gray-700 p-4 rounded-lg flex justify-between items-center">
              <div>
                <p className="text-gray-400 text-sm">Today's Net Balance</p>
                <p className={`text-2xl font-bold ${parseFloat(dashboardData.financials.todayNetBalance) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                  {formatCurrency(dashboardData.financials.todayNetBalance)}
                </p>
              </div>
              <Activity className={`h-8 w-8 ${parseFloat(dashboardData.financials.todayNetBalance) >= 0 ? 'text-green-400' : 'text-red-400'}`} />
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AdminDashboard;