// src/components/GameWinLossToggleList.tsx
import React, { useEffect, useState, useCallback } from "react";
import { RefreshCw } from "lucide-react";
import { apiFetch } from "../utils/api";

type ToggleRow = {
  uid       : string;
  name      : string;
  blacklist : boolean;
  whitelist : boolean;
};

export default function GameWinLossToggleList() {
  const [rows, setRows]       = useState<ToggleRow[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError]     = useState<string | null>(null);

  // load all users & toggles
  const fetchAll = useCallback(async () => {
    setLoading(true);
    setError(null);
    try {
      const data = await apiFetch<ToggleRow[]>("/admin/users/game-toggles");
      setRows(data);
    } catch (err: any) {
      setError(err.message || "Failed to load toggles");
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchAll();
  }, [fetchAll]);

  // update a single user
  const updateRow = async (
    uid: string,
    newBlacklist: boolean,
    newWhitelist: boolean
  ) => {
    setError(null);
    try {
      await apiFetch(`/admin/user/${uid}/game-toggle`, {
        method: "PUT",
        body: JSON.stringify({ blacklist: newBlacklist, whitelist: newWhitelist }),
      });
      await fetchAll();
    } catch (err: any) {
      setError(`Error updating ${uid}: ${err.message}`);
    }
  };

  // bulk actions
  const bulkAction = async (
    action: "blacklist-all" | "whitelist-all" | "disable-all"
  ) => {
    setError(null);
    setLoading(true);
    try {
      await apiFetch("/admin/users/game-toggles", {
        method: "PUT",
        body: JSON.stringify({ action }),
      });
      await fetchAll();
    } catch (err: any) {
      setError(`Bulk action failed: ${err.message}`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="p-4 ml-0 lg:ml-[256px] space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-semibold text-white">Original Game Toggle Management</h1>
        <button
          onClick={fetchAll}
          disabled={loading}
          className="flex items-center space-x-1 px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:opacity-50"
        >
          <RefreshCw size={16} /> <span>Refresh</span>
        </button>
      </div>

      <div className="flex space-x-4">
        <button
          onClick={() => bulkAction("blacklist-all")}
          disabled={loading}
          className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700 disabled:opacity-50"
        >
          Blacklist All
        </button>
        <button
          onClick={() => bulkAction("whitelist-all")}
          disabled={loading}
          className="px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700 disabled:opacity-50"
        >
          Whitelist All
        </button>
        <button
          onClick={() => bulkAction("disable-all")}
          disabled={loading}
          className="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700 disabled:opacity-50"
        >
          Disable All
        </button>
      </div>

      {loading && <p>Loading…</p>}
      {error && <p className="text-red-500">{error}</p>}

      {!loading && !error && (
        <div className="overflow-auto border rounded">
          <table className="min-w-full bg-white">
            <thead className="bg-gray-100">
              <tr>
                <th className="px-4 py-2 text-left">User ID</th>
                <th className="px-4 py-2 text-left">Name</th>
                <th className="px-4 py-2 text-center">Blacklist</th>
                <th className="px-4 py-2 text-center">Whitelist</th>
              </tr>
            </thead>
            <tbody>
              {rows.map(({ uid, name, blacklist, whitelist }) => (
                <tr key={uid} className="border-t hover:bg-gray-50">
                  <td className="px-4 py-2">{uid}</td>
                  <td className="px-4 py-2">{name}</td>
                  <td className="px-4 py-2 text-center">
                    <input
                      type="checkbox"
                      checked={blacklist}
                      onChange={() => updateRow(uid, !blacklist, false)}
                      className="h-5 w-5 cursor-pointer"
                    />
                  </td>
                  <td className="px-4 py-2 text-center">
                    <input
                      type="checkbox"
                      checked={whitelist}
                      onChange={() => updateRow(uid, false, !whitelist)}
                      className="h-5 w-5 cursor-pointer"
                    />
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}
