
import React, { useState, useEffect } from "react";
import axios from "axios";

// TypeScript interfaces
interface Transaction {
  id: string;
  transaction_id: string;
  user_name: string;
  user_id: string;
  game_title?: string;
  transaction_type: string;
  amount: number;
  currency_code: string;
  transaction_timestamp: string;
}

interface PaginationState {
  totalCount: number;
  totalPages: number;
  currentPage: number;
  limit: number;
}

interface TransactionResponse {
  transactions: Transaction[];
  pagination: PaginationState;
}

enum TabType {
  ALL_BETS = 'all',
  LUCKY_SPORTS = 'luckySports'
}

const History: React.FC = () => {
   const staffId  = localStorage.getItem('currentUserId');
  const [transactions, setTransactions] = useState<Transaction[]>([]);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [pagination, setPagination] = useState<PaginationState>({
    totalCount: 0,
    totalPages: 0,
    currentPage: 1,
    limit: 10 // Fixed to 10 items per page
  });
  const [loading, setLoading] = useState<boolean>(false);
  const [userId, setUserId] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<TabType>(TabType.ALL_BETS);

  useEffect(() => {
    // Reset to page 1 when changing tabs
    setPagination(prev => ({ ...prev, currentPage: 1 }));
    fetchTransactionData();
  }, [activeTab]);

  useEffect(() => {
    // This effect runs when pagination page or userId changes
    if (transactions.length > 0) { // Only if we've loaded data at least once
      fetchTransactionData();
    }
  }, [pagination.currentPage, userId]);
  
  // Debounced search effect
  useEffect(() => {
    const timer = setTimeout(() => {
      fetchTransactionData();
    }, 500);

    return () => clearTimeout(timer);
  }, [searchTerm]);
// ⬇️ put this inside your component – it replaces the old fetchTransactionData
const fetchTransactionData = async (): Promise<void> => {
  setLoading(true);

  try {
    /* ------------------------------------------------------------------
       1.  Build base URL
       ------------------------------------------------------------------ */
    const base = "https://api.air1bet.com/betHistory";

    let url: string;

    if (activeTab === TabType.ALL_BETS) {
      // all bets *except* LuckySports
      url = `${base}/transactions?page=${pagination.currentPage}&limit=10&excludeGame=LuckySports`;
      if (userId) {
        url = `${base}/transactions/user/${encodeURIComponent(
          userId
        )}?page=${pagination.currentPage}&limit=10&excludeGame=LuckySports`;
      }
    } else {
      // LuckySports only
      url = `${base}/transactions/luckysports?page=${pagination.currentPage}&limit=10`;
      if (userId) url += `&userId=${encodeURIComponent(userId)}`;
    }

    /* ------------------------------------------------------------------
       2.  Optional search term
       ------------------------------------------------------------------ */
    if (searchTerm) url += `&search=${encodeURIComponent(searchTerm)}`;

    /* ------------------------------------------------------------------
       3.  Auth header – MUST be present now that routes are behind
           `router.use(protectStaff)`
       ------------------------------------------------------------------ */
    const token = localStorage.getItem("token"); // or whatever key you store it under
const headers: Record<string, string> = {
  ...(token   ? { Authorization: `Bearer ${token}` } : {}),
  ...(staffId ? { 'x-staff-id': staffId }            : {})
};

    /* ------------------------------------------------------------------
       4.  Go!
       ------------------------------------------------------------------ */
    const { data } = await axios.get<TransactionResponse>(url, { headers });

    setTransactions(data.transactions);
    setPagination(data.pagination);
  } catch (err) {
    console.error("Failed to fetch transaction data:", err);
  } finally {
    setLoading(false);
  }
};

  // const fetchTransactionData = async (): Promise<void> => {
  //   setLoading(true);
  //   try {
  //     let url;
      
  //     if (activeTab === TabType.ALL_BETS) {
  //       // URL for all bets except LuckySports
  //       url = `https://api.air1bet.com/betHistory/transactions?page=${pagination.currentPage}&limit=10&excludeGame=LuckySports`;
        
  //       if (searchTerm) {
  //         url += `&search=${encodeURIComponent(searchTerm)}`;
  //       }

  //       if (userId) {
  //         url = `https://api.air1bet.com/betHistory/transactions/user/${userId}?page=${pagination.currentPage}&limit=10&excludeGame=LuckySports`;
  //         if (searchTerm) {
  //           url += `&search=${encodeURIComponent(searchTerm)}`;
  //         }
  //       }
  //     } else {
  //       // URL for LuckySports transactions only
  //       url = `https://api.air1bet.com/betHistory/transactions/luckysports?page=${pagination.currentPage}&limit=10`;
        
  //       if (searchTerm) {
  //         url += `&search=${encodeURIComponent(searchTerm)}`;
  //       }

  //       if (userId) {
  //         url += `&userId=${encodeURIComponent(userId)}`;
  //       }
  //     }

  //     const response = await axios.get<TransactionResponse>(url);
  //     setTransactions(response.data.transactions);
  //     setPagination(response.data.pagination);
  //   } catch (error) {
  //     console.error('Failed to fetch transaction data:', error);
  //   } finally {
  //     setLoading(false);
  //   }
  // };

  const handleSearchChange = (event: React.ChangeEvent<HTMLInputElement>): void => {
    setSearchTerm(event.target.value);
  };

  const handleUserFilterChange = (event: React.ChangeEvent<HTMLInputElement>): void => {
    setUserId(event.target.value || null);
  };

  const handlePageChange = (newPage: number): void => {
    setPagination(prev => ({ ...prev, currentPage: newPage }));
  };

  const handleClearFilters = (): void => {
    setSearchTerm('');
    setUserId(null);
    setPagination(prev => ({ ...prev, currentPage: 1 }));
  };

  const handleTabChange = (tab: TabType): void => {
    if (tab !== activeTab) {
      setActiveTab(tab);
      setPagination(prev => ({ ...prev, currentPage: 1 })); // Reset to first page when changing tabs
    }
  };

  // Format timestamp to readable date & time
  const formatDate = (timestamp: string): string => {
    return new Date(timestamp).toLocaleString();
  };

  // Format amount with color depending on transaction type
  const formatAmount = (amount: number, type: string): JSX.Element => {
    const colorMap = {
      'WIN': 'text-green-600',
      'LOSS': 'text-red-600',
      'BET': 'text-gray-800'
    };

    const signPrefix = type === 'WIN' ? '+' : type === 'LOSS' ? '-' : '';
    const colorClass = colorMap[type as keyof typeof colorMap] || 'text-gray-800';

    return <span className={colorClass}>{signPrefix}{Math.abs(amount).toFixed(2)}</span>;
  };

  // Simplified pagination with just prev/next buttons and page number
  const renderPagination = (): JSX.Element => {
    return (
      <div className="flex items-center justify-center mt-6">
        <button 
          onClick={() => handlePageChange(pagination.currentPage - 1)}
          disabled={pagination.currentPage === 1}
          className="px-4 py-2 mr-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
          aria-label="Previous page"
        >
          Previous
        </button>
        
        <div className="text-center px-4 py-2 bg-gray-50 rounded border min-w-[80px]">
          {pagination.currentPage} / {pagination.totalPages || 1}
        </div>
        
        <button 
          onClick={() => handlePageChange(pagination.currentPage + 1)}
          disabled={pagination.currentPage === pagination.totalPages || pagination.totalPages === 0}
          className="px-4 py-2 ml-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
          aria-label="Next page"
        >
          Next
        </button>
      </div>
    );
  };

  // Render tabs
  const renderTabs = (): JSX.Element => {
    return (
      <div className="border-b border-gray-200 mb-6">
        <nav className="-mb-px flex space-x-8">
          <button
            onClick={() => handleTabChange(TabType.ALL_BETS)}
            className={`py-4 px-1 border-b-2 font-medium text-sm ${
              activeTab === TabType.ALL_BETS
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            All Bets
          </button>
          <button
            onClick={() => handleTabChange(TabType.LUCKY_SPORTS)}
            className={`py-4 px-1 border-b-2 font-medium text-sm ${
              activeTab === TabType.LUCKY_SPORTS
                ? 'border-blue-500 text-blue-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            LuckySports
          </button>
        </nav>
      </div>
    );
  };

  // Table view rendering
  const renderTableView = (): JSX.Element => (
    <div className="overflow-x-auto">
      <table className="w-full bg-white shadow-md rounded-lg overflow-hidden">
        <thead className="bg-gray-100 border-b">
          <tr>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Game</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Coin</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
          </tr>
        </thead>
        <tbody className="divide-y divide-gray-200">
          {transactions.map((transaction) => (
            <tr key={transaction.id} className="hover:bg-gray-50 transition-colors">
              <td className="px-4 py-3 whitespace-nowrap">{transaction.user_name}</td>
              <td className="px-4 py-3 whitespace-nowrap">{transaction.game_title || 'Unknown'}</td>
              <td className="px-4 py-3 whitespace-nowrap">{transaction.currency_code|| 'Unknown'}</td>
              <td className="px-4 py-3 whitespace-nowrap">{transaction.transaction_type}</td>
              <td className="px-4 py-3 text-right whitespace-nowrap">
                {formatAmount(transaction.amount, transaction.transaction_type)}
              </td>
              <td className="px-4 py-3 whitespace-nowrap">{formatDate(transaction.transaction_timestamp)}</td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );

  // Loading state
  if (loading && transactions.length === 0) {
    return (
      <div className="ml-0 lg:ml-[256px] p-6">
        <div className="flex items-center justify-center h-64">
          <div className="text-gray-500">Loading transaction data...</div>
        </div>
      </div>
    );
  }

  return (
    <div className="ml-0 lg:ml-[256px] p-6">
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h1 className="text-2xl font-bold mb-6">Bet History</h1>
        
        {/* Tab Navigation */}
        {/* {renderTabs()} */}
        
        {/* Filters and controls */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
          <div>
            <label htmlFor="search-input" className="block text-sm font-medium text-gray-700 mb-1">Search</label>
            <input
              id="search-input"
              type="text"
              placeholder="Search by name, ID or session"
              value={searchTerm}
              onChange={handleSearchChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label htmlFor="user-id-input" className="block text-sm font-medium text-gray-700 mb-1">User ID</label>
            <input
              id="user-id-input"
              type="text"
              placeholder="Filter by User ID"
              value={userId || ''}
              onChange={handleUserFilterChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>
        
        {/* Info and clear filters */}
        <div className="mb-4">
          <span className="text-gray-600">
            Showing {transactions.length} of {pagination.totalCount} transactions
            {activeTab === TabType.LUCKY_SPORTS && " for LuckySports"}
          </span>
          {(searchTerm || userId) && (
            <button 
              onClick={handleClearFilters}
              className="ml-4 text-blue-500 hover:text-blue-700"
              aria-label="Clear filters"
            >
              Clear filters
            </button>
          )}
        </div>
        
        {/* No results message */}
        {transactions.length === 0 && !loading && (
          <div className="text-center py-10">
            <p className="text-gray-500">
              No transactions found
              {activeTab === TabType.LUCKY_SPORTS && " for LuckySports"}
            </p>
          </div>
        )}
        
        {/* Transaction table */}
        {renderTableView()}
        
        {/* Pagination */}
        {pagination.totalPages > 1 && renderPagination()}
      </div>
    </div>
  );
};

export default History;