import React, { useState, useEffect } from 'react';
import axios from 'axios';

// Define interfaces
interface Team {
  ownername: string;
  referalCode: string;
  createdAt: string;
  updatedAt: string;
  owner_id: number;
  owner_name: string;
  owner_email: string;
  owner_avatar: string;
  owner_level: number;
  members: TeamMember[];
}

interface TeamMember {
  id: number;
  name: string;
  email: string;
  avatar: string;
  level: number;
  balance: number | null;
  games_played: number;
}

interface UserWithTeam {
  id: number;
  name: string;
  email: string;
  avatar: string;
  level: number;
  balance: number | null;
  games_played: number;
  refree: string | null;
  referalcode: string;
  team_name: string | null;
  children: UserWithTeam[];
}

const Referal: React.FC = () => {
  const [teams, setTeams] = useState<Team[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [expandedTeam, setExpandedTeam] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'teams' | 'hierarchy'>('teams');
  const [usersWithTeams, setUsersWithTeams] = useState<UserWithTeam[]>([]);

  const API_BASE_URL = 'https://api.air1bet.com/affiliateAdmin';

  useEffect(() => {
    fetchTeams();
    // fetchUsersWithTeams();
  }, []);

  const fetchTeams = async () => {
    try {
      setLoading(true);
      const response = await axios.get(`${API_BASE_URL}/teams`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      if (response.data.success) {
        setTeams(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching teams:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchUsersWithTeams = async () => {
    try {
      const response = await axios.get(`${API_BASE_URL}/users-with-teams`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      if (response.data.success) {
        setUsersWithTeams(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching users with teams:', error);
    }
  };

  const toggleTeamExpansion = (teamId: string) => {
    setExpandedTeam(expandedTeam === teamId ? null : teamId);
  };

  // Filter teams based on search term
  const filteredTeams = teams.filter(team =>
   team.members.length > 0 &&( (team.owner_name || '').toLowerCase().includes(searchTerm.toLowerCase()) ||
    (team.referalCode || '').toLowerCase().includes(searchTerm.toLowerCase()))
  );

  // Generate random color based on string
  const stringToColor = (string: string): string => {
    let hash = 0;
    for (let i = 0; i < string.length; i++) {
      hash = string.charCodeAt(i) + ((hash << 5) - hash);
    }
    let color = '#';
    for (let i = 0; i < 3; i++) {
      const value = (hash >> (i * 8)) & 0xFF;
      color += ('00' + value.toString(16)).substr(-2);
    }
    return color;
  };

  // Format date
  const formatDate = (dateString: string): string => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Render hierarchy recursively
  const renderHierarchy = (users: UserWithTeam[], level: number = 0) => {
    return (
      <div className={`ml-${level * 4}`}>
        {users.map(user => (
          <div key={user.id} className="mb-4">
            <div className="flex items-center p-4 bg-gray-50 rounded-lg border border-gray-200">
              {user.avatar ? (
                <img
                  src={user.avatar}
                  alt={user.name || 'User'}
                  className="h-10 w-10 rounded-full mr-3"
                />
              ) : (
                <div
                  className="h-10 w-10 rounded-full flex items-center justify-center text-white font-bold mr-3"
                  style={{ backgroundColor: stringToColor(user.name || '') }}
                >
                  {(user.name || '').charAt(0).toUpperCase() || '?'}
                </div>
              )}
              <div className="flex-1">
                <div className="text-sm font-medium text-gray-900">{user.name || 'Unknown'}</div>
                <div className="text-xs text-gray-500">{user.email || 'No email'}</div>
              </div>
              <div className="flex items-center space-x-3">
                {user.team_name && (
                  <span className="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                    Team: {user.team_name}
                  </span>
                )}
                <span className="px-3 py-1 bg-green-100 text-green-800 rounded-full text-xs font-medium">
                  Level {user.level || 1}
                </span>
              </div>
            </div>
            {user.children && user.children.length > 0 && (
              <div className="mt-2 border-l-2 border-gray-200 pl-4">
                {renderHierarchy(user.children, level + 1)}
              </div>
            )}
          </div>
        ))}
      </div>
    );
  };

  return (
          <div className='ml-0 lg:ml-[256px]'>
  <div className="min-h-screen bg-gray-50 p-6">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center mb-6">
          <h1 className="text-2xl font-bold text-gray-800">Affiliate Management</h1>
          <div className="relative mt-4 md:mt-0 w-full md:w-64">
            <input
              type="text"
              placeholder="Search teams..."
              className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500"
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <svg
              xmlns="http://www.w3.org/2000/svg"
              className="h-5 w-5 absolute left-3 top-2.5 text-gray-400"
              fill="none"
              viewBox="0 0 24 24"
              stroke="currentColor"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
              />
            </svg>
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200 mb-6">
          <nav className="flex space-x-8">
            <button
              onClick={() => setActiveTab('teams')}
              className={`py-2 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'teams'
                  ? 'border-indigo-500 text-indigo-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Teams
            </button>
            {/* <button
              onClick={() => setActiveTab('hierarchy')}
              className={`py-2 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'hierarchy'
                  ? 'border-indigo-500 text-indigo-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              User Hierarchy
            </button> */}
          </nav>
        </div>

        {/* Content */}
        {activeTab === 'teams' ? (
          <div className="bg-white rounded-xl shadow-md overflow-hidden">
            {loading ? (
              <div className="flex justify-center items-center h-64">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-500"></div>
              </div>
            ) : filteredTeams.length > 0 ? (
              <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Team</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Owner</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Referral Code</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Members</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
                      <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {filteredTeams.map((team) => (
                      <React.Fragment key={team.ownername}>
                        <tr
                          className="hover:bg-gray-50 cursor-pointer transition-colors"
                          onClick={() => toggleTeamExpansion(team.ownername)}
                        >
                          <td className="px-6 py-4 whitespace-nowrap">
                            <div className="flex items-center">
                              <div
                                className="h-10 w-10 rounded-full flex items-center justify-center text-white font-bold"
                                style={{ backgroundColor: stringToColor(team.owner_name || '') }}
                              >
                                {(team.owner_name || '').charAt(0).toUpperCase() || '?'}
                              </div>
                              <div className="ml-4">
                                <div className="text-sm font-medium text-gray-900">{team.owner_name || 'Unknown'}</div>
                              </div>
                            </div>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <div className="flex items-center">
                              {/* {team.owner_avatar ? (
                                <img
                                  src={team.owner_avatar}
                                  alt={team.owner_name || 'User'}
                                  className="h-8 w-8 rounded-full mr-3"
                                />
                              ) : (
                                <div
                                  className="h-8 w-8 rounded-full flex items-center justify-center text-white font-bold mr-3"
                                  style={{ backgroundColor: stringToColor(team.owner_name || '') }}
                                >
                                  {(team.owner_name || '').charAt(0).toUpperCase() || '?'}
                                </div>
                              )} */}
                              <div>
                                {/* <div className="text-sm font-medium text-gray-900">{team.owner_name || 'Unknown'}</div> */}
                                <div className="text-xs text-gray-500">{team.owner_email || 'No email'}</div>
                              </div>
                            </div>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <span className="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-indigo-100 text-indigo-800">
                              {team.referalCode || 'N/A'}
                            </span>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {team.members.length}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {formatDate(team.createdAt)}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <button className="text-indigo-600 hover:text-indigo-900">
                              {expandedTeam === team.ownername ? (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                  <path fillRule="evenodd" d="M14.707 12.707a1 1 0 01-1.414 0L10 9.414l-3.293 3.293a1 1 0 01-1.414-1.414l4-4a1 1 0 011.414 0l4 4a1 1 0 010 1.414z" clipRule="evenodd" />
                                </svg>
                              ) : (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                  <path fillRule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clipRule="evenodd" />
                                </svg>
                              )}
                            </button>
                          </td>
                        </tr>
                        {expandedTeam === team.ownername && (
                          <tr>
                            <td colSpan={6} className="px-6 py-4 bg-gray-50">
                              <div className="animate-fadeIn">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Team Details</h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                                  {/* Team Info Card */}
                                  <div className="bg-white p-4 rounded-lg border border-gray-200">
                                    <h4 className="text-sm font-medium text-gray-500 mb-3">Team Information</h4>
                                    <div className="space-y-3">
                                      <div className="flex justify-between">
                                        <span className="text-sm text-gray-600">Team ID:</span>
                                        <span className="text-sm font-medium">{team.ownername}</span>
                                      </div>
                                      <div className="flex justify-between">
                                        <span className="text-sm text-gray-600">Owner Level:</span>
                                        <span className="text-sm font-medium">
                                          <span className="px-2 py-1 text-xs rounded bg-green-100 text-green-800">
                                            Level {team.owner_level || 1}
                                          </span>
                                        </span>
                                      </div>
                                      <div className="flex justify-between">
                                        <span className="text-sm text-gray-600">Created:</span>
                                        <span className="text-sm font-medium">{formatDate(team.createdAt)}</span>
                                      </div>
                                      <div className="flex justify-between">
                                        <span className="text-sm text-gray-600">Updated:</span>
                                        <span className="text-sm font-medium">{formatDate(team.updatedAt)}</span>
                                      </div>
                                    </div>
                                  </div>
                                  {/* Referral Card */}
                                  <div className="bg-indigo-50 p-4 rounded-lg border border-indigo-100">
                                    <h4 className="text-sm font-medium text-indigo-700 mb-3">Referral Information</h4>
                                    <div className="flex items-center mb-4 p-2 bg-white rounded-lg">
                                      <svg
                                        xmlns="http://www.w3.org/2000/svg"
                                        className="h-5 w-5 text-indigo-500 mr-2"
                                        fill="none"
                                        viewBox="0 0 24 24"
                                        stroke="currentColor"
                                      >
                                        <path
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                          strokeWidth={2}
                                          d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4"
                                        />
                                      </svg>
                                      <div className="flex-1">
                                        <span className="text-sm font-medium">Referral Code:</span>
                                        <div className="flex items-center mt-1">
                                          <code className="bg-gray-100 px-2 py-1 rounded text-sm">{team.referalCode || 'N/A'}</code>
                                          <button
                                            className="ml-2 text-indigo-600 hover:text-indigo-800"
                                            onClick={(e) => {
                                              e.stopPropagation();
                                              if (team.referalCode) {
                                                navigator.clipboard.writeText(team.referalCode);
                                              }
                                            }}
                                          >
                                            <svg
                                              xmlns="http://www.w3.org/2000/svg"
                                              className="h-4 w-4"
                                              fill="none"
                                              viewBox="0 0 24 24"
                                              stroke="currentColor"
                                            >
                                              <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={2}
                                                d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2"
                                              />
                                            </svg>
                                          </button>
                                        </div>
                                      </div>
                                    </div>
                                    <div className="flex items-start bg-white p-2 rounded-lg">
                                      <svg
                                        xmlns="http://www.w3.org/2000/svg"
                                        className="h-5 w-5 text-indigo-500 mr-2 mt-0.5"
                                        fill="none"
                                        viewBox="0 0 24 24"
                                        stroke="currentColor"
                                      >
                                        <path
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                          strokeWidth={2}
                                          d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                                        />
                                      </svg>
                                      <span className="text-sm">Share this code with new users to add them to your team.</span>
                                    </div>
                                  </div>
                                </div>
                                {/* Team Members */}
                                <div>
                                  <h4 className="text-md font-medium text-gray-900 mb-3">Team Members ({team.members.length})</h4>
                                  {team.members.length === 0 ? (
                                    <div className="bg-white p-8 rounded-lg border border-gray-200 text-center text-gray-500">
                                      No team members found
                                    </div>
                                  ) : (
                                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                      {team.members.map(member => (
                                        <div key={member.id} className="bg-white p-4 rounded-lg border border-gray-200 flex items-center">
                                          {/* {member.avatar ? (
                                            <img
                                              src={member.avatar}
                                              alt={member.name || 'User'}
                                              className="h-10 w-10 rounded-full mr-3"
                                            />
                                          ) : (
                                            <div
                                              className="h-10 w-10 rounded-full flex items-center justify-center text-white font-bold mr-3"
                                              style={{ backgroundColor: stringToColor(member.name || '') }}
                                            >
                                              {(member.name || '').charAt(0).toUpperCase() || '?'}
                                            </div>
                                          )} */}
                                          <div>
                                            <div className="text-sm font-medium text-gray-900">{member.name || 'Unknown'}</div>
                                            <div className="flex items-center text-xs text-gray-500 mt-1">
                                              <span className="px-2 py-0.5 bg-green-100 text-green-800 rounded mr-2">
                                                Level {member.level || 1}
                                              </span>
                                            
                                            </div>
                                          </div>
                                        </div>   
                                      ))}
                                    </div>
                                  )}
                                </div>
                              </div>
                            </td>
                          </tr>
                        )}
                      </React.Fragment>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div className="p-8 text-center text-gray-500">
                No teams found matching your search
              </div>
            )}
          </div>
        ) : (
          <div className="bg-white rounded-xl shadow-md p-6">
            <h3 className="text-lg font-medium text-gray-900 mb-4">User Hierarchy</h3>
            {usersWithTeams.length === 0 ? (
              <div className="flex justify-center items-center h-64">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-500"></div>
              </div>
            ) : (
              renderHierarchy(usersWithTeams)
            )}
          </div>
        )}
      </div>

      {/* <style jsx>{`
        @keyframes fadeIn {
          from { opacity: 0; transform: translateY(-10px); }
          to { opacity: 1; transform: translateY(0); }
        }
        .animate-fadeIn {
          animation: fadeIn 0.3s ease-out forwards;
        }
      `}</style> */}
    </div>
          </div>
  
  );
};

export default Referal;