// import React, { useEffect, useMemo, useState } from "react";
// import { ChevronDownIcon, ArrowPathIcon, MagnifyingGlassIcon } from "@heroicons/react/24/outline";
// import { apiFetch } from "../utils/api";

// // ──────────────────────────────────────────────────────────────
// //  UserWagerReports.tsx
// //  • Fetches and displays wager progress for all users
// //  • Progress bar fills based on wager vs first_deposit_amount_2x
// //  • Search, sort, and client-side pagination included
// //  • Matches the lightweight, Tailwind-first style of your codebase
// // ──────────────────────────────────────────────────────────────

// type WagerRow = {
//   id: number;
//   name: string;
//   email: string;
//   wager: number; // current wager volume
//   first_deposit_amount: number; // 1x
//   first_deposit_amount_2x: number; // target (max)
//   first_deposit_at: string | null;
//   user_created_at: string;
// };

// type ApiResponse = {
//   success: boolean;
//   data: WagerRow[];
// };

// const fmtAmt = (n: number | null | undefined) => {
//   if (n == null || Number.isNaN(n)) return "-";
//   return new Intl.NumberFormat(undefined, { maximumFractionDigits: 4 }).format(n);
// };

// const fmtDateTime = (iso: string | null) => {
//   if (!iso) return "-";
//   const d = new Date(iso);
//   if (Number.isNaN(d.getTime())) return "-";
//   return d.toLocaleString();
// };

// const clamp = (n: number, lo = 0, hi = 1) => Math.max(lo, Math.min(hi, n));

// const columns = [
//   { key: "id", label: "User ID" },
//   { key: "name", label: "Name" },
//   { key: "email", label: "Email" },
//   { key: "wager", label: "Wager" },

// ];

// export default function UserWagerReports() {
//   const [rows, setRows] = useState<WagerRow[]>([]);
//   const [loading, setLoading] = useState(true);
//   const [error, setError] = useState<string | null>(null);
//   const [q, setQ] = useState("");
//   const [sortKey, setSortKey] = useState<keyof WagerRow>("first_deposit_amount_2x");
//   const [sortDir, setSortDir] = useState<"asc" | "desc">("desc");
//   const [page, setPage] = useState(1);
//   const pageSize = 10;

//   const fetchData = async () => {
//     try {
//       setLoading(true);
//       setError(null);
//       // If your apiFetch is already configured with a base, you can drop the host part.
//       const res = await apiFetch<ApiResponse>("/api/deposits/admin/all-wagers");
//       if (!res || !(res as any).success) throw new Error("Unexpected response");
//       setRows((res as ApiResponse).data || []);
//     } catch (e: any) {
//       // fallback to absolute URL if relative fails
//       try {
//         const fallback = await apiFetch<ApiResponse>(
//           "https://api.air1bet.com/api/deposits/admin/all-wagers"
//         );
//         if (!fallback || !(fallback as any).success) throw new Error("Unexpected response");
//         setRows((fallback as ApiResponse).data || []);
//       } catch (e2: any) {
//         setError(e2?.message || "Failed to load wagers");
//       }
//     } finally {
//       setLoading(false);
//     }
//   };

//   useEffect(() => {
//     fetchData();
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, []);

//   const filtered = useMemo(() => {
//     const term = q.trim().toLowerCase();
//     const base = term
//       ? rows.filter(r =>
//           r.name.toLowerCase().includes(term) ||
//           r.email.toLowerCase().includes(term) ||
//           String(r.id).includes(term)
//         )
//       : rows;

//     const sorted = [...base].sort((a, b) => {
//       const va = a[sortKey];
//       const vb = b[sortKey];
//       if (va == null && vb == null) return 0;
//       if (va == null) return sortDir === "asc" ? -1 : 1;
//       if (vb == null) return sortDir === "asc" ? 1 : -1;
//       if (typeof va === "number" && typeof vb === "number") {
//         return sortDir === "asc" ? va - vb : vb - va;
//       }
//       const sa = String(va).toLowerCase();
//       const sb = String(vb).toLowerCase();
//       if (sa < sb) return sortDir === "asc" ? -1 : 1;
//       if (sa > sb) return sortDir === "asc" ? 1 : -1;
//       return 0;
//     });

//     return sorted;
//   }, [rows, q, sortKey, sortDir]);

//   const totalPages = Math.max(1, Math.ceil(filtered.length / pageSize));
//   const paged = filtered.slice((page - 1) * pageSize, page * pageSize);

//   const onSort = (key: keyof WagerRow) => {
//     if (key === sortKey) {
//       setSortDir(d => (d === "asc" ? "desc" : "asc"));
//     } else {
//       setSortKey(key);
//       setSortDir("asc");
//     }
//   };

//   return (
//     <div className="ml-0 lg:ml-[256px] p-2">
//       <div className="max-w-7xl mx-auto p-6 space-y-6 bg-gray-50 min-h-screen">
//         <div className="flex items-center justify-between">
//           <h1 className="text-3xl font-bold text-gray-900">User Turnover Report</h1>
//           <button
//             onClick={fetchData}
//             className="inline-flex items-center gap-2 rounded-lg bg-blue-600 text-white px-4 py-2 hover:bg-blue-700"
//           >
//             <ArrowPathIcon className="w-5 h-5" /> Refresh
//           </button>
//         </div>

//         {/* Controls */}
//         <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
//           <div className="relative w-full sm:w-80">
//             <MagnifyingGlassIcon className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 -translate-y-1/2" />
//             <input
//               value={q}
//               onChange={(e) => { setPage(1); setQ(e.target.value); }}
//               placeholder="Search by name, email, or ID"
//               className="w-full pl-10 pr-3 py-2 rounded-lg border border-gray-300 bg-white focus:outline-none focus:ring-2 focus:ring-blue-500"
//             />
//           </div>

//           <div className="flex items-center gap-2">
//             <label className="text-sm text-gray-600">Sort by</label>
//             <select
//               value={String(sortKey)}
//               onChange={(e) => setSortKey(e.target.value as keyof WagerRow)}
//               className="rounded-lg border border-gray-300 bg-white px-3 py-2 text-sm"
//             >
//               {columns.map(c => (
//                 <option key={c.key} value={c.key}>{c.label}</option>
//               ))}
//             </select>
//             <button
//               onClick={() => setSortDir(d => d === "asc" ? "desc" : "asc")}
//               className="rounded-lg border border-gray-300 bg-white px-3 py-2 text-sm"
//               title="Toggle sort direction"
//             >
//               {sortDir === "asc" ? "Asc" : "Desc"}
//             </button>
//           </div>
//         </div>

//         {/* Content */}
//         {loading ? (
//           <div className="text-gray-600 text-lg animate-pulse">Loading wagers…</div>
//         ) : error ? (
//           <div className="text-red-600">{error}</div>
//         ) : filtered.length === 0 ? (
//           <div className="text-gray-600">No results.</div>
//         ) : (
//           <div className="overflow-x-auto bg-white rounded-lg shadow">
//             <table className="min-w-full text-sm">
//               <thead className="bg-gray-100 text-gray-700">
//                 <tr>
//                   {columns.map(c => (
//                     <th
//                       key={c.key}
//                       onClick={() => onSort(c.key as keyof WagerRow)}
//                       className="px-4 py-3 font-semibold text-left cursor-pointer select-none"
//                     >
//                       <span className="inline-flex items-center gap-1">
//                         {c.label}
//                         {sortKey === c.key && (
//                           <ChevronDownIcon className={`w-4 h-4 transition-transform ${sortDir === "desc" ? "rotate-180" : ""}`} />
//                         )}
//                       </span>
//                     </th>
//                   ))}
//                   <th className="px-4 py-3 font-semibold text-left">Progress (Wager → 2×)</th>
//                 </tr>
//               </thead>
//               <tbody>
//                 {paged.map((r) => {
//                   const max = r.first_deposit_amount_2x || 0;
//                   const cur = r.wager || 0;
//                   const pct = max > 0 ? clamp(cur / max) : 0;
//                   const pctText = `${(pct * 100).toFixed(2)}%`;

//                   return (
//                     <tr key={r.id} className="border-t last:border-b">
//                       <td className="px-4 py-3 whitespace-nowrap text-gray-900">{r.id}</td>
//                       <td className="px-4 py-3 whitespace-nowrap">{r.name}</td>
//                       <td className="px-4 py-3 whitespace-nowrap text-gray-700">{r.email}</td>
//                       <td className="px-4 py-3 whitespace-nowrap">{fmtAmt(r.wager)}</td>

//                       <td className="px-4 py-3">
//                         {/* Slider-style progress bar */}
//                         <div className="flex flex-col gap-1 min-w-[240px]">
//                           <div className="w-full h-3 rounded-full bg-gray-200 overflow-hidden">
//                             <div
//                               className="h-full bg-blue-600"
//                               style={{ width: `${pct * 100}%` }}
//                               aria-valuemin={0}
//                               aria-valuemax={100}
//                               aria-valuenow={pct * 100}
//                               role="progressbar"
//                             />
//                           </div>
//                           <div className="flex items-center justify-between text-xs text-gray-600">
//                             <span>Wager: {fmtAmt(cur)}</span>
//                             <span>{pctText}</span>
//                             <span>Target: {fmtAmt(max)}</span>
//                           </div>
//                           {/* Visual (non-interactive) range to mimic a slider */}
//                           {/* <input
//                             type="range"
//                             min={0}
//                             max={max || 1}
//                             value={Math.min(cur, max || 0)}
//                             readOnly
//                             className="w-full accent-blue-600 cursor-default"
//                           /> */}
//                         </div>
//                       </td>
//                     </tr>
//                   );
//                 })}
//               </tbody>
//             </table>
//           </div>
//         )}

//         {/* Pagination */}
//         {filtered.length > 0 && (
//           <div className="flex items-center justify-between text-sm text-gray-600">
//             <span>
//               Showing {Math.min((page - 1) * pageSize + 1, filtered.length)}–
//               {Math.min(page * pageSize, filtered.length)} of {filtered.length}
//             </span>
//             <div className="flex items-center gap-2">
//               <button
//                 disabled={page <= 1}
//                 onClick={() => setPage(p => Math.max(1, p - 1))}
//                 className="px-3 py-1 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed"
//               >Previous</button>
//               <span>Page {page} / {totalPages}</span>
//               <button
//                 disabled={page >= totalPages}
//                 onClick={() => setPage(p => Math.min(totalPages, p + 1))}
//                 className="px-3 py-1 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed"
//               >Next</button>
//             </div>
//           </div>
//         )}
//       </div>
//     </div>
//   );
// }

import React, { useEffect, useMemo, useState } from "react";
import { ChevronDownIcon, ArrowPathIcon, MagnifyingGlassIcon } from "@heroicons/react/24/outline";
import { apiFetch } from "../utils/api";

// ──────────────────────────────────────────────────────────────
//  NEW TYPE: Matches your new backend API
// ──────────────────────────────────────────────────────────────
type WagerRow = {
  id: number;
  name: string;
  email: string;
  wager: number;
  total_deposit: number;
  target: number;           // 2x total_deposit
  target_percentage: number; // 0 to 100+ (already calculated)
  user_created_at: string;
};

type ApiResponse = {
  success: boolean;
  data: WagerRow[];
};

// ──────────────────────────────────────────────────────────────
//  Formatters
// ──────────────────────────────────────────────────────────────
const fmtAmt = (n: number | null | undefined) => {
  if (n == null || Number.isNaN(n)) return "-";
  return new Intl.NumberFormat(undefined, { maximumFractionDigits: 4 }).format(n);
};

const fmtDateTime = (iso: string | null) => {
  if (!iso) return "-";
  const d = new Date(iso);
  if (Number.isNaN(d.getTime())) return "-";
  return d.toLocaleString();
};

// Clamp percentage to 0–100 for progress bar (optional: allow >100)
const clampPct = (n: number) => Math.max(0, Math.min(100, n));

// ──────────────────────────────────────────────────────────────
//  Columns (updated)
// ──────────────────────────────────────────────────────────────
const columns = [
  { key: "id", label: "User ID" },
  { key: "name", label: "Name" },
  { key: "email", label: "Email" },
  { key: "wager", label: "Wager" },
  { key: "total_deposit", label: "Total Deposit" },
  { key: "target", label: "Target (2x)" },
  { key: "target_percentage", label: "% Achieved" },
] as const;

export default function UserWagerReports() {
  const [rows, setRows] = useState<WagerRow[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [q, setQ] = useState("");
  const [sortKey, setSortKey] = useState<keyof WagerRow>("target_percentage");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("desc");
  const [page, setPage] = useState(1);
  const pageSize = 10;

  // ──────────────────────────────────────────────────────────────
  //  Fetch Data
  // ──────────────────────────────────────────────────────────────
  const fetchData = async () => {
    try {
      setLoading(true);
      setError(null);

      const res = await apiFetch<ApiResponse>("/api/deposits/admin/all-wagers");
      if (!res?.success) throw new Error("Failed to fetch data");

      setRows(res.data || []);
    } catch (e: any) {
      // Fallback
      try {
        const fallback = await apiFetch<ApiResponse>("https://api.air1bet.com/api/deposits/admin/all-wagers");
        if (!fallback?.success) throw new Error("Fallback failed");
        setRows(fallback.data || []);
      } catch (e2: any) {
        setError(e2?.message || "Failed to load data");
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  // ──────────────────────────────────────────────────────────────
  //  Filtering & Sorting
  // ──────────────────────────────────────────────────────────────
  const filtered = useMemo(() => {
    const term = q.trim().toLowerCase();
    const base = term
      ? rows.filter(r =>
          r.name.toLowerCase().includes(term) ||
          r.email.toLowerCase().includes(term) ||
          String(r.id).includes(term)
        )
      : rows;

    const sorted = [...base].sort((a, b) => {
      const va = a[sortKey];
      const vb = b[sortKey];
      if (va == null && vb == null) return 0;
      if (va == null) return sortDir === "asc" ? -1 : 1;
      if (vb == null) return sortDir === "asc" ? 1 : -1;

      if (typeof va === "number" && typeof vb === "number") {
        return sortDir === "asc" ? va - vb : vb - va;
      }

      const sa = String(va).toLowerCase();
      const sb = String(vb).toLowerCase();
      return sortDir === "asc" ? (sa < sb ? -1 : 1) : (sa > sb ? -1 : 1);
    });

    return sorted;
  }, [rows, q, sortKey, sortDir]);

  const totalPages = Math.max(1, Math.ceil(filtered.length / pageSize));
  const paged = filtered.slice((page - 1) * pageSize, page * pageSize);

  const onSort = (key: keyof WagerRow) => {
    if (key === sortKey) {
      setSortDir(d => (d === "asc" ? "desc" : "asc"));
    } else {
      setSortKey(key);
      setSortDir("desc");
    }
  };

  // ──────────────────────────────────────────────────────────────
  //  Render
  // ──────────────────────────────────────────────────────────────
  return (
    <div className="ml-0 lg:ml-[256px] p-2">
      <div className="max-w-7xl mx-auto p-6 space-y-6 bg-gray-50 min-h-screen">
        <div className="flex items-center justify-between">
          <h1 className="text-3xl font-bold text-gray-900">User Turnover Report</h1>
          <button
            onClick={fetchData}
            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 text-white px-4 py-2 hover:bg-blue-700 transition"
          >
            <ArrowPathIcon className="w-5 h-5" /> Refresh
          </button>
        </div>

        {/* Controls */}
        <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
          <div className="relative w-full sm:w-80">
            <MagnifyingGlassIcon className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 -translate-y-1/2" />
            <input
              value={q}
              onChange={(e) => { setPage(1); setQ(e.target.value); }}
              placeholder="Search by name, email, or ID"
              className="w-full pl-10 pr-3 py-2 rounded-lg border border-gray-300 bg-white focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
          </div>

          <div className="flex items-center gap-2">
            <label className="text-sm text-gray-600">Sort by</label>
            <select
              value={String(sortKey)}
              onChange={(e) => setSortKey(e.target.value as keyof WagerRow)}
              className="rounded-lg border border-gray-300 bg-white px-3 py-2 text-sm"
            >
              {columns.map(c => (
                <option key={c.key} value={c.key}>{c.label}</option>
              ))}
            </select>
            <button
              onClick={() => setSortDir(d => d === "asc" ? "desc" : "asc")}
              className="rounded-lg border border-gray-300 bg-white px-3 py-2 text-sm"
              title="Toggle sort direction"
            >
              {sortDir === "asc" ? "Asc" : "Desc"}
            </button>
          </div>
        </div>

        {/* Content */}
        {loading ? (
          <div className="text-gray-600 text-lg animate-pulse">Loading wagers…</div>
        ) : error ? (
          <div className="text-red-600">{error}</div>
        ) : filtered.length === 0 ? (
          <div className="text-gray-600">No results.</div>
        ) : (
          <div className="overflow-x-auto bg-white rounded-lg shadow">
            <table className="min-w-full text-sm">
              <thead className="bg-gray-100 text-gray-700">
                <tr>
                  {columns.map(c => (
                    <th
                      key={c.key}
                      onClick={() => onSort(c.key as keyof WagerRow)}
                      className="px-4 py-3 font-semibold text-left cursor-pointer select-none hover:bg-gray-200 transition"
                    >
                      <span className="inline-flex items-center gap-1">
                        {c.label}
                        {sortKey === c.key && (
                          <ChevronDownIcon className={`w-4 h-4 transition-transform ${sortDir === "desc" ? "rotate-180" : ""}`} />
                        )}
                      </span>
                    </th>
                  ))}
                  <th className="px-4 py-3 font-semibold text-left">Progress</th>
                </tr>
              </thead>
              <tbody>
                {paged.map((r) => {
                  const pct = clampPct(r.target_percentage); // 0–100
                  const over = r.target_percentage > 100 ? `+${(r.target_percentage - 100).toFixed(1)}%` : "";

                  return (
                    <tr key={r.id} className="border-t hover:bg-gray-50 transition">
                      <td className="px-4 py-3 whitespace-nowrap text-gray-900">{r.id}</td>
                      <td className="px-4 py-3 whitespace-nowrap">{r.name}</td>
                      <td className="px-4 py-3 whitespace-nowrap text-gray-700">{r.email}</td>
                      <td className="px-4 py-3 whitespace-nowrap">{fmtAmt(r.wager)}</td>
                      <td className="px-4 py-3 whitespace-nowrap">{fmtAmt(r.total_deposit)}</td>
                      <td className="px-4 py-3 whitespace-nowrap">{fmtAmt(r.target)}</td>
                      <td className="px-4 py-3 whitespace-nowrap">
                        <span className={r.target_percentage >= 100 ? "text-green-600 font-medium" : ""}>
                          {r.target_percentage.toFixed(2)}%
                          {over && <span className="text-green-500 ml-1"> {over}</span>}
                        </span>
                      </td>
                      <td className="px-4 py-3">
                        <div className="flex flex-col gap-1 min-w-[200px]">
                          <div className="w-full h-3 rounded-full bg-gray-200 overflow-hidden relative">
                            <div
                              className={`h-full transition-all duration-300 ${r.target_percentage >= 100 ? "bg-green-500" : "bg-blue-600"}`}
                              style={{ width: `${pct}%` }}
                            />
                            {r.target_percentage > 100 && (
                              <div className="absolute inset-0 flex items-center justify-end pr-1">
                                <span className="text-xs text-white font-bold">100%+</span>
                              </div>
                            )}
                          </div>
                          <div className="flex items-center justify-between text-xs text-gray-600">
                            <span>{fmtAmt(r.wager)}</span>
                            <span className="font-medium">{r.target_percentage.toFixed(1)}%</span>
                            <span>{fmtAmt(r.target)}</span>
                          </div>
                        </div>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        )}

        {/* Pagination */}
        {filtered.length > 0 && (
          <div className="flex items-center justify-between text-sm text-gray-600">
            <span>
              Showing {Math.min((page - 1) * pageSize + 1, filtered.length)}–
              {Math.min(page * pageSize, filtered.length)} of {filtered.length}
            </span>
            <div className="flex items-center gap-2">
              <button
                disabled={page <= 1}
                onClick={() => setPage(p => Math.max(1, p - 1))}
                className="px-3 py-1 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed"
              >Previous</button>
              <span>Page {page} / {totalPages}</span>
              <button
                disabled={page >= totalPages}
                onClick={() => setPage(p => Math.min(totalPages, p + 1))}
                className="px-3 py-1 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed"
              >Next</button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
