const _ = require("lodash");
const assert = require("assert");
const md5 = require("md5");
const Result = require("./Result");
const Rule = require("../Rule");
const UserRule = require("../../Users/Rule");
const Model = require("../../General/Model");
const Socket = require("../../General/Socket");
const { encode } = require("../../General/Buffer");
const H = require("../../General/Helper");
const C = require("../../General/Constant");
const config = require("../../config");

const Players = require("./Players");
const Winners = require("./Winners");

const houseEdge = config.house;

var io = null,
  startTimeWaiting = null,
  startTimeStarted = null,
  status = null,
  bustedNumber = null,
  gameMd5 = null,
  history = [],
  hash = H.randomString(64),
  gameID = null,
  autoCashoutTimer = [],
  crashTimeout = null;

const Crash = {};

Crash.idle = function (io) {
  wait(io);
};

wait = function (io) {
  var j = 0;
  for (j in autoCashoutTimer) {
    clearTimeout(autoCashoutTimer[j]);
  }

  status = "waiting";
  onWaiting(io);
  H.wait(5000).then(() => {
    start(io);
  });
};

start = function (io) {
  status = "started";
  onStarted(io);
  H.wait(crashTimeout).then(() => {
    bust(io);
  });
};

bust = function (io) {
  status = "busted";
  onBusted(io);
  H.wait(5000).then(() => {
    wait(io);
  });
};

/*
 * Events when game is waiting
 */
onWaiting = function (io) {
  startTimeWaiting = new Date();
  gameMd5 = md5(hash);
  gameID = H.makeGameID();

  io.emit(
    C.WAITING_CRASH,
    encode({
      time: 5000,
      hash: hash,
      md5: gameMd5,
      game_id: gameID,
      players: Players.all(),
    })
  );

  //Bots Playing
  let currentHour = new Date().getHours();

  UserRule.getBots("crash", currentHour, (result) => {
    if (!result) return;
    if (_.isUndefined(result)) return;

    result.forEach((bot, i) => {
      let id = _.toNumber(bot.id);
      let coin = _.lowerCase(bot.coin);
      let amount = H.getRandomInt(2000) / 100000000;

      if (coin === "doge") amount = H.getRandomInt(10000000000) / 100000000;

      if (coin === "trx") amount = H.getRandomInt(1000000000) / 10000000;

      if (
        coin === "usdt" &&
        coin === "usdp" &&
        coin === "usdc" &&
        coin === "tusd" &&
        coin === "busd"
      )
        amount = H.getRandomInt(1000000000) / 100000000;

      UserRule.getUserInfo(id, (user) => {
        if (user) {
          let data = {
            payout: H.getRandomBetween(150, 600),
            amount: H.CryptoSet(amount, coin),
            coin: coin,
            isB: true,
          };

          H.wait(H.getRandomBetween(100, 2300)).then(() => {
            Crash.add(io, id, data, user);
          });
        }
      });
    });
  });
};

/*
 * Events when game is started
 */
onStarted = function (io) {
  io.emit(
    C.STARTED_CRASH,
    encode({
      players: Players.all(),
      md5: gameMd5,
      time: 100,
    })
  );

  startTimeStarted = new Date();

  //Get bankroll amount
  // UserRule.getFullBankRoll((bank) => {

  //     let allPlayers = Players.all();
  //     let playersAmount = 0;
  //     let bankAmount = 0;

  //     //Get round amount
  //     if (allPlayers.length !== 0) {
  //         allPlayers.forEach((player, i) => {
  //             bankAmount += bank[_.lowerCase(player.coin)];
  //             playersAmount += _.toNumber(player.amount);
  //         })
  //     }

  // var result = Result.generateResult(hash, playersAmount, _.toNumber(bankAmount));
  var result = Result.generateResult(hash, [], []);

  hash = result.hash;
  bustedNumber = result.crash;
  crashTimeout = Result.calculateTimeout(result.crash); // timeout

  //Make a Crash Record
  Model.query(
    "INSERT INTO crashs(hash, gid) VALUES($1, $2)",
    [hash, gameID],
    function (err, r) {
      if (err) {
        return console.log("CRASH RECORD ERROR: 122", err);
      }
    }
  );

  let players = Players.all();

  //Auto Cashout Players
  if (players.length !== 0) {
    players.forEach((player, i) => {
      autoCashoutTimer[i] = setTimeout(() => {
        Crash.out(io, _.toNumber(player.uid), player);
      }, Result.calculateTimeout(_.toNumber(player.cashout) / 100));
    });
  }
  // })
};

/*
 * Events when game is busted
 */
onBusted = function (io) {
  let data = {
    time: crashTimeout,
    md5: gameMd5,
    hash: hash,
    players: Players.all(),
    winners: Winners.all(),
    amount: bustedNumber * 100,
    game_id: gameID,
  };

  //Send Busted to All Players
  io.emit(C.BUSTED_CRASH, encode(data));

  var record = {
    amount: bustedNumber * 100,
    game_id: gameID,
  };

  history.push(record);

  if (history.length >= 10) history = _.drop(history, history.length - 10);

  Model.query(
    "UPDATE bets SET result = $1, hash = $2 WHERE gid = $3",
    [bustedNumber, hash, gameID],
    function (err) {
      if (err) {
        return console.log("Error on Busted from CRASH: 190", err);
      }
    }
  );

  /**
   * Update Crash Record
   */
  Model.query(
    "UPDATE crashs SET busted = $1 WHERE gid = $2",
    [bustedNumber, gameID],
    function (err) {
      if (err) {
        return console.log("Error on crashs: 179", err);
      }
    }
  );

  //Clear players and winners
  Players.clear();
  Winners.clear();
};

/*
 * Add Player to the Game
 */
Crash.add = function (io, id, data, botinfo = null) {
  let { amount, coin, payout, isB } = data;

  if (!coin && !amount && !payout && !id) return;

  id = _.toNumber(id);
  payout = _.toNumber(payout);

  if (status !== "waiting") {
    // Send alert to client
    if (Socket.get(id)) {
      Socket.get(id).emit(
        C.ERROR_CRASH,
        encode({
          uid: id,
          message: "Game is Running",
        })
      );
    }
    return;
  }

  //Check if user has in queue ( security )
  let check = Players.exists(id);

  if (check) return console.log("exists", id);

  //Security
  payout = Math.max(parseFloat(1.01 * 100).toFixed(2), payout);

  let c;

  if (botinfo === null) c = Socket.get(id);
  else c = null;

  Rule.CanPlay(id, data, c, "crash", (status, err) => {
    if (status !== true) {
      if (Socket.get(id)) {
        return Socket.get(id).emit(
          C.ERROR_CRASH,
          encode({
            uid: id,
            message: status,
            code: err,
          })
        );
      }
    }

    coin = _.lowerCase(coin);
    amount = H.CryptoSet(_.toNumber(amount), coin);

    assert(amount > 0);

    UserRule.getUserOrBot(id, botinfo, (user, er) => {
      if (!user) return;
      if (er) return console.log("er");

      //Make a Bet Record
      Model.query(
        "INSERT INTO bets(game, gid, name, uid, coin, amount, profit) VALUES($1, $2, $3, $4, $5, $6, $7)",
        ["crash", gameID, user.name, id, coin, amount, H.CryptoSet(0.0, coin)],
        function (er, result) {
          if (er) {
            return console.log("ERROR ON BET CRASH: 288", er);
          }
        }
      );
      
      //Reduce Credit
      UserRule.reduceBalance(id, amount, coin, (newBalance, err) => {
        if (err) {
          return console.log("error on Reduce BALANCE CRASH: 296");
        }

        //Send Credit Change to Client
        if (botinfo === null) {
          if (Socket.get(id))
            Socket.get(id).emit(
              C.UPDATE_CREDIT,
              encode({
                coin: coin,
                value: newBalance,
              })
            );
        }

        UserRule.addBankRoll("crash", amount, coin, c, (status) => {
          //Temporary Reduce Profit
          UserRule.updateProfit(isB, false, id, amount, coin, (isOk) => {
            if (!isOk) {
              return console.log("Error updateProfit on Crash: 307");
            } else {
              var tempData = {
                uid: id,
                isB: isB,
                session: "crash",
                status: "playing",
                name: user.name,
                avatar: user.avatar,
                hash: hash,
                coin: coin,
                gid: gameID,
                amount: amount,
                cashout: payout,
                created: new Date(),
                in: true,
              };

              Players.set(tempData);
              io.emit(C.PLAY_CRASH, encode(tempData));
            }
          });
        });
      });
    });
  });
};

/*
 * Cashout Player
 */
Crash.out = function (io, id, data) {
  if (status !== "started") return;

  let kiked = startTimeStarted;
  let fix = false;

  // Cashout Time ( new method )
  if (!_.isUndefined(data)) {
    if (!_.isUndefined(data.token2)) {
      var token2 = data.token2;
      var c = token2.substring(token2.length, token2.length - 4);
      kiked = c;
      fix = true;
    }
  }

  id = _.toNumber(id);

  var player = Players.get(id);

  if (!player) {
    console.log("no player crash -347", id);
    return;
  }

  if (player.in != true) return;
  if (!player.amount) return;
  if (!player.coin) return;

  var amount = _.toNumber(player.amount);
  var coin = _.lowerCase(player.coin);

  var result = Result.calculateWinning(amount, kiked, fix);
  var profit = _.toNumber(result.won);
  var cashout = result.cashout * 100;

  var temp = player;
  temp.won = profit;
  temp.hash = hash;
  temp.in = false;
  temp.current = cashout;

  //Add To Winenrs
  Winners.set(temp);

  //Broadcast To All Clients
  io.emit(C.FINISH_CRASH, encode(temp));

  //Update Bet Record
  Model.query(
    "UPDATE bets SET profit = $1, cashout = $2 WHERE gid = $3 AND uid = $4",
    [H.CryptoSet(profit, coin), cashout, gameID, id],
    function (err) {
      if (err) {
        return console.log("Error on crash out from CRASH: 434", err);
      }
    }
  );

  //Calculate HouseEdge
  var percent = (houseEdge / 100) * H.CryptoSet(amount);
  var calculateHouse = H.CryptoSet(amount - percent);

  var amountAndProfit = profit + _.toNumber(calculateHouse);
  amountAndProfit = H.CryptoSet(amountAndProfit);

  //Remove From Players
  Players.remove(id);

  // If is bot, return from here
  if (player.isB === true) {
    UserRule.updateProfit(true, true, id, amountAndProfit, coin, (isOk) => {});
    return;
  }

  //Add Balance
  UserRule.addBalance(id, amountAndProfit, coin, (newBalance, error) => {
    if (error) {
      return console.log("Error Ading Balance on Crash: 408");
    }

    UserRule.reduceBankRoll("crash", amountAndProfit, coin, (status, err) => {
      if (err) {
        return console.log("Error reduceBankRoll on Crash: 413");
      }
      UserRule.updateProfit(false, true, id, amountAndProfit, coin, (isOk) => {
        if (!isOk) {
          return console.log("Error updateProfit on Crash: 417");
        }
        //Send Credit Change to Client
        H.wait(250).then(() => {
          if (Socket.get(id)) {
            Socket.get(id).emit(
              C.UPDATE_CREDIT,
              encode({
                coin: coin,
                value: newBalance,
              })
            );
          }
        });
      });
    });
  });
};

/*
 * Get Status
 */
Crash.onStatus = function (client) {
  var timer,
    n = new Date();

  if (status === "waiting") {
    timer = 5000 - (n - startTimeWaiting);
  } else {
    timer = n - startTimeStarted;
  }

  H.wait(500).then(() => {
    client.emit(
      C.STATUS_CRASH,
      encode({
        time: timer,
        players: Players.all(),
        winners: Winners.all(),
        crashes: history,
        status: status,
        md5: gameMd5,
        amount: bustedNumber * 100,
      })
    );
  });
};

/*
 * Get then Players
 */
Crash.onPlayers = function (client) {
  H.wait(500).then(() => {
    client.emit(
      C.PLAYERS_CRASH,
      encode({
        players: Players.all(),
        winners: Winners.all(),
      })
    );
  });
};

/*
 * Get the Full History
 */
Crash.onHistory = function (client) {
  Model.query(
    "SELECT busted, hash, date, gid FROM crashs WHERE CAST(crashs.busted as TEXT) != $1 ORDER by date DESC LIMIT 50",
    [""],
    function (err, results) {
      if (err) {
        console.log("error on Crash Bets: 247", err);
        return;
      }
      client.emit(
        C.HISTORY_CRASH,
        encode({
          history: results.rows,
        })
      );
    }
  );
};

module.exports = Crash;
