const _ = require("lodash");
const assert = require("assert");
const Rule = require("../Rule");
const Hash = require("../Hash");
const Queue = require("../../General/Queue");
const { encode } = require("../../General/Buffer");
const C = require("../../General/Constant");
const H = require("../../General/Helper");
const Result = require("./Result");

class Mine {
  constructor(io, client, data, id) {
    this.io = io;
    this.client = client;
    this.data = data;
    this.id = _.toNumber(id);
  }

  /*
   * Send Data To Client
   */
  emit(data) {
    this.client.emit(C.PLAY_MINE, encode(data));
  }

  /*
   * Intialize Game
   */
  init() {
    let { command } = this.data;

    switch (command) {
      case "play":
        return this.play();

      case "clicked":
        return this.clicked();

      case "cashout":
        return this.end();
    }
  }

  /*
   * Play Game
   */
  play() {
    let self = this;

    Rule.CanPlay(self.id, self.data, self.client, "mine", (status, err) => {
      if (status !== true) {
        return this.emit({
          command: "error",
          uid: self.id,
          message: status,
          code: err,
        });
      }

      Rule.preparePlay(self.client, self.id, "mine", self.data, (isOk) => {
        if (!isOk) {
          console.log("Error mine Playing: 21");
          return false;
        }

        //Game Choosed Mines
        let { mine } = self.data;

        if (!mine) return;

        mine = parseFloat(mine);

        // Validate mines length ( Security )
        const allowed = [3, 5, 10, 24];

        if (!_.includes(allowed, parseInt(mine))) {
          return console("mine hack !");
        }

        //Generate Random Hash and Result
        let rand = Hash.make();
        let hash = rand.hash;
        let result = Result.generateMines(hash, mine);

        //Update Queue and Save Result, Mines, Hash
        Queue.update(self.id, "uid", "mines", mine);
        Queue.update(self.id, "uid", "result", result);
        Queue.update(self.id, "uid", "hash", hash);
        Queue.update(self.id, "uid", "selected", []);
        Queue.update(self.id, "uid", "timeClicked", 0);

        //Send Play Game
        self.emit({
          command: "play",
        });
      });
    });
  }

  /*
   * Click On the Land
   */
  clicked() {
    //Choosed Land
    let { land } = this.data;
    

    // if (!land) return;
    if (land===undefined || land ===null) return;

    land = parseInt(land);
    if (isNaN(land)) return;

    

    //Get The Client Game Info
    let info = Queue.getOne("uid", this.id);

    if (!info) {
      console.log("Client Not Playing!", this.id);
      return false;
    }

    let hash = info.hash;
    let mines = info.mines;
    let result = info.result;
    let selected = info.selected;
    let coin = info.data.coin;
    let amount = info.data.amount;
    let timeClicked = info.timeClicked;
    timeClicked += 1;

    Queue.update(this.id, "uid", "timeClicked", timeClicked);

    if (!amount || !coin || !result || !mines) return;

    land = parseInt(land);

    // Check repeated land ( Security )
    if (_.includes(selected, land)) {
      return;
    } else {
      selected.push(land);
    }

    //Check Limit Profit
    Rule.checkLimited(this.id, this.data.coin, (status) => {
      var bomb = false;
      result.forEach((number, i) => {
        if (parseFloat(land) === parseFloat(number.num)) bomb = true;
      });

      // Limit Profit
      // if (!status) {
      //     result = _.drop(result, 1);
      //     result.push({
      //         num: land,
      //         hash: hash
      //     });
      //     bomb = true;
      // }

      if (bomb) {
        //Bomb ! let's End Game.
        return this.lost(result, hash);
      } else {
        //Safe ! let's Send Bonus.
        return this.setBonus(amount, coin, land, mines);
      }
    });
  }

  /*
   * Set Bonus
   */
  setBonus(amount, coin, land, mines) {
    let profit = (_.toNumber(amount) * mines) / 10;
    profit = H.CryptoSet(profit, coin);
    return this.emit({
      command: "clicked",
      profit: profit,
      coin: coin,
      id: land,
    });
  }

  /*
   * End Game / Client Lost
   */
  lost(result, hash) {
    let info = Queue.getOne("uid", this.id);

    let profit = -info.data.amount;
    let isWinner = false;
    Rule.prepareBusted(
      this.client,
      this.io,
      this.id,
      "mine",
      this.data,
      profit,
      result,
      hash,
      isWinner,
      (res) => {
        return this.emit({
          command: "busted",
          profit: profit,
          mines: result,
        });
      }
    );
  }

  /*
   * CashOut / End the Game / Client Win
   */
  end() {
    // let { profit } = this.data;

    //Get The Client Game Info

    let info = Queue.getOne("uid", this.id);
    console.log(info,"---->>>>>>>>>>>info")

    if (!info) {
      console.log("Client Not Playing!", this.id);
      return false;
    }

    let profit = (_.toNumber(info.data.amount) * info.mines) / 10;
    console.log(profit,"------>>>>>profitttt")

    profit = profit * info.timeClicked;

    let result = info.result;
    let mines = info.mines;
    let hash = info.hash;

    let isWinner = true;

    Rule.prepareBusted(
      this.client,
      this.io,
      this.id,
      "mine",
      info.data,
      profit,
      result,
      hash,
      isWinner,
      (res) => {
        if (res) {
          this.emit({
            command: "cashout",
            profit: profit,
            mines: result,
          });
        }
      }
    );
  }
}

module.exports = Mine;
