const _ = require('lodash');
const assert = require('assert');
const Rule = require('../Rule');
const UserRule = require('../../Users/Rule');
const Hash = require('../Hash');
const Queue = require('../../General/Queue');
const { encode } = require('../../General/Buffer');
const C = require('../../General/Constant');
const H = require('../../General/Helper');
const Result = require('./Result');

class SnakeAndLadders {
    constructor(io, client, data, id) {
        this.io = io
        this.client = client
        this.data = data
        this.id = _.toNumber(id)
    }

    /*
     * Send Data To Client
     */
    emit(data) {
        this.client.emit(C.PLAY_SNAKEANDLADDERS, encode(data))
    }

    /*
     * Intialize Game
     */
    init() {
        let { command } = this.data;

        switch (command) {
            case 'play':
                return this.play();

            case 'clicked':
                return this.clicked();

            case 'cashout':
                return this.cashout();

            case 'next':
                return this.next();
        }
    }

    /*
     * Play Game
     */
    play() {
        let self = this;

        Rule.CanPlay(self.id, self.data, self.client, 'snakeandladders', (can, err) => {
            if (can !== true) {
                return self.emit({
                    command: 'error',
                    uid: self.id,
                    message: can,
                    code: err
                })
            }

            Rule.preparePlay(self.client, self.id, 'snakeandladders', self.data, (isOk) => {
                if (!isOk) {
                    console.log('Error mine Playing: 21')
                    return false;
                }

                Rule.checkLimited(self.id, self.data.coin, (status) => {

                    //Generate Result
                    let random = Result.generateDice(status, 89);

                    let result = random.result;
                    let hash = random.hash;

                    //Update Queue
                    Queue.update(self.id, 'uid', 'result', result);
                    Queue.update(self.id, 'uid', 'hash', hash);
                    Queue.update(this.id, 'uid', 'profit', 0);

                    //Send Play Game
                    self.emit({
                        command: 'play',
                        target: result
                    })
                })
            })
        })
    }

    /*
     * Click on the Land
     */
    clicked() {
        let { land } = this.data;

        if (!land) return console.log('no land !')

        land = parseFloat(land);

        //Get The Client Game Info
        let info = Queue.getOne('uid', this.id)

        if (!info) {
            console.log('Client Not Playing!', this.id)
            return false;
        }

        let hash = info.hash;
        let result = info.result;
        let coin = info.data.coin;
        let amount = info.data.amount;

        if (!amount || !coin || !result) return;

        let lost = false;

        //Traps, Snaks
        if (land === 66 || land === 62 || land === 44 || land === 28 || land === 28 || land === 26 || land === 80 || land === 22 || land === 11 || land === 45 || land === 84)
            lost = true;

        if (lost) {
            //let's End Game.
            let profit = 0.00000000;
            let isWinner = false;
            Rule.prepareBusted(this.client, this.io, this.id, 'snakeandladders', this.data, profit, result, hash, isWinner, (res) => {
                return this.emit({
                    command: 'busted'
                });
            })
        }
        else {
            //let's Send Bonus.
            let profit = (_.toNumber(amount) * land) / 10;
            profit = H.CryptoSet(profit, coin);

            let plus = _.toNumber(info.profit) + _.toNumber(profit);
            Queue.update(this.id, 'uid', 'profit', plus);

            this.emit({
                command: 'clicked',
                profit: profit,
                coin: coin
            })
        }
    }

    /*
     * Next roll
     */
    next() {
        //Get The Client Game Info
        let info = Queue.getOne('uid', this.id)

        if (!info) {
            console.log('Client Not Playing!', this.id)
            return false;
        }

        UserRule.getMaxAndCurrentProfit(this.id, (p) => {
            let status = false;

            let plus = parseFloat(p.current) + parseFloat(this.data.profit)

            if (parseFloat(p.max) > plus)
                status = true;

            //Generate Result
            let random = Result.generateDice(status, this.data.currentLand);

            let result = random.result;
            let hash = random.hash;

            //Update Queue
            Queue.update(this.id, 'uid', 'result', result);
            Queue.update(this.id, 'uid', 'hash', hash);

            //Send Play Game
            this.emit({
                command: 'next',
                target: result
            })

        })
    }

    /*
     * CashOut / End the Game / Client Win
     */
    cashout() {
        // let { profit } = this.data;

        //Get The Client Game Info
        let info = Queue.getOne('uid', this.id)

        if (!info) {
            console.log('Client Not Playing!', this.id)
            return false;
        }

        let result = info.result;
        let hash = info.hash;
        let profit = info.profit;

        let isWinner = true;

        Rule.prepareBusted(this.client, this.io, this.id, 'snakeandladders', info.data, profit, result, hash, isWinner, (res) => {
            if (res) {
                this.emit({
                    command: 'cashout'
                })
            }
        })
    }
}

module.exports = SnakeAndLadders;