const assert = require("assert");
const _ = require("lodash");
const Rule = require("../Rule");
const Hash = require("../Hash");
const Queue = require("../../General/Queue");
const { encode } = require("../../General/Buffer");
const C = require("../../General/Constant");
const H = require("../../General/Helper");
const Result = require("./Result");

class Tower {
  constructor(io, client, data, id) {
    this.io = io;
    this.data = data;
    this.client = client;
    this.id = _.toNumber(id);
  }

  /*
   * Send Data To Client
   */
  emit(data) {
    this.client.emit(C.PLAY_TOWER, encode(data));
  }

  /*
   * Intialize Game
   */
  init() {
    let { command } = this.data;

    switch (command) {
      case "play":
        return this.play();

      case "clicked":
        return this.clicked();

      case "cashout":
        return this.cashout();
    }
  }

  /*
   * Play Game
   */
  play() {
    let self = this;

    Rule.CanPlay(self.id, self.data, self.client, "tower", (can, err) => {
      if (can !== true) {
        return this.emit({
          command: "error",
          uid: self.id,
          message: can,
          code: err,
        });
      }

      Rule.preparePlay(self.client, self.id, "tower", self.data, (isOk) => {
        if (!isOk) {
          console.log("Error Tower Playing: 59");
          return false;
        }

        //Generate Random Hash and Result
        let rand = Hash.make();
        let hash = rand.hash;
        let result = Result.generate(hash, 4);

        //Update Queue and Save Result, Mines, Hash
        Queue.update(self.id, "uid", "result", result);
        Queue.update(self.id, "uid", "hash", hash);
        Queue.update(self.id, "uid", "selected", []);
        Queue.update(self.id, "uid", "timeClicked", 0);

        //Send Play Game
        this.emit({
          command: "play",
        });
      });
    });
  }

  /*
   * Click On the Land
   */
  clicked() {
    //Choosed Land
    let { land } = this.data;

    if (!land) return;

    //Get The Client Game Info
    let info = Queue.getOne("uid", this.id);

    if (!info) {
      console.log("Client Not Playing!");
      return false;
    }

    let hash = info.hash;
    let result = info.result;
    let selected = info.selected;
    let coin = info.data.coin;
    let amount = info.data.amount;
    let timeClicked = info.timeClicked;
    timeClicked += 1;

    Queue.update(this.id, "uid", "timeClicked", timeClicked);

    if (!amount || !coin || !result) return;

    land = parseInt(land);

    //Check Limit Profit
    Rule.checkLimited(this.id, this.data.coin, (status) => {
      var bomb = false;

      result.forEach((number, i) => {
        if (parseFloat(land) === parseFloat(number.num)) bomb = true;
      });

      // if (!status || !bomb) {
      //     result = _.drop(result, 1);
      //     result.push({
      //         num: land,
      //         hash: hash
      //     });
      //     bomb = true;
      // }

      if (bomb) {
        //Bomb ! let's End Game.
        return this.lost(result, hash);
      } else {
        //Safe ! let's Send Bonus.
        return this.setBonus(amount, coin, land);
      }
    });
  }

  /*
   * Set Bonus
   */
  setBonus(amount, coin, land) {
    var info = Queue.getOne("uid", this.id);
    let result = Result.generate(info.hash, 4);
    Queue.update(this.id, "uid", "result", result);
    let profit = (_.toNumber(amount) * 3) / 10;
    profit = H.CryptoSet(profit, coin);
    return this.emit({
      command: "clicked",
      profit: profit,
      coin: coin,
      id: land,
    });
  }

  /*
   * End Game / Client Lost
   */
  lost(result, hash) {
    var info = Queue.getOne("uid", this.id);
    let profit = -info.data.amount;
    let isWinner = false;
    Rule.prepareBusted(
      this.client,
      this.io,
      this.id,
      "tower",
      this.data,
      profit,
      result,
      hash,
      isWinner,
      (res) => {
        return this.emit({
          command: "busted",
          profit: profit,
          mines: result,
        });
      }
    );
  }

  /*
   * Cashout the  Game
   */
  cashout() {
    // let { profit } = this.data;

    //Get The Client Game Info
    let info = Queue.getOne("uid", this.id);

    if (!info) {
      console.log("Client Not Playing!", this.id);
      return false;
    }

    let profit = (_.toNumber(info.data.amount) * 3) / 10;
    profit = profit * info.timeClicked;

    let result = info.result;
    let hash = info.hash;

    let isWinner = true;

    Rule.prepareBusted(
      this.client,
      this.io,
      this.id,
      "mine",
      info.data,
      profit,
      result,
      hash,
      isWinner,
      (res) => {
        if (res) {
          this.emit({
            command: "cashout",
            profit: profit,
          });
        }
      }
    );
  }
}

module.exports = Tower;
