const express = require('express');
const pg = require('../General/Model');

const router = express.Router();

// Get balance for a specific coin by uid and coin symbol
router.get('/api/balance', async (req, res) => {
  try {
    const { uid, coin_symbol } = req.query;

    // Validate input
    if (!uid || !coin_symbol) {
      return res.status(400).json({ error: 'uid and coin_symbol are required' });
    }

    // List of valid coin symbols based on credits table
    const validCoins = [
      'btc', 'eth', 'ltc', 'bch', 'usdt', 'trx', 'doge', 'ada', 'xrp', 'bnb',
      'usdp', 'nexo', 'mkr', 'tusd', 'usdc', 'busd', 'nc', 'inr', 'shib',
      'matic', 'sc', 'mvr', 'bjb', 'aed', 'npr', 'pkr'
    ];

    // Validate coin_symbol
    if (!validCoins.includes(coin_symbol.toLowerCase())) {
      return res.status(400).json({ error: 'Invalid coin_symbol' });
    }

    // Query to fetch balance
    const query = `
      SELECT ${coin_symbol} AS balance
      FROM credits
      WHERE uid = $1
    `;
    const result = await pg.query(query, [uid]);

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const balance = result.rows[0].balance || 0;
    res.json({
      uid,
      coin_symbol,
      balance: parseFloat(Number(balance).toFixed(8)) // Format to 8 decimal places
    });
  } catch (error) {
    console.error('Error fetching balance:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

module.exports = router;