const crypto = require('crypto');
const axios = require('axios');
const pg = require('../General/Model');

class PaymentService {
    constructor(config) {
        this.DEPOSIT_API_KEY = config.DEPOSIT_API_KEY;
        this.WEBHOOK_ACCESS_KEY = config.WEBHOOK_ACCESS_KEY;
        this.WEBHOOK_PRIVATE_KEY = config.WEBHOOK_PRIVATE_KEY;
        this.BASE_URL = config.BASE_URL || 'https://pay-crm.com';
        this.PROJECT_ID = config.PROJECT_ID;
    }

    // Signature generation for webhook verification
   
    generateWebhookSignature(transactions) {
        // Use JSON.stringify with specific options to match PHP's JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
        const transactionsJson = JSON.stringify(transactions, null, 0)
            .replace(/\\"/g, '"')  // Remove escaped quotes
            .replace(/\\\//g, '/');  // Remove escaped slashes
    
        const md5Transactions = crypto
            .createHash('md5')
            .update(transactionsJson)
            .digest('hex');
        
        return crypto
            .createHash('sha1')
            .update(`${this.WEBHOOK_ACCESS_KEY}${this.WEBHOOK_PRIVATE_KEY}${md5Transactions}`)
            .digest('hex');
    }

    async createDeposit(depositData) {
        try {
            // Validate required fields
            const requiredFields = [
                'amount', 'currency', 'payment_system',
                'custom_user_id'
            ];
            requiredFields.forEach(field => {
                if (!depositData[field]) {
                    throw new Error(`Missing required field: ${field}`);
                }
            });

            const payload = {
                amount: depositData.amount,
                currency: depositData.currency,
                payment_system: depositData.payment_system || "phonepe",
                custom_transaction_id: depositData.custom_transaction_id || '',
                custom_user_id: depositData.custom_user_id,
                return_url: depositData.return_url,
                language: "EN",
                webhook_id:2472417,
            };

            const response = await axios.post(
                `${this.BASE_URL}/Remotes/create-payment-page`, 
                payload,
                {
                    headers: {
                        'apikey': this.DEPOSIT_API_KEY
                    }
                }
            );

            if (response.data.success) {
                // Store deposit in database
                const createQuery = `
                    INSERT INTO apaydeposits 
                    (order_id, user_id, amount, currency, payment_system, 
                     custom_transaction_id, status, payment_details)
                    VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
                    RETURNING *
                `;
                const values = [
                    response.data.order_id,
                    depositData.custom_user_id,
                    depositData.amount,
                    depositData.currency,
                    "upi",
                    depositData.custom_transaction_id || null,
                    'Pending',
                    JSON.stringify(response.data || {})
                ];

                await pg.query(createQuery, values);
                return response.data;
            } else {
                // Handle failed deposit
                const failedQuery = `
                    INSERT INTO apaydeposits 
                    (order_id, user_id, amount, currency, payment_system, 
                     custom_transaction_id, status, error_reason)
                    VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
                `;
                const failedValues = [
                    crypto.randomBytes(16).toString('hex'),
                    depositData.custom_user_id,
                    depositData.amount,
                    depositData.currency,
                    depositData.payment_system,
                    depositData.custom_transaction_id || null,
                    'Failed',
                    response.data.message || 'Unknown error'
                ];
                await pg.query(failedQuery, failedValues);

                throw new Error(response.data.message);
            }
        } catch (error) {
            console.error('Deposit creation error:', error);
            throw error;
        }
    }

    async getDepositInfo(orderId) {
        try {
            const response = await axios.get(`${this.BASE_URL}/Remotes/deposit-info`, {
                params: {
                    project_id: this.PROJECT_ID,
                    order_id: orderId
                },
                headers: {
                    'apikey': this.DEPOSIT_API_KEY
                }
            });

            return response.data;
        } catch (error) {
            console.error('Deposit info retrieval error:', error);
            throw error;
        }
    }

    async processWebhook(webhookData) {
        try {
            const { access_key, signature, transactions } = webhookData;
            console.log(webhookData);
            // Validate access key
            if (access_key !== this.WEBHOOK_ACCESS_KEY) {
                throw new Error('Invalid webhook access key');
            }

            // Verify webhook signature with precise matching
            const expectedSignature = this.generateWebhookSignature(transactions);
            if (signature !== expectedSignature) {
                console.error('Signature Mismatch', {
                    received: signature,
                    expected: expectedSignature
                });
                throw new Error('Invalid webhook signature');
            }

            // Transaction processing logic remains the same
            for (const transaction of transactions) {
                const { order_id, status, amount, custom_user_id, currency } = transaction;

                // Update deposit status
                await this.updateDepositStatus(order_id, status, webhookData);

                // Handle transaction based on status
                if (status === 'Success') {
                    await this.creditUserBalance(custom_user_id, amount,currency);
                } else if (['Failed', 'Rejected'].includes(status)) {
                    await this.handleFailedTransaction(order_id, status);
                }
            }

            return { status: 'OK' };
        } catch (error) {
            console.error('Webhook processing error:', error);
            throw error;
        }
    }
    async updateDepositStatus(orderId, status, webhookData) {
        const updateQuery = `
            UPDATE apaydeposits 
            SET status = $1, 
                updated_at = CURRENT_TIMESTAMP,
                webhook_response = $2
            WHERE order_id = $3
        `;
        await pg.query(updateQuery, [status, JSON.stringify(webhookData), orderId]);
    }
    async creditUserBalance(userId, amount, currency) {
        const creditQuery = `
            UPDATE credits
            SET ${currency.toLowerCase()} = COALESCE(${currency.toLowerCase()}, 0) + $1
            WHERE uid = $2
        `;
        await pg.query(creditQuery, [amount, userId]);
    }
    async handleFailedTransaction(orderId, status) {
        const failQuery = `
            UPDATE apaydeposits 
            SET status = $1, 
                error_reason = $2
            WHERE order_id = $3
        `;
        await pg.query(failQuery, [status, 'Payment gateway rejection', orderId]);
    }
}

module.exports = PaymentService;