const pg = require('../General/Model');
// Get user transaction history
const axios = require('axios');

const getTransactionHistory = async (req, res) => {
  try {
    const { page = 1, limit = 10, search = '', userId = null } = req.query;
    const offset = (Number(page) - 1) * Number(limit);

    // Fetch live game history from the API
    let liveTransactions = [];
    try {
      const apiResponse = await axios.get(`https://api.boss707.com/jsGamesv2/historyAdmin`);

      if (apiResponse.data.success && Array.isArray(apiResponse.data.data)) {
        liveTransactions = apiResponse.data.data.map(transaction => ({
          id: transaction.id,
          transaction_id: transaction.external_transaction_id,
          round_id: transaction.id,
          user_id: transaction.client_user_id,
          session: transaction.serial_number || null,
          transaction_type: transaction.transaction_type.toUpperCase(),
          amount: parseFloat(transaction.amount),
          currency_code: transaction.currency,
          transaction_timestamp: new Date(transaction.timestamp),
          reason: transaction.transaction_type.toUpperCase(),
          round_finished: true,
          transaction_status: transaction.transaction_status.toUpperCase(),
          user_name: transaction.username || null, // Use username from API response
          game_title: transaction.game_name,
          game_vendor: 'live',
          source: 'live_games'
        }));
      }
    } catch (apiError) {
      console.error('Error fetching live game history:', apiError.message);
      // Continue with bets data even if API fails
    }

    // Query for bets table with JOIN to users table
    let query = `
      SELECT 
        b.gid as id,
        b.gid::text as transaction_id,
        b.gid as round_id,
        b.uid as user_id,
        NULL::text as session,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'BET' 
        END as transaction_type,
        b.amount,
        b.coin as currency_code,
        b.created as transaction_timestamp,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'DRAW' 
        END as reason,
        TRUE as round_finished,
        CASE WHEN b.hash IS NULL THEN 'ERROR' ELSE 'SUCCESS' END as transaction_status,
        u.name as user_name, -- Fetch username from users table
        b.game as game_title,
        CASE 
          WHEN b.slot IS NOT NULL THEN b.slot 
          ELSE 'internal' 
        END as game_vendor,
        'bets' as source
      FROM bets b
      LEFT JOIN users u ON b.uid = u.id -- Join with users table
      WHERE 1=1
      ${search ? `AND (u.name ILIKE $1 OR b.gid::text ILIKE $1)` : ''}
      ${userId ? `AND b.uid = ${search ? '$2' : '$1'}` : ''}
    `;

    let countQuery = `
      SELECT COUNT(*) 
      FROM bets b
      LEFT JOIN users u ON b.uid = u.id
      WHERE 1=1
      ${search ? `AND (u.name ILIKE $1 OR b.gid::text ILIKE $1)` : ''}
      ${userId ? `AND b.uid = ${search ? '$2' : '$1'}` : ''}
    `;

    let params = [];
    if (search) {
      params.push(`%${search}%`);
    }
    if (userId) {
      params.push(userId);
    }

    query += ` ORDER BY transaction_timestamp DESC`;
    const betsParams = [...params];

    const [betsResult, countResult] = await Promise.all([
      pg.query(query, betsParams),
      pg.query(countQuery, betsParams)
    ]);

    // Combine bets and live transactions
    const betsTransactions = betsResult.rows;
    let allTransactions = [...betsTransactions, ...liveTransactions];

    // Apply filters
    if (search) {
      const searchLower = search.toLowerCase();
      allTransactions = allTransactions.filter(t =>
        (t.user_name && t.user_name.toLowerCase().includes(searchLower)) ||
        (t.transaction_id && t.transaction_id.toLowerCase().includes(searchLower)) ||
        (t.session && t.session.toLowerCase().includes(searchLower))
      );
    }

    if (userId) {
      allTransactions = allTransactions.filter(t => t.user_id.toString() === userId);
    }

    // Sort by transaction_timestamp (latest first)
    allTransactions.sort((a, b) => new Date(b.transaction_timestamp) - new Date(a.transaction_timestamp));

    // Apply pagination
    const totalCount = allTransactions.length;
    const totalPages = Math.ceil(totalCount / Number(limit));
    const paginatedTransactions = allTransactions.slice(offset, offset + Number(limit));

    res.status(200).json({
      transactions: paginatedTransactions,
      pagination: {
        totalCount,
        totalPages,
        currentPage: Number(page),
        limit: Number(limit)
      }
    });

  } catch (error) {
    console.error('Error fetching transaction history:', error);
    res.status(500).json({ error: 'Failed to fetch transaction history' });
  }
};

const getUserTransactionHistory = async (req, res) => {
  try {
    const userId = req.params.userId;
    const {
      page = 1,
      limit = 10,
      search = '',
      game_title = '',
      game_vendor = '',
      date_from = '',
      date_to = '',
      transaction_status = '',
      transaction_type = ''
    } = req.query;

    if (!userId) {
      return res.status(400).json({ error: 'User ID is required' });
    }

    const offset = (Number(page) - 1) * Number(limit);

    // Fetch live game history (unchanged)
    let liveTransactions = [];
    try {
      const apiResponse = await axios.get(`https://api.boss707.com/jsGamesv2/history?userid=${userId}`);
      if (apiResponse.data.success && Array.isArray(apiResponse.data.data)) {
        liveTransactions = apiResponse.data.data.map(transaction => ({
          id: transaction.id,
          transaction_id: transaction.external_transaction_id,
          round_id: transaction.id,
          user_id: transaction.client_user_id,
          session: transaction.serial_number || null,
          transaction_type: transaction.transaction_type.toUpperCase(),
          amount: parseFloat(transaction.amount),
          currency_code: transaction.currency,
          transaction_timestamp: new Date(transaction.timestamp),
          reason: transaction.transaction_type.toUpperCase(),
          round_finished: true,
          transaction_status: transaction.transaction_status.toUpperCase(),
          user_name: null,
          game_title: transaction.game_name,
          game_vendor: 'live',
          source: 'live_games'
        }));
      }
    } catch (apiError) {
      console.error('Error fetching live game history:', apiError.message);
    }

    // Build dynamic query conditions
    let conditions = ['b.uid = $1'];
    let params = [userId];
    let paramIndex = 2;

    if (search) {
      conditions.push(`(b.name ILIKE $${paramIndex} OR b.gid::text ILIKE $${paramIndex} OR b.game ILIKE $${paramIndex})`);
      params.push(`%${search}%`);
      paramIndex++;
    }

    if (game_title) {
      conditions.push(`b.game ILIKE $${paramIndex}`);
      params.push(`%${game_title}%`);
      paramIndex++;
    }

    if (game_vendor) {
      conditions.push(`(b.slot ILIKE $${paramIndex} OR $${paramIndex} = 'internal')`);
      params.push(game_vendor);
      paramIndex++;
    }

    if (date_from) {
      conditions.push(`b.created >= $${paramIndex}`);
      params.push(date_from);
      paramIndex++;
    }

    if (date_to) {
      conditions.push(`b.created <= $${paramIndex}`);
      params.push(date_to);
      paramIndex++;
    }

    if (transaction_status) {
      conditions.push(`(CASE WHEN b.hash IS NULL THEN 'ERROR' ELSE 'SUCCESS' END) = $${paramIndex}`);
      params.push(transaction_status.toUpperCase());
      paramIndex++;
    }

    if (transaction_type) {
      conditions.push(`(CASE 
        WHEN b.profit > 0 THEN 'WIN'
        WHEN b.profit < 0 THEN 'LOSS'
        ELSE 'BET' 
      END) = $${paramIndex}`);
      params.push(transaction_type.toUpperCase());
      paramIndex++;
    }

    const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';

    // Query for bets table
    let query = `
      SELECT 
        b.gid as id,
        b.gid::text as transaction_id,
        b.gid as round_id,
        b.uid as user_id,
        NULL::text as session,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'BET' 
        END as transaction_type,
        b.amount,
        b.coin as currency_code,
        b.created as transaction_timestamp,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'DRAW' 
        END as reason,
        TRUE as round_finished,
        CASE WHEN b.hash IS NULL THEN 'ERROR' ELSE 'SUCCESS' END as transaction_status,
        b.name as user_name,
        b.game as game_title,
        CASE 
          WHEN b.slot IS NOT NULL THEN b.slot 
          ELSE 'internal' 
        END as game_vendor,
        'bets' as source
      FROM bets b
      ${whereClause}
    `;

    let countQuery = `
      SELECT COUNT(*) 
      FROM bets b
      ${whereClause}
    `;

    const transactionsResult = await pg.query(query, params);
    const countResult = await pg.query(countQuery, params);

    // Combine bets and live transactions
    let allTransactions = [...transactionsResult.rows, ...liveTransactions];

    // Apply filters to live transactions
    if (search || game_title || game_vendor || date_from || date_to || transaction_status || transaction_type) {
      allTransactions = allTransactions.filter(t => {
        let matches = true;
        const searchLower = search.toLowerCase();
        if (search) {
          matches = matches && (
            (t.user_name && t.user_name.toLowerCase().includes(searchLower)) ||
            (t.transaction_id && t.transaction_id.toLowerCase().includes(searchLower)) ||
            (t.game_title && t.game_title.toLowerCase().includes(searchLower))
          );
        }
        if (game_title) matches = matches && t.game_title.toLowerCase().includes(game_title.toLowerCase());
        if (game_vendor) matches = matches && t.game_vendor.toLowerCase() === game_vendor.toLowerCase();
        if (date_from) matches = matches && new Date(t.transaction_timestamp) >= new Date(date_from);
        if (date_to) matches = matches && new Date(t.transaction_timestamp) <= new Date(date_to);
        if (transaction_status) matches = matches && t.transaction_status === transaction_status.toUpperCase();
        if (transaction_type) matches = matches && t.transaction_type === transaction_type.toUpperCase();
        return matches;
      });
    }

    // Sort by transaction_timestamp (latest first)
    allTransactions.sort((a, b) => new Date(b.transaction_timestamp) - new Date(a.transaction_timestamp));

    // Apply pagination
    const totalCount = allTransactions.length;
    const totalPages = Math.ceil(totalCount / Number(limit));
    const paginatedTransactions = allTransactions.slice(offset, offset + Number(limit));

    res.status(200).json({
      transactions: paginatedTransactions,
      pagination: {
        totalCount,
        totalPages,
        currentPage: Number(page),
        limit: Number(limit)
      }
    });

  } catch (error) {
    console.error('Error fetching user transaction history:', error);
    res.status(500).json({ error: 'Failed to fetch user transaction history' });
  }
};


const getLuckySportsTransactionHistory = async (req, res) => {
  try {
    const { page = 1, limit = 10, search = '', userId = null } = req.query;
    const offset = (Number(page) - 1) * Number(limit);

    // Fetch live game history from the API
    let liveTransactions = [];
    try {
      const apiResponse = await axios.get(`https://api.boss707.com/jsGamesv2/historyAdmin`);

      if (apiResponse.data.success && Array.isArray(apiResponse.data.data)) {
        liveTransactions = apiResponse.data.data
          .filter(transaction => transaction.game_name === "LuckySports") // Filter for LuckySports only
          .map(transaction => ({
            id: transaction.id,
            transaction_id: transaction.external_transaction_id,
            round_id: transaction.id,
            user_id: transaction.client_user_id,
            session: transaction.serial_number || null,
            transaction_type: transaction.transaction_type.toUpperCase(),
            amount: parseFloat(transaction.amount),
            currency_code: transaction.currency,
            transaction_timestamp: new Date(transaction.timestamp),
            reason: transaction.transaction_type.toUpperCase(),
            round_finished: true,
            transaction_status: transaction.transaction_status.toUpperCase(),
            user_name: transaction.username || null,
            game_title: transaction.game_name,
            game_vendor: 'live',
            source: 'live_games'
          }));
      }
    } catch (apiError) {
      console.error('Error fetching live game history:', apiError.message);
      // Continue with bets data even if API fails
    }

    // Query for bets table with JOIN to users table - filtered for LuckySports
    let query = `
      SELECT 
        b.gid as id,
        b.gid::text as transaction_id,
        b.gid as round_id,
        b.uid as user_id,
        NULL::text as session,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'BET' 
        END as transaction_type,
        b.amount,
        b.coin as currency_code,
        b.created as transaction_timestamp,
        CASE 
          WHEN b.profit > 0 THEN 'WIN'
          WHEN b.profit < 0 THEN 'LOSS'
          ELSE 'DRAW' 
        END as reason,
        TRUE as round_finished,
        CASE WHEN b.hash IS NULL THEN 'ERROR' ELSE 'SUCCESS' END as transaction_status,
        u.name as user_name,
        b.game as game_title,
        CASE 
          WHEN b.slot IS NOT NULL THEN b.slot 
          ELSE 'internal' 
        END as game_vendor,
        'bets' as source
      FROM bets b
      LEFT JOIN users u ON b.uid = u.id
      WHERE b.game = 'LuckySports'
      ${search ? `AND (u.name ILIKE $1 OR b.gid::text ILIKE $1)` : ''}
      ${userId ? `AND b.uid = ${search ? '$2' : '$1'}` : ''}
    `;

    let countQuery = `
      SELECT COUNT(*) 
      FROM bets b
      LEFT JOIN users u ON b.uid = u.id
      WHERE b.game = 'LuckySports'
      ${search ? `AND (u.name ILIKE $1 OR b.gid::text ILIKE $1)` : ''}
      ${userId ? `AND b.uid = ${search ? '$2' : '$1'}` : ''}
    `;

    let params = [];
    if (search) {
      params.push(`%${search}%`);
    }
    if (userId) {
      params.push(userId);
    }

    query += ` ORDER BY transaction_timestamp DESC`;
    const betsParams = [...params];

    const [betsResult, countResult] = await Promise.all([
      pg.query(query, betsParams),
      pg.query(countQuery, betsParams)
    ]);

    // Combine bets and live transactions
    const betsTransactions = betsResult.rows;
    let allTransactions = [...betsTransactions, ...liveTransactions];

    // Apply filters
    if (search) {
      const searchLower = search.toLowerCase();
      allTransactions = allTransactions.filter(t =>
        (t.user_name && t.user_name.toLowerCase().includes(searchLower)) ||
        (t.transaction_id && t.transaction_id.toLowerCase().includes(searchLower)) ||
        (t.session && t.session.toLowerCase().includes(searchLower))
      );
    }

    if (userId) {
      allTransactions = allTransactions.filter(t => t.user_id.toString() === userId);
    }

    // Sort by transaction_timestamp (latest first)
    allTransactions.sort((a, b) => new Date(b.transaction_timestamp) - new Date(a.transaction_timestamp));

    // Apply pagination
    const totalCount = allTransactions.length;
    const totalPages = Math.ceil(totalCount / Number(limit));
    const paginatedTransactions = allTransactions.slice(offset, offset + Number(limit));

    res.status(200).json({
      transactions: paginatedTransactions,
      pagination: {
        totalCount,
        totalPages,
        currentPage: Number(page),
        limit: Number(limit)
      }
    });

  } catch (error) {
    console.error('Error fetching LuckySports transaction history:', error);
    res.status(500).json({ error: 'Failed to fetch LuckySports transaction history' });
  }
};


// const getUserTransactionHistory = async (req, res) => {
//   try {
//     const userId = req.params.userId;
//     const { page = 1, limit = 10, search = '' } = req.query;
    
//     if (!userId) {
//       return res.status(400).json({ error: 'User ID is required' });
//     }
    
//     const offset = (Number(page) - 1) * Number(limit);
    
//     // Fetch live game history from the API
//     let liveTransactions = [];
//     try {
//       const apiResponse = await axios.get(`https://api.boss707.com/jsGamesv2/history?userid=${userId}`);
//       if (apiResponse.data.success && Array.isArray(apiResponse.data.data)) {
//         liveTransactions = apiResponse.data.data.map(transaction => ({
//           id: transaction.id,
//           transaction_id: transaction.external_transaction_id,
//           round_id: transaction.id,
//           user_id: transaction.client_user_id,
//           session: transaction.serial_number || null,
//           transaction_type: transaction.transaction_type.toUpperCase(),
//           amount: parseFloat(transaction.amount),
//           currency_code: transaction.currency,
//           transaction_timestamp: new Date(transaction.timestamp),
//           reason: transaction.transaction_type.toUpperCase(),
//           round_finished: true,
//           transaction_status: transaction.transaction_status.toUpperCase(),
//           user_name: null, // API doesn't provide user name
//           game_title: transaction.game_name,
//           game_vendor: 'live', // Assuming live games have a specific vendor
//           source: 'live_games'
//         }));
//       }
//     } catch (apiError) {
//       console.error('Error fetching live game history:', apiError.message);
//       // Continue with bets data even if API fails
//     }
    
//     // Query for bets table
//     let query = `
//       SELECT 
//         b.gid as id,
//         b.gid::text as transaction_id,
//         b.gid as round_id,
//         b.uid as user_id,
//         NULL::text as session,
//         CASE 
//           WHEN b.profit > 0 THEN 'WIN'
//           WHEN b.profit < 0 THEN 'LOSS'
//           ELSE 'BET' 
//         END as transaction_type,
//         b.amount,
//         b.coin as currency_code,
//         b.created as transaction_timestamp,
//         CASE 
//           WHEN b.profit > 0 THEN 'WIN'
//           WHEN b.profit < 0 THEN 'LOSS'
//           ELSE 'DRAW' 
//         END as reason,
//         TRUE as round_finished,
//         CASE WHEN b.hash IS NULL THEN 'ERROR' ELSE 'SUCCESS' END as transaction_status,
//         b.name as user_name,
//         b.game as game_title,
//         CASE 
//           WHEN b.slot IS NOT NULL THEN b.slot 
//           ELSE 'internal' 
//         END as game_vendor,
//         'bets' as source
//       FROM bets b
//       WHERE b.uid = $1
//       ${search ? `AND (b.name ILIKE $2 OR b.gid::text ILIKE $2)` : ''}
//     `;
    
//     let countQuery = `
//       SELECT COUNT(*) 
//       FROM bets b
//       WHERE b.uid = $1
//       ${search ? `AND (b.name ILIKE $2 OR b.gid::text ILIKE $2)` : ''}
//     `;
    
//     let params = [userId];
//     if (search) {
//       params.push(`%${search}%`);
//     }
    
//     const transactionsResult = await pg.query(query, params);
    
//     const countParams = [userId];
//     if (search) {
//       countParams.push(`%${search}%`);
//     }
    
//     const countResult = await pg.query(countQuery, countParams);
    
//     // Combine bets and live transactions
//     const betsTransactions = transactionsResult.rows;
//     let allTransactions = [...betsTransactions, ...liveTransactions];
    
//     // Apply search filter to live transactions
//     if (search) {
//       const searchLower = search.toLowerCase();
//       allTransactions = allTransactions.filter(t => 
//         (t.user_name && t.user_name.toLowerCase().includes(searchLower)) ||
//         (t.transaction_id && t.transaction_id.toLowerCase().includes(searchLower)) ||
//         (t.session && t.session.toLowerCase().includes(searchLower))
//       );
//     }
    
//     // Sort by transaction_timestamp (latest first)
//     allTransactions.sort((a, b) => new Date(b.transaction_timestamp) - new Date(a.transaction_timestamp));
    
//     // Apply pagination
//     const totalCount = allTransactions.length;
//     const totalPages = Math.ceil(totalCount / Number(limit));
//     const paginatedTransactions = allTransactions.slice(offset, offset + Number(limit));
    
//     res.status(200).json({
//       transactions: paginatedTransactions,
//       pagination: {
//         totalCount,
//         totalPages,
//         currentPage: Number(page),
//         limit: Number(limit)
//       }
//     });
    
//   } catch (error) {
//     console.error('Error fetching user transaction history:', error);
//     res.status(500).json({ error: 'Failed to fetch user transaction history' });
//   }
// };

// Get transaction summary stats
const getTransactionStats = async (req, res) => {
  try {
    const { userId = null } = req.query;
    
    let query = `
      SELECT 
        (
          (SELECT COUNT(*) FROM js_game_transactions ${userId ? 'WHERE user_id = $1' : ''}) +
          (SELECT COUNT(*) FROM bets ${userId ? 'WHERE uid = $1' : ''})
        ) as total_transactions,
        (
          (SELECT COUNT(*) FROM js_game_transactions WHERE amount > 0 ${userId ? 'AND user_id = $1' : ''}) +
          (SELECT COUNT(*) FROM bets WHERE profit > 0 ${userId ? 'AND uid = $1' : ''})
        ) as winning_transactions,
        (
          (SELECT COUNT(*) FROM js_game_transactions WHERE amount < 0 ${userId ? 'AND user_id = $1' : ''}) +
          (SELECT COUNT(*) FROM bets WHERE profit < 0 ${userId ? 'AND uid = $1' : ''})
        ) as losing_transactions,
        (
          (SELECT COALESCE(SUM(amount), 0) FROM js_game_transactions ${userId ? 'WHERE user_id = $1' : ''}) +
          (SELECT COALESCE(SUM(profit), 0) FROM bets ${userId ? 'WHERE uid = $1' : ''})
        ) as net_amount
    `;
    
    let params = [];
    if (userId) {
      params.push(userId);
    }
    
    const result = await pg.query(query, params);
    res.status(200).json(result.rows[0]);
    
  } catch (error) {
    console.error('Error fetching transaction stats:', error);
    res.status(500).json({ error: 'Failed to fetch transaction stats' });
  }
};
const getUserBetWinCount=async (req, res) => {
  const userId = req.params.userId;

  if (!userId || isNaN(userId)) {
    return res.status(400).json({ error: 'Invalid userId' });
  }

  try {
    // Query bets table for bet and win counts
    const betsQuery = `
      SELECT 
        COUNT(*) as bet_count, 
        COUNT(CASE WHEN profit > 0 THEN 1 END) as win_count 
      FROM bets 
      WHERE uid = $1
    `;
    const betsResult = await pg.query(betsQuery, [userId]);

    let betsBetCount = 0;
    let betsWinCount = 0;

    if (betsResult.rows && betsResult.rows.length > 0) {
      betsBetCount = parseInt(betsResult.rows[0].bet_count) || 0;
      betsWinCount = parseInt(betsResult.rows[0].win_count) || 0;
    }

    // Query game_transactions table for bet and win counts
    const txQuery = `
      SELECT 
        transaction_type, 
        COUNT(*) as count 
      FROM game_transactions 
      WHERE user_id = $1 AND transaction_type IN ('bet', 'win') 
      GROUP BY transaction_type
    `;
    const txResult = await pg.query(txQuery, [userId]);

    let txBetCount = 0;
    let txWinCount = 0;

    if (txResult.rows && txResult.rows.length > 0) {
      txResult.rows.forEach(row => {
        const count = parseInt(row.count) || 0;
        if (row.transaction_type === 'bet') {
          txBetCount += count;
        } else if (row.transaction_type === 'win') {
          txWinCount += count;
        }
      });
    }

    // Combine counts from both tables
    const result = {
      betCount: betsBetCount + txBetCount,
      winCount: betsWinCount + txWinCount
    };

    console.log(`User ${userId} - Bet Count: ${result.betCount}, Win Count: ${result.winCount} (Bets table: ${betsBetCount}, ${betsWinCount}; Transactions table: ${txBetCount}, ${txWinCount})`);

    res.status(200).json(result);
  } catch (error) {
    console.error('Error fetching bet and win counts:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
}
module.exports = {
  getTransactionHistory,
  getUserTransactionHistory,
  getTransactionStats,
  getUserBetWinCount,
  getLuckySportsTransactionHistory 
};