/* ------------------------------------------------------------------------
   bonusCron.js  –  unified bonus engine
   • Daily bonus   – issued every calendar day     (IST)
   • Weekly bonus  – issued only on Sunday       (IST)
   • Monthly bonus – issued only on the last day of the month (IST)
   A bonus is claimable until 23 : 59 : 59.999 IST of its payout day.
   When the window closes it is flagged is_unclaimable = TRUE and the
   matching column (dailybonus / weeklybonus / monthlybonus) in userbonus
   is zeroed, so the UI never shows duplicate claim buttons.
   ------------------------------------------------------------------------ */

const cron                   = require('node-cron');
const pg                     = require('../General/Model');
const { calculateWagerBonus } = require('./calculatewagerbonus');
const logger                 = require('./logger');          // adjust path

/* ───────────────────────────── helpers ──────────────────────────────── */

/** Mon-based week bounds that contain `now` (both UTC dates) */
function getWeekBounds(now = new Date()) {
  const start = new Date(now);
  const end   = new Date(now);
  const day   = now.getDay();                               // 0 = Sun … 6

  start.setDate(start.getDate() - (day === 0 ? 6 : day - 1));
  start.setHours(0, 0, 0, 0);

  end.setDate(start.getDate() + 6);
  end.setHours(23, 59, 59, 999);

  return { start, end };
}

/** 23:59:59.999 IST for the calendar day of `date` */
function istEndOfDay(date = new Date()) {
  // IST = UTC+05:30  →  23:59:59 IST == 18:29:59 UTC
  return new Date(Date.UTC(
    date.getUTCFullYear(),
    date.getUTCMonth(),
    date.getUTCDate(),
    18, 29, 59, 999
  ));
}
/* ───────────────────────  retro-expiry helper  ──────────────────────
   Sometimes older rows were created with a 24-hour claim window that
   stretches into the *next* payout day.  If the “period day” (IST)
   in the row’s created_at is earlier than TODAY’s period day we treat
   it as expired right now.  */
function forceExpireIfPastPeriod(row, now = new Date()) {
  const istCreated = new Date(row.created_at.getTime() + 5.5 * 60 * 60 * 1000);
  const istNow     = new Date(now.getTime()       + 5.5 * 60 * 60 * 1000);

  if (row.bonus_type === 'daily') {
    return istCreated.getDate() !== istNow.getDate();
  }
  if (row.bonus_type === 'weekly') {
    // week index 0-53 (Mon-based) – force expire if different week nr
    const week = d => {
      const t = new Date(d); t.setHours(0,0,0,0);
      t.setDate(t.getDate() - ((t.getDay() + 6) % 7));   // back to Monday
      return +t;                                         // epoch Monday
    };
    return week(istCreated) !== week(istNow);
  }
    if (row.bonus_type === 'monthly') {
   const lastDayOfCreatedMonth =
        new Date(istCreated.getFullYear(),
                 istCreated.getMonth() + 1, 0).getDate();     // 30 for June
    /*  expire if the row wasn’t created on that month-end day OR
        if it belongs to a previous month altogether                 */
    return (istCreated.getDate() !== lastDayOfCreatedMonth) ||
               (istCreated.getMonth() !== istNow.getMonth())     ||
         (istCreated.getFullYear() !== istNow.getFullYear());
  }
  return false;
}

/** true if `now` is the payout day for the period (IST calendar) */
function isPayoutDay(period, now = new Date()) {
  const istNow = new Date(now.getTime() + 5.5 * 60 * 60 * 1000); // shift to IST
  if (period === 'daily')   return true;
  if (period === 'weekly')  return istNow.getDay() === 0;                       // Sun
  if (period === 'monthly') return istNow.getDate() ===
                               new Date(istNow.getFullYear(),
                                        istNow.getMonth() + 1,
                                        0).getDate();                           // last day
  return false;
}

/** zero-out the period column in userbonus */
async function clearUserBonusCol(userid, period) {
  await pg.query(
    `UPDATE userbonus SET ${period}bonus = 0 WHERE userid = $1`,
    [userid]
  );
}

/* ───────────────────────────  utilities for UI timers  ──────────────── */

function calculateTimeRemaining(deadline) {
  const remainingMs = deadline.getTime() - Date.now();
  if (remainingMs <= 0)
    return { expired: true, hours: 0, minutes: 0, seconds: 0, totalSeconds: 0 };

  const totalSeconds = Math.floor(remainingMs / 1000);
  return {
    expired     : false,
    hours       : Math.floor(totalSeconds / 3600),
    minutes     : Math.floor((totalSeconds % 3600) / 60),
    seconds     : totalSeconds % 60,
    totalSeconds
  };
}

function getNextAvailability(period, lastReset /* Date|null */, now = new Date()) {
  const istNow = new Date(now.getTime() + 5.5 * 60 * 60 * 1000);
  const next   = new Date(istNow);

  if (period === 'daily') {
    next.setDate(next.getDate() + 1);
    next.setHours(0, 0, 0, 0);
  } else if (period === 'weekly') {               // next Monday 00:00 IST
    const d = istNow.getDay();                    // 0=Sun … 6
    next.setDate(next.getDate() + ((8 - d) % 7));
    next.setHours(0, 0, 0, 0);
  } else if (period === 'monthly') {              // 1st of next month 00:00 IST
    next.setMonth(next.getMonth() + 1, 1);
    next.setHours(0, 0, 0, 0);
  }
  return next;
}

/* ───────────────────────────  bootstrap helpers  ────────────────────── */

async function ensureUserBonusRecord(userid) {
  const { rowCount } = await pg.query(
    'SELECT 1 FROM userbonus WHERE userid = $1 LIMIT 1', [userid]
  );
  if (rowCount) return;

  await pg.query(`
    INSERT INTO userbonus (
      userid, name,
      totalbonus, vipbonus, specialbonus, generalbonus,
      dailybonus, weeklybonus, monthlybonus,
      joiningbonus, rakebonus,
      actualdailybonus, actualweeklybonus, actualmonthlybonus,
      last_daily_reset, last_weekly_reset, last_monthly_reset
    ) VALUES (
      $1, $2,
      0,0,0,0,
      0,0,0,
      0,0,
      0,0,0,
      NOW(), NOW(), NOW()
    )`, [userid, `User ${userid}`]);

  logger.info(`Created userbonus row for ${userid}`);
}

/* ───────────────────────  per-period processor  ─────────────────────── */

/* ─────────────────────  Period bonus (final)  ───────────────────── */
async function processPeriodBonus(userid, period, now = new Date()) {
  /* run only on calendar payout day (IST) */
  if (!isPayoutDay(period, now)) return false;

  /* date boundaries used by calculateWagerBonus */
  let start, end;
  if (period === 'daily') {
    start = new Date(now); start.setHours(0, 0, 0, 0);
    end   = new Date(now); end.setHours(23, 59, 59, 999);
  } else if (period === 'weekly') {
    ({ start, end } = getWeekBounds(now));
  } else { // monthly
    start = new Date(now.getUTCFullYear(), now.getUTCMonth(), 1, 0, 0, 0, 0);
    end   = new Date(now.getUTCFullYear(), now.getUTCMonth() + 1, 0,
                     23, 59, 59, 999);
  }

  const resetCol  = `last_${period}_reset`;
  const bonusCol  = `${period}bonus`;

  /* skip if we've already stamped this period for this user */
  const { rows: [ub] } = await pg.query(
    `SELECT ${resetCol} FROM userbonus WHERE userid = $1`, [userid]);
  if (ub?.[resetCol] && new Date(ub[resetCol]) >= start) return false;

  logger.info(`┌─ ${period.toUpperCase()} | user ${userid}`);

  /* retire any still-open rows and clear column */
  await pg.query(`
    UPDATE bonus_history
       SET is_unclaimable = TRUE
     WHERE userid = $1
       AND bonus_type = $2
       AND is_claimed = FALSE
       AND is_unclaimable = FALSE`,
    [userid, period]);
  await clearUserBonusCol(userid, period);

  /* eligibility calc */
  const { eligible, bonusAmount, wagerChange, reason } =
        await calculateWagerBonus(userid, period, start, end);

  let finalAmount = bonusAmount;
  if (!eligible || bonusAmount <= 0) {
    finalAmount = 0;                                     // still create row
    logger.info(`│  not eligible (${reason}) – inserting zero-amount bonus`);
  }

  /* 23:59:59.999 IST claim deadline */
  const claimDeadline = istEndOfDay(now);

  /* insert fresh row (even if 0) */
  const { rows: [inserted] } = await pg.query(`
    INSERT INTO bonus_history (
      userid, bonus_type, wager_change, bonus_amount,
      claim_deadline, is_claimed, is_unclaimable, created_at
    ) VALUES (
      $1, $2, $3, $4,
      $5, FALSE, FALSE, NOW()
    ) RETURNING id`,
    [userid, period, wagerChange, finalAmount, claimDeadline]);

  /* stamp userbonus */
  await pg.query(`
    UPDATE userbonus
       SET ${bonusCol} = $1,
           ${resetCol} = NOW()
     WHERE userid   = $2`,
    [finalAmount, userid]);

  logger.info(`│  ✓ new ${period} bonus #${inserted.id} (${finalAmount})`);
  logger.info(`│    ↳ ${bonusCol} in userbonus set to ${finalAmount}`);
  return true;                                            // row always created
}


/* ───────────────────  full calculation loop (one cron tick) ─────────── */

/* ─────────────────  Full bonus calculation loop  ───────────────── */
async function processBonusCalculation({ now = new Date() } = {}) {
  logger.info('==== bonus run start ====');

  /* 1) expire rows whose deadline passed, and clear their UB column */
 /* 1) expire rows: (a) natural deadline passed   OR
                     (b) row belongs to a *past* period (old 24 h bug) */
const { rows: openRows } = await pg.query(`
  SELECT id, userid, bonus_type, claim_deadline, created_at
    FROM bonus_history
   WHERE is_claimed     = FALSE
     AND is_unclaimable = FALSE`);

const toExpire = openRows.filter(r =>
      r.claim_deadline < now || forceExpireIfPastPeriod(r, now));

if (toExpire.length) {
  const ids = toExpire.map(r => r.id);
  await pg.query(`
    UPDATE bonus_history
       SET is_unclaimable = TRUE
     WHERE id = ANY($1::bigint[])`, [ids]);
  logger.info(`expired → ${toExpire.length} rows [${ids.join(',')}]`);
  for (const { userid, bonus_type } of toExpire) {
    await clearUserBonusCol(userid, bonus_type);
    logger.info(`│  column reset ${bonus_type}bonus → 0  for user ${userid}`);
  }
} else {
  logger.info('expired → 0');
}

  /* 2) active wager users */
  const { rows: users } = await pg.query(`
    SELECT DISTINCT uid
      FROM userwager
     WHERE COALESCE(
             NULLIF(REPLACE(wager, ',', ''), ''), '0')::NUMERIC > 0`);

  let created = 0, errors = 0;

  for (const { uid: userid } of users) {
    try {
      await pg.query('BEGIN');
      await ensureUserBonusRecord(userid);

      for (const p of ['daily', 'weekly', 'monthly'])
        if (await processPeriodBonus(userid, p, now)) created++;

      await pg.query('COMMIT');
    } catch (err) {
      await pg.query('ROLLBACK');
      errors++;
      logger.error(`User ${userid} failed: ${err.message}`, { stack: err.stack });
    }
  }

  logger.info(`run completed – bonuses created ${created}  errors ${errors}`);
  return { users: users.length, created, errors };
}


/* ───────────────────  live countdown emitter (socket.io) ────────────── */

async function getUserBonusTimers(userid) {
  // active claimables
  const { rows: active } = await pg.query(`
    SELECT id, bonus_type, bonus_amount, wager_change,
           claim_deadline, created_at
      FROM bonus_history
     WHERE userid         = $1
       AND is_claimed     = FALSE
       AND is_unclaimable = FALSE
       AND claim_deadline >= NOW()
     ORDER BY created_at DESC`, [userid]);

  // last reset stamps
  const { rows: [ub] } = await pg.query(`
    SELECT last_daily_reset, last_weekly_reset, last_monthly_reset
      FROM userbonus WHERE userid = $1`, [userid]);

  const now = new Date();
  const timers = active.map(b => ({
    id            : b.id,
    type          : b.bonus_type,
    amount        : Number(b.bonus_amount),
    wagerChange   : Number(b.wager_change),
    claimDeadline : b.claim_deadline,
    timeRemaining : calculateTimeRemaining(new Date(b.claim_deadline)),
    createdAt     : b.created_at
  }));

  return {
    userid,
    activeBonuses       : timers,
    nextBonusAvailable  : {
      daily  : getNextAvailability('daily',   ub?.last_daily_reset,   now),
      weekly : getNextAvailability('weekly',  ub?.last_weekly_reset,  now),
      monthly: getNextAvailability('monthly', ub?.last_monthly_reset, now)
    },
    currentTime         : now
  };
}

function startBonusTimerUpdates(io) {
  setInterval(async () => {
    for (const socket of io.sockets.sockets.values()) {
      if (!socket.userid || !socket.connected) continue;
      try {
        socket.emit('bonusTimerUpdate', await getUserBonusTimers(socket.userid));
      } catch (err) {
        logger.error(`Timer err user ${socket.userid}: ${err.message}`);
      }
    }
  }, 1000);
}

/* ───────────────────────  CRON registration  ────────────────────────── */

function initBonusCalculationCron(io) {
  /*  **TESTING** → every minute: '* * * * *'
      **PRODUCTION** → midnight IST: '0 0 * * *'                           */
  const job = cron.schedule(
    '0 0 * * *',                                // adjust as needed
    () => processBonusCalculation()
            .catch(e => logger.error('cron run failed', e)),
    { scheduled: true, timezone: 'Asia/Kolkata' }
  );

  logger.info('cron → registered');
  startBonusTimerUpdates(io);
  return job;
}

/* ────────────────────────────  exports  ─────────────────────────────── */

module.exports = {
  processBonusCalculation,
  getUserBonusTimers,
  initBonusCalculationCron,
  debugBonusCalculation  : (userid, period = 'daily', when = new Date()) =>
                              processPeriodBonus(userid, period, when),
  manualTriggerBonusCalculation : opts => processBonusCalculation(opts)
};
