const pg = require('../General/Model');   // pooled node-postgres instance

/* ---------- admin: read / update multipliers ---------- */
exports.getConfig = async (_req, res, next) => {
  try {
    const { rows } = await pg.query(`
    SELECT 
        bonus_type,
        deposit_multiplier,
        wager_multiplier,
        min_deposit,
        updated_at
    FROM bonus_config 
    ORDER BY bonus_type
`);
    res.json(rows);
  } catch (e) { next(e); }
};

exports.upsertConfig = async (req, res, next) => {
  const { bonus_type, deposit_multiplier, wager_multiplier, min_deposit } = req.body;
  try {
    const { rows:[row] } = await pg.query(
      `INSERT INTO bonus_config (bonus_type, deposit_multiplier, wager_multiplier, min_deposit)
       VALUES ($1,$2,$3,$4)
       ON CONFLICT (bonus_type)
         DO UPDATE SET deposit_multiplier=$2,
                       wager_multiplier=$3,
                       min_deposit=$4,
                       updated_at = now()
       RETURNING *`,
      [bonus_type, deposit_multiplier, wager_multiplier, min_deposit]
    );
    res.json(row);
  } catch (e) { next(e); }
};

/* ---------- helper: extract UID safely ---------- */
const getUid = req =>
        Number(req.body.uid || req.params.uid || req.query.uid);

/* ---------- flag: has the user chosen a bonus? ---------- */
exports.selectionFlag = async (req, res, next) => {
  const uid = getUid(req);
  try {
    const { rows:[row] } = await pg.query(
      'SELECT selected,bonus_type FROM user_bonus WHERE uid=$1', [uid]);
    res.json({ selected: !!row?.selected ,bonus_type: row?.bonus_type || null});
  } catch (e) { next(e); }
};

/* ---------- player: progress ---------- */
exports.progress = async (req, res, next) => {
  const uid = getUid(req);
  try {
    const { rows:[row] } = await pg.query(
      `SELECT *,
              ROUND(wager_progress / NULLIF(wager_required,0) * 100,2)::numeric
                AS progress_pct
         FROM user_bonus
        WHERE uid=$1`,
      [uid]
    );
    res.json(row);
  } catch (e) { next(e); }
};

/* ---------- player: claim ---------- */
/* controllers/bonusController.js — replace exports.claim */
exports.claim = async (req, res, next) => {
  const uid = getUid(req);

  try {
    await pg.query('BEGIN');

    /* lock the bonus row */
const { rows:[ub] } = await pg.query(`
    SELECT 
        uid,
        bonus_type,
        first_deposit,
        bonus_amount,
        wager_required,
        wager_progress,
        status,
        created_at,
        updated_at,
        claimed_at,
        selected
    FROM user_bonus 
    WHERE uid=$1 
    FOR UPDATE
`, [uid]);

    if (!ub || ub.status !== 'completed')
      return res.status(400).json({ error: 'Bonus not claimable yet' });

    /* 1 USD = 1 BJB → add bonus_amount to credits.bjb */
    await pg.query(
      `INSERT INTO credits (uid, bjb)
         VALUES ($1,$2)
         ON CONFLICT (uid)
         DO UPDATE SET bjb = credits.bjb + EXCLUDED.bjb`,
      [uid, ub.bonus_amount]);


    /* mark claimed */
    await pg.query(
      `UPDATE user_bonus
          SET status='claimed',
              claimed_at = now()
        WHERE uid=$1`,
      [uid]);

    await pg.query('COMMIT');
    res.json({ bonus_amount: ub.bonus_amount });
  } catch (err) {
    await pg.query('ROLLBACK');
    next(err);
  } finally {
    
  }
};


/* ---------- player: choose bonus ---------- */
exports.selectBonus = async (req, res, next) => {
  const { bonus_type, uid } = req.body;          // uid + bonus_type in body
  if (!['sports','casino'].includes(bonus_type))
    return res.status(400).json({ error: 'Invalid bonus type' });


  try {
    await pg.query('BEGIN');

    /* 1) mark choice in users */
    await pg.query(
      'UPDATE users SET bonus_type=$1 WHERE id=$2',
      [bonus_type, uid]);

    /* 2) set flag + bonus_type in user_bonus */
    await pg.query(
      `UPDATE user_bonus
          SET bonus_type=$1,
              selected   = true
        WHERE uid=$2`,
      [bonus_type, uid]);

    await pg.query('COMMIT');
    res.json({ message: 'Bonus type selected', bonus_type });
  } catch (e) {
    await pg.query('ROLLBACK');
    next(e);
  } finally {  }
};

// controllers/bonusController.js  (append at bottom)
exports.publicConfig = async (_req, res, next) => {
  try {
    const { rows } = await pg.query(
      `SELECT bonus_type,
              deposit_multiplier,
              wager_multiplier,
              min_deposit
         FROM bonus_config
        ORDER BY bonus_type`);
    res.json(rows);                           // list of two rows: sports & casino
  } catch (e) { next(e); }
};

// update bonus config data 
exports.updateBonusConfig = async (req, res, next) => {
  try {
    const { bonus_type, deposit_multiplier, wager_multiplier, min_deposit } = req.body;

    if (!bonus_type) {
      return res.status(400).json({ error: "bonus_type is required" });
    }

    const { rowCount } = await pg.query(
      `UPDATE bonus_config
          SET deposit_multiplier = $1,
              wager_multiplier = $2,
              min_deposit = $3
        WHERE bonus_type = $4`,
      [deposit_multiplier, wager_multiplier, min_deposit, bonus_type]
    );

    if (rowCount === 0) {
      return res.status(404).json({ error: `No config found for bonus_type "${bonus_type}"` });
    }

    res.json({ message: "Bonus config updated successfully." });
  } catch (e) {
    next(e);
  }
};exports.updateBonusConfig = async (req, res, next) => {
  try {
    const { bonus_type, deposit_multiplier, wager_multiplier, min_deposit } = req.body;

    if (!bonus_type) {
      return res.status(400).json({ error: "bonus_type is required" });
    }

    const { rowCount } = await pg.query(
      `UPDATE bonus_config
          SET deposit_multiplier = $1,
              wager_multiplier = $2,
              min_deposit = $3
        WHERE bonus_type = $4`,
      [deposit_multiplier, wager_multiplier, min_deposit, bonus_type]
    );

    if (rowCount === 0) {
      return res.status(404).json({ error: `No config found for bonus_type "${bonus_type}"` });
    }

    res.json({ message: "Bonus config updated successfully." });
  } catch (e) {
    next(e);
  }
};
