const express = require('express');
const router = express.Router();
const multer = require('multer');
const ClubBannerController = require('./controller');
const fs = require('fs').promises;
const path = require('path');

// Configure multer for memory storage
const upload = multer({
    storage: multer.memoryStorage(),
    limits: {
        fileSize: 5 * 1024 * 1024, // 5MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype.startsWith('image/')) {
            cb(null, true);
        } else {
            cb(new Error('Only image files are allowed'));
        }
    }
});

// Middleware to check club ownership
const isClubOwner = async (req, res, next) => {
    try {
        const clubId = parseInt(req.params.clubId);
        const userId = parseInt(req.body.userId); // Getting userId from request body

        if (!clubId || !userId) {
            return res.status(400).json({
                success: false,
                error: 'Invalid club ID or user ID'
            });
        }

      const query = `
    SELECT 
        id,
        name,
        unique_club_id,
        owner_id,
        profile_picture,
        created_at,
        is_active,
        owner_earnings_percentage,
        agent_earnings_percentage,
        min_active_players_threshold,
        active_player_wager_threshold,
        member_earnings_percentage,
        parent_club_id,
        clubrake
    FROM clubs 
    WHERE id = $1 
    AND owner_id = $2 
    AND is_active = true
`;

        const result = await pg.query(query, [clubId, userId]);

        if (result.rows.length === 0) {
            return res.status(403).json({
                success: false,
                error: 'Unauthorized: You must be the club owner to perform this action'
            });
        }

        req.club = result.rows[0];
        next();
    } catch (error) {
        return res.status(500).json({
            success: false,
            error: 'Error verifying club ownership'
        });
    }
};

async function getImageBase64(imagePath) {
    try {
        const fullPath = path.join('/var/www/html/hellogames/ibitplay/backend/clubmembership/clubbanners/images', imagePath);
        const imageBuffer = await fs.readFile(fullPath);
        const base64Image = imageBuffer.toString('base64');
        const extension = path.extname(imagePath).substring(1);
        return `data:image/${extension};base64,${base64Image}`;
    } catch (error) {
        console.error('Error reading image:', error);
        return null;
    }
}


router.get('/clubs/banner-image/:imagePath(*)', async (req, res) => {
    try {
        const imagePath = req.params.imagePath;
        const fullPath = path.join('/var/www/html/hellogames/ibitplay/backend/clubmembership/clubbanners/images', imagePath);
        
        try {
            await fs.access(fullPath);
        } catch (error) {
            return res.status(404).json({
                success: false,
                error: 'Image not found'
            });
        }

        // Set appropriate content type based on file extension
        const ext = path.extname(imagePath).toLowerCase();
        const contentType = {
            '.jpg': 'image/jpeg',
            '.jpeg': 'image/jpeg',
            '.png': 'image/png',
            '.gif': 'image/gif'
        }[ext] || 'application/octet-stream';

        res.setHeader('Content-Type', contentType);
        const imageBuffer = await fs.readFile(fullPath);
        res.send(imageBuffer);

    } catch (error) {
        res.status(500).json({
            success: false,
            error: 'Error serving image'
        });
    }
});

// Create new banner
router.post('/clubs/:clubId/banners',
    isClubOwner,
    upload.single('image'),
    async (req, res) => {
        try {
            const { clubId } = req.params;
            const { title, userId } = req.body;
            
            if (!req.file) {
                return res.status(400).json({
                    success: false,
                    error: 'Image file is required'
                });
            }

            const result = await ClubBannerController.createBanner(
                parseInt(clubId),
                parseInt(userId),
                title,
                req.file
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(error.message.includes('Unauthorized') ? 403 : 500)
               .json({ success: false, error: error.message });
        }
    }
);

// Update banner
router.put('/clubs/:clubId/banners/:bannerId',
    isClubOwner,
    upload.single('image'),
    async (req, res) => {
        try {
            const { clubId, bannerId } = req.params;
            const { title, userId } = req.body;
            
            const updates = {
                title,
                imageFile: req.file // Optional
            };

            const result = await ClubBannerController.updateBanner(
                parseInt(bannerId),
                parseInt(clubId),
                parseInt(userId),
                updates
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(error.message.includes('unauthorized') ? 403 : 500)
               .json({ success: false, error: error.message });
        }
    }
);

// Delete banner
router.delete('/clubs/:clubId/banners/:bannerId',
    isClubOwner,
    async (req, res) => {
        try {
            const { clubId, bannerId } = req.params;
            const { userId } = req.body;
            
            const result = await ClubBannerController.deleteBanner(
                parseInt(bannerId),
                parseInt(clubId),
                parseInt(userId)
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(error.message.includes('unauthorized') ? 403 : 500)
               .json({ success: false, error: error.message });
        }
    }
);

// Get club banners
router.get('/clubs/:clubId/banners',
    async (req, res) => {
        try {
            const { clubId } = req.params;
            const { userId, includeImages } = req.body; // Add includeImages option

            const banners = await ClubBannerController.getClubBanners(
                parseInt(clubId)
            );
            
            // Process banners
            const bannersWithPaths = await Promise.all(banners.map(async banner => {
                const result = {
                    ...banner,
                    image_path: `/clubmembership/clubbanners/images/${banner.image_path}`,
                    image_url: `/api/clubs/banner-image/${banner.image_path}` // Direct image URL
                };

                // Include base64 image data if requested
                if (includeImages) {
                    result.image_base64 = await getImageBase64(banner.image_path);
                }

                return result;
            }));
            
            res.json({
                success: true,
                data: bannersWithPaths
            });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    }
);

// Send banner notification
router.post('/clubs/:clubId/banners/:bannerId/notify',
    isClubOwner,
    async (req, res) => {
        try {
            const { clubId, bannerId } = req.params;
            const { title, body, data, userId } = req.body;
            
            if (!title || !body) {
                return res.status(400).json({
                    success: false,
                    error: 'Title and body are required for notification'
                });
            }

            const result = await ClubBannerController.sendBannerNotification(
                parseInt(bannerId),
                parseInt(clubId),
                parseInt(userId),
                { title, body, data }
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(error.message.includes('unauthorized') ? 403 : 500)
               .json({ success: false, error: error.message });
        }
    }
);

// Get banner notifications for a member
router.get('/clubs/:clubId/banner-notifications',
    async (req, res) => {
        try {
            const { clubId } = req.params;
            const { userId, includeImages } = req.body;
            
            const query = `
                SELECT 
                    cbn.*,
                    cbns.is_sent,
                    cbns.is_read,
                    cbns.sent_at,
                    cb.title as banner_title,
                    cb.image_path as banner_image
                FROM club_banner_notifications cbn
                JOIN club_banner_notification_status cbns ON cbn.id = cbns.notification_id
                JOIN club_banners cb ON cbn.banner_id = cb.id
                WHERE cbn.club_id = $1 AND cbns.user_id = $2
                ORDER BY cbn.created_at DESC
            `;
            
            const result = await pg.query(query, [parseInt(clubId), parseInt(userId)]);
            
            // Process notifications with images
            const notifications = await Promise.all(result.rows.map(async notification => {
                const processedNotification = {
                    ...notification,
                    banner_image: `/clubmembership/clubbanners/images/${notification.banner_image}`,
                    banner_image_url: `/api/clubs/banner-image/${notification.banner_image}` // Direct image URL
                };

                // Include base64 image data if requested
                if (includeImages) {
                    processedNotification.banner_image_base64 = await getImageBase64(notification.banner_image);
                }

                return processedNotification;
            }));
            
            res.json({
                success: true,
                data: notifications
            });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    }
);

// Mark banner notification as read
router.put('/clubs/banner-notifications/:notificationId/read',
    async (req, res) => {
        try {
            const { notificationId } = req.params;
            const { userId } = req.body;
            
            const query = `
                UPDATE club_banner_notification_status
                SET is_read = true
                WHERE notification_id = $1 AND user_id = $2
                RETURNING *
            `;
            
            const result = await pg.query(query, [parseInt(notificationId), parseInt(userId)]);
            
            res.json({
                success: true,
                data: result.rows[0]
            });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    }
);

module.exports = router;