const admin = require('../../firabsenotifcation/config');
const pg = require('../../General/Model');

class ClubNotificationController {
    async sendClubNotification(clubId, ownerId, notification) {
        try {
            await pg.query('BEGIN');

            // Verify club ownership
           const clubQuery = `
    SELECT 
        id,
        name,
        unique_club_id,
        owner_id,
        profile_picture,
        created_at,
        is_active,
        owner_earnings_percentage,
        agent_earnings_percentage,
        min_active_players_threshold,
        active_player_wager_threshold,
        member_earnings_percentage,
        parent_club_id,
        clubrake
    FROM clubs 
    WHERE id = $1 AND owner_id = $2 AND is_active = true
`;
            const clubResult = await pg.query(clubQuery, [clubId, ownerId]);
            
            if (clubResult.rows.length === 0) {
                throw new Error('Unauthorized: Not the club owner');
            }

            // Insert notification
            const notifQuery = `
                INSERT INTO club_notifications 
                (club_id, sender_id, title, body, type, additional_data)
                VALUES ($1, $2, $3, $4, $5, $6)
                RETURNING id
            `;
            const notifResult = await pg.query(notifQuery, [
                clubId,
                ownerId,
                notification.title,
                notification.body,
                notification.type || 'club',
                notification.data || {}
            ]);
            
            const notificationId = notifResult.rows[0].id;

            // Get all club members
            const membersQuery = `
                SELECT 
                    cm.user_id,
                    uft.token,
                    uft.is_active
                FROM club_memberships cm
                LEFT JOIN user_fcm_tokens uft ON cm.user_id = uft.user_id
                WHERE cm.club_id = $1
                AND cm.role != 'owner'
            `;
            const members = await pg.query(membersQuery, [clubId]);

            // Create notification status records for all members
            const statusPromises = members.rows.map(member => {
                return pg.query(
                    `INSERT INTO club_notification_status 
                    (notification_id, user_id, is_sent, sent_at)
                    VALUES ($1, $2, $3, $4)`,
                    [
                        notificationId,
                        member.user_id,
                        member.token ? true : false,
                        member.token ? 'NOW()' : null
                    ]
                );
            });

            await Promise.all(statusPromises);

            // Send FCM notifications to members with valid tokens
            const validMembers = members.rows.filter(m => m.token && m.is_active);
            
            if (validMembers.length > 0) {
                const messages = validMembers.map(member => ({
                    token: member.token,
                    notification: {
                        title: notification.title,
                        body: notification.body
                    },
                    data: {
                        ...notification.data,
                        notificationId: notificationId.toString(),
                        type: 'club_notification',
                        clubId: clubId.toString()
                    }
                }));

                await Promise.allSettled(messages.map(msg => admin.messaging().send(msg)));
            }

            await pg.query('COMMIT');
            
            return {
                notificationId,
                totalMembers: members.rows.length,
                sentCount: validMembers.length
            };

        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        }
    }
    async getClubNotifications(clubId) {
        const query = `
            SELECT *
            FROM club_notifications
            WHERE club_id = $1
            ORDER BY created_at DESC
        `;
        
        const result = await pg.query(query, [clubId]);
        return result.rows;
    }

    async markClubNotificationAsRead(notificationId, userId) {
        const query = `
            UPDATE club_notification_status
            SET is_read = true
            WHERE notification_id = $1 AND user_id = $2
            RETURNING *
        `;
        
        const result = await pg.query(query, [notificationId, userId]);
        return result.rows[0];
    }
}

module.exports = new ClubNotificationController();