const express = require('express');
const router = express.Router();
const ClubNotificationController = require('./controller');

const isClubOwner = async (req, res, next) => {
    try {
        const clubId = parseInt(req.params.clubId);
        const userId = parseInt(req.body.userId); // Getting userId from request body

        if (!clubId || !userId) {
            return res.status(400).json({
                success: false,
                error: 'Invalid club ID or user ID'
            });
        }

      const query = `
    SELECT 
        id,
        name,
        unique_club_id,
        owner_id,
        profile_picture,
        created_at,
        is_active,
        owner_earnings_percentage,
        agent_earnings_percentage,
        min_active_players_threshold,
        active_player_wager_threshold,
        member_earnings_percentage,
        parent_club_id,
        clubrake
    FROM clubs 
    WHERE id = $1 
    AND owner_id = $2 
    AND is_active = true
`;

        const result = await pg.query(query, [clubId, userId]);

        if (result.rows.length === 0) {
            return res.status(403).json({
                success: false,
                error: 'Unauthorized: You must be the club owner to perform this action'
            });
        }

        // Store the club data for potential use in the route handler
        req.club = result.rows[0];
        next();
    } catch (error) {
        return res.status(500).json({
            success: false,
            error: 'Error verifying club ownership'
        });
    }
};

// Send notification to all club members
router.post('/clubs/:clubId/notifications', 
    async (req, res) => {
        try {
            const { clubId } = req.params;
            const { title, body, type, data ,userId} = req.body;
            
            const result = await ClubNotificationController.sendClubNotification(
                parseInt(clubId),
                userId,
                { title, body, type, data }
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(error.message.includes('Unauthorized') ? 403 : 500)
               .json({ success: false, error: error.message });
        }
    }
);

// Get club notifications for a member
router.get('/clubs/:clubId/notifications',
    async (req, res) => {
        try {
            const { clubId } = req.params;
            const notifications = await ClubNotificationController.getClubNotifications(
                parseInt(clubId)
            );
            
            res.json({
                success: true,
                data: notifications
            });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    }
);

// Mark notification as read
router.put('/clubs/notifications/:notificationId/read',
    async (req, res) => {
        try {
            const { notificationId } = req.params;
            const result = await ClubNotificationController.markClubNotificationAsRead(
                parseInt(notificationId),
                req.body.userid
            );
            
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    }
);

module.exports = router;