// src/controllers/clubController.js
const pg = require('../General/Model');
const IDGenerator = require('./idgenretor');
const path = require('path');
const fs = require('fs');
const util = require('util');
const { promisify } = require('util');
const { moveFileToClubFolder } = require('./fileupload');

const mkdir = util.promisify(fs.mkdir);
const unlink = util.promisify(fs.unlink);
const rename = util.promisify(fs.rename);
const readdir = util.promisify(fs.readdir);
const rmdir = util.promisify(fs.rmdir);

class ClubController {
  // Create Club
  async createClub(req, res) {
    const {
      userId,
      clubName,
      ownerEarningsPercentage = 10.0,
      agentEarningsPercentage = 20.0,
      memberEarningsPercentage = 5.0
    } = req.body;
    console.log('Request body:', req.body);
    const rmdir = promisify(fs.rm);
    const baseUploadFolder = '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubprofile';
    const clubFolderPath = path.join(baseUploadFolder, clubName);
    const profilePicture = req.file ? req.file.filename : null;
    const tempUploadDir = req.tempUploadDir;
  const tempFilename = req.profileFilename;
    try {
      await pg.query('BEGIN');

     
    // Fetch clubrake from siteconfig
    const siteConfigQuery = `SELECT clubrake FROM siteconfig LIMIT 1`;
    const siteConfigResult = await pg.query(siteConfigQuery);

    if (!siteConfigResult.rows.length || siteConfigResult.rows[0].clubrake == null) {
      throw new Error('Club rake value not found in siteconfig.');
    }

    const clubRake = siteConfigResult.rows[0].clubrake;

      // Generate unique club and agent codes
      const uniqueClubId = IDGenerator.generateClubId();
      const uniqueAgentCode = IDGenerator.generateAgentCode();

        // Move file to final location
    let finalFilename = null;
    if (tempFilename) {
      finalFilename = await moveFileToClubFolder(tempUploadDir, clubName, tempFilename);
    }
      // Insert new club
      const clubQuery = `
        INSERT INTO clubs 
        (name, unique_club_id, owner_id, profile_picture, 
        owner_earnings_percentage, agent_earnings_percentage, 
        member_earnings_percentage, min_active_players_threshold, 
        active_player_wager_threshold,clubrake)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9,$10)
        RETURNING id
      `;
      const clubValues = [
        clubName,
        uniqueClubId,
        userId,
        profilePicture,
        ownerEarningsPercentage,
        agentEarningsPercentage,
        memberEarningsPercentage,
        5,     // Default min active players
        200.0, // Default active player wager threshold
        clubRake
      ];
      const clubResult = await pg.query(clubQuery, clubValues);
      const clubId = clubResult.rows[0].id;

      // Add user as owner to club memberships
      const membershipQuery = `
        INSERT INTO club_memberships 
        (user_id, club_id, role, unique_agent_code, joined_at)
        VALUES ($1, $2, $3, $4, CURRENT_TIMESTAMP)
      `;
      await pg.query(membershipQuery, [
        userId,
        clubId,
        'owner',
        uniqueAgentCode
      ]);

      await pg.query('COMMIT');

      res.status(201).json({
        message: 'Club created successfully',
        clubId: clubId,
        uniqueClubId: uniqueClubId,
        uniqueAgentCode: uniqueAgentCode
      });
    } catch (error) {
      await pg.query('ROLLBACK');
       // Cleanup club folder and contents
    try {
      if (fs.existsSync(clubFolderPath)) {
        await rmdir(clubFolderPath, { recursive: true, force: true });
      }
    } catch (cleanupError) {
      console.error('Failed to clean up folder:', cleanupError.message);
    }
      res.status(500).json({ error: error.message });
    } finally {
      client.release();
    }
  }
 


  async updateClub(req, res) {
    const { 
      clubId, 
      clubName, 
      ownerEarningsPercentage, 
      agentEarningsPercentage, 
      memberEarningsPercentage 
    } = req.body;
    
    try {
      await pg.query('BEGIN');
      
      // Get current club details
      const currentClub = await pg.query(
        'SELECT name, profile_picture FROM clubs WHERE id = $1',
        [clubId]
      );
      
      if (!currentClub.rows.length) {
        throw new Error('Club not found');
      }
  
      let profilePicturePath = currentClub.rows[0].profile_picture;
      const currentClubName = currentClub.rows[0].name;
      const baseDir = '/var/www/html/hellogames/ibitplay/backend/clubmembership/clubprofile';
      
      // Handle directory rename if club name is changing
      if (clubName && clubName !== currentClubName) {
        const oldClubDir = path.join(baseDir, currentClubName);
        const newClubDir = path.join(baseDir, clubName);
        
        try {
          if (fs.existsSync(oldClubDir)) {
            // If old directory exists, try to rename it first
            try {
              await fs.promises.rename(oldClubDir, newClubDir);
              console.log('Directory renamed successfully');
            } catch (renameError) {
              console.error('Direct rename failed, falling back to copy and delete:', renameError);
              
              // If rename fails, create new directory and copy contents
              await mkdir(newClubDir, { recursive: true });
              const files = await readdir(oldClubDir);
              
              // Copy all files from old to new directory
              for (const file of files) {
                const oldPath = path.join(oldClubDir, file);
                const newPath = path.join(newClubDir, file);
                await fs.promises.copyFile(oldPath, newPath);
              }
              
              // Remove old directory and its contents
              await fs.promises.rm(oldClubDir, { recursive: true, force: true });
            }
          } else {
            // If old directory doesn't exist, just create the new one
            await mkdir(newClubDir, { recursive: true });
          }
        } catch (error) {
          console.error('Error handling directory update:', error);
          throw new Error('Failed to update club directory structure');
        }
      }
      
      // Handle profile picture update if new file is uploaded
      if (req.file) {
        console.log('Uploaded file:', req.file);
        
        const targetClubDir = path.join(baseDir, clubName || currentClubName);
        
        // Ensure target directory exists
        await mkdir(targetClubDir, { recursive: true });
        
        // Delete old profile picture if it exists
        if (profilePicturePath) {
          const oldPath = path.join(targetClubDir, profilePicturePath);
          try {
            await unlink(oldPath);
            console.log('Deleted old file:', oldPath);
          } catch (error) {
            console.error('Error deleting old profile picture:', error);
          }
        }
  
        // Move new file from temp location to club directory
        try {
          const newFileName = req.file.filename;
          const newFilePath = path.join(targetClubDir, newFileName);
          
          console.log('Moving file from:', req.file.path);
          console.log('Moving file to:', newFilePath);
          
          await rename(req.file.path, newFilePath);
          profilePicturePath = newFileName;
          
          console.log('File moved successfully. New path:', profilePicturePath);
        } catch (error) {
          console.error('Error moving new profile picture:', error);
          throw new Error(`Failed to save profile picture: ${error.message}`);
        }
      }
      
      // Update club details in database
      const updateQuery = `
        UPDATE clubs 
        SET 
          name = COALESCE($2, name),
          profile_picture = COALESCE($3, profile_picture),
          owner_earnings_percentage = COALESCE($4, owner_earnings_percentage),
          agent_earnings_percentage = COALESCE($5, agent_earnings_percentage),
          member_earnings_percentage = COALESCE($6, member_earnings_percentage)
        WHERE id = $1
        RETURNING *
      `;
      
      const updateResult = await pg.query(updateQuery, [
        clubId,
        clubName,
        profilePicturePath,
        ownerEarningsPercentage,
        agentEarningsPercentage,
        memberEarningsPercentage
      ]);
      
      await pg.query('COMMIT');
      
      res.status(200).json({
        message: 'Club updated successfully',
        updatedClub: updateResult.rows[0]
      });
    } catch (error) {
      await pg.query('ROLLBACK');
      console.error('Update club error:', error);
      res.status(500).json({ error: error.message });
    }
  }
  // Update Club
  // async updateClub(req, res) {
  //   const { 
  //     clubId, 
  //     clubName, 
  //     ownerEarningsPercentage, 
  //     agentEarningsPercentage, 
  //     memberEarningsPercentage 
  //   } = req.body;
    
  //   try {
  //     await pg.query('BEGIN');
      
  //     // Get current club details
  //     const currentClub = await pg.query(
  //       'SELECT name, profile_picture FROM clubs WHERE id = $1',
  //       [clubId]
  //     );
      
  //     if (!currentClub.rows.length) {
  //       throw new Error('Club not found');
  //     }
      
  //     let profilePicturePath = currentClub.rows[0].profile_picture;
      
  //     // Handle profile picture update if new file is uploaded
  //     if (req.file) {
  //       const oldClubName = currentClub.rows[0].name.replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
  //       const newClubName = (clubName || currentClub.rows[0].name).replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
        
  //       // Delete old profile picture if it exists
  //       if (profilePicturePath) {
  //         const oldPath = path.join('clubsProfile', oldClubName, profilePicturePath);
  //         try {
  //           fs.unlink(oldPath);
  //         } catch (error) {
  //           console.error('Error deleting old profile picture:', error);
  //         }
  //       }
        
  //       // If club name is being updated, move the new file to correct directory
  //       if (clubName && oldClubName !== newClubName) {
  //         const oldFilePath = req.file.path;
  //         const newDir = path.join('clubsProfile', newClubName);
  //         fs.mkdir(newDir, { recursive: true });
  //         const newFilePath = path.join(newDir, req.file.filename);
  //         fs.rename(oldFilePath, newFilePath);
  //       }
        
  //       profilePicturePath = req.file.filename;
  //     }
      
  //     // Update club details
  //     const updateQuery = `
  //       UPDATE clubs 
  //       SET 
  //         name = COALESCE($2, name),
  //         profile_picture = COALESCE($3, profile_picture),
  //         owner_earnings_percentage = COALESCE($4, owner_earnings_percentage),
  //         agent_earnings_percentage = COALESCE($5, agent_earnings_percentage),
  //         member_earnings_percentage = COALESCE($6, member_earnings_percentage)
  //       WHERE id = $1
  //       RETURNING *
  //     `;
      
  //     const updateResult = await pg.query(updateQuery, [
  //       clubId,
  //       clubName,
  //       profilePicturePath,
  //       ownerEarningsPercentage,
  //       agentEarningsPercentage,
  //       memberEarningsPercentage
  //     ]);
      
  //     await pg.query('COMMIT');
      
  //     res.status(200).json({
  //       message: 'Club updated successfully',
  //       updatedClub: updateResult.rows[0]
  //     });
  //   } catch (error) {
  //     await pg.query('ROLLBACK');
  //     res.status(500).json({ error: error.message });
  //   }
  // }

  async getClubProfilePicture(req, res) {
    const { clubId } = req.params;
    
    try {
      // Get club details
      const club = await pg.query(
        'SELECT name, profile_picture FROM clubs WHERE id = $1',
        [clubId]
      );
      
      if (!club.rows.length) {
        return res.status(404).json({ error: 'Club not found' });
      }
      
      const { name, profile_picture } = club.rows[0];
      
      if (!profile_picture) {
        return res.status(404).json({ error: 'No profile picture found' });
      }
      
      const clubName = name.replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
      const imagePath = path.join('clubsProfile', clubName, profile_picture);
      
      // Check if file exists
      try {
        fs.access(imagePath);
      } catch (error) {
        return res.status(404).json({ error: 'Profile picture file not found' });
      }
      
      // Send file
      res.sendFile(path.resolve(imagePath));
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  }

  async getClubProfile(req, res) {
    const { userId } = req.params;
    
    try {
      // Get club details
   const club = await pg.query(`
    SELECT 
        id,
        name,
        unique_club_id,
        owner_id,
        profile_picture,
        created_at,
        is_active,
        owner_earnings_percentage,
        agent_earnings_percentage,
        min_active_players_threshold,
        active_player_wager_threshold,
        member_earnings_percentage,
        parent_club_id,
        clubrake
    FROM clubs 
    WHERE owner_id = $1
`, [userId]);
      
      if (!club.rows.length) {
        return res.status(404).json({ error: 'Club not found' });
      }
      
      const clubData = club.rows[0];
      const { id: clubId,name, profile_picture } = clubData;

      const { rows: userCountResult } = await pg.query(
        `
        SELECT COUNT(*) AS total_users 
        FROM club_memberships 
        WHERE club_id = $1 AND role != 'owner'
        `,
        [clubId]
      );
      const totalUsers = parseInt(userCountResult[0].total_users, 10) || 0;
      if (!profile_picture) {
        return res.json({
          ...clubData,
          total_users: totalUsers,
          profile_picture: null
        });
      }
      
      const clubName = name.replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
      // Use absolute path
      const imagePath = path.join('/var/www/html/hellogames/ibitplay/backend/clubmembership/clubprofile', clubName, profile_picture);
      
      try {
        // Read the image file
        const imageBuffer = await fs.promises.readFile(imagePath);
        // Convert to base64
        const base64Image = imageBuffer.toString('base64');
        
        // Send both club data and image
        res.json({
          ...clubData,
          total_users: totalUsers,
          profile_picture_data: {
            data: base64Image,
            contentType: path.extname(profile_picture).substring(1)
          }
        });
        
      } catch (error) {
        console.error('File access error:', error);
        // Return club data even if image is not found
        res.json({
          ...clubData,
          total_users: totalUsers,
          profile_picture_data: null,
          error: 'Image not found'
        });
      }
    } catch (error) {
      console.error('General error:', error);
      res.status(500).json({ error: error.message });
    }
  }
  // Delete Club
  async deleteClub(req, res) {
    const { clubId } = req.params;
    

    try {
      await pg.query('BEGIN');

      // Delete club profile picture
      const clubQuery = 'SELECT profile_picture FROM clubs WHERE id = $1';
      const clubResult = await pg.query(clubQuery, [clubId]);

      if (clubResult.rows.length > 0 && clubResult.rows[0].profile_picture) {
        const imagePath = path.join(__dirname, '../../clubsProfile', clubResult.rows[0].profile_picture);
        if (fs.existsSync(imagePath)) {
          fs.unlinkSync(imagePath);
        }
      }

      // Delete club and memberships
      await pg.query('DELETE FROM club_memberships WHERE club_id = $1', [clubId]);
      await pg.query('DELETE FROM clubs WHERE id = $1', [clubId]);

      await pg.query('COMMIT');

      res.status(200).json({ message: 'Club deleted successfully' });
    } catch (error) {
      await pg.query('ROLLBACK');
      res.status(500).json({ error: error.message });
    } finally {
      client.release();
    }
  }
 // Error response helper function
generateErrorResponse(table, invalidParams, allowedParams) {
  return {
    error: 'Invalid parameters',
    message: `Invalid parameters detected for table '${table}'`,
    details: {
      invalidParameters: invalidParams,
      allowedParameters: allowedParams,
      example: `Example valid request: /api/data/${table}?${allowedParams[0]}=value`
    }
  };
}
  // Fetch Data for All Tables
  async fetchData(req, res) {
    const { tableName } = req.params;
    const queryParams = req.query;
    
    try {
      // Define valid tables and their allowed parameters
      const tableConfig = {
        'clubs': {
          validParams: ['unique_club_id', 'owner_id', 'parent_club_id'],
          paramMapping: {
            'uniqueClubId': 'unique_club_id',
            'ownerId': 'owner_id',
            'parentClubId': 'parent_club_id'
          }
        },
        'club_memberships': {
          validParams: ['user_id', 'club_id', 'role', 'agent_id', 'parent_club_id'],
          paramMapping: {
            'userId': 'user_id',
            'clubId': 'club_id',
            'role': 'role',
            'agentId': 'agent_id',
            'parentClubId': 'parent_club_id'
          }
        },
        'club_earnings_configurations': {
          validParams: ['club_id'],
          paramMapping: {
            'clubId': 'club_id'
          }
        },
        'club_earnings_log': {
          validParams: ['club_id'],
          paramMapping: {
            'clubId': 'club_id'
          }
        }
      };
  
      // Validate table name
      if (!tableConfig[tableName]) {
        return res.status(400).json({ 
          error: 'Invalid table name',
          message: `Valid tables are: ${Object.keys(tableConfig).join(', ')}`
        });
      }
  
      const currentTableConfig = tableConfig[tableName];
      
      // Validate parameters
      const invalidParams = [];
      const allowedParams = Object.keys(currentTableConfig.paramMapping);
      
      // Check for invalid parameters in the request
      Object.keys(queryParams).forEach(param => {
        if (!allowedParams.includes(param)) {
          invalidParams.push(param);
        }
      });
  
      // If invalid parameters were found, return detailed error message
      if (invalidParams.length > 0) {
        return res.status(400).json({
          error: 'Invalid parameters detected',
          message: `The following parameters are not allowed for table '${tableName}': ${invalidParams.join(', ')}`,
          allowedParameters: {
            table: tableName,
            allowed: allowedParams,
            description: `Only the following parameters are allowed for '${tableName}': ${allowedParams.join(', ')}`
          }
        });
      }
  
      // Build query based on table-specific parameters
      let query = `SELECT * FROM ${tableName}`;
      const conditions = [];
      const values = [];
      let paramCount = 1;
  
      // Process only valid parameters for the current table
      const paramMapping = {};
      for (const [key, value] of Object.entries(queryParams)) {
        if (allowedParams.includes(key)) {
          paramMapping[key] = value;
        }
      }
  
      // Build query conditions
      for (const [paramName, paramValue] of Object.entries(paramMapping)) {
        if (paramValue != null) {
          const dbColumn = currentTableConfig.paramMapping[paramName];
          conditions.push(`${dbColumn} = $${paramCount}`);
          values.push(paramValue);
          paramCount++;
        }
      }
  
      if (conditions.length > 0) {
        query += ` WHERE ${conditions.join(' AND ')}`;
      }
  
      // Add order by id for consistent results
      query += ' ORDER BY id';
  
      // Log query for debugging
      console.log('Executing query:', {
        sql: query,
        values: values,
        table: tableName,
        requestedParams: Object.keys(queryParams)
      });
  
      const result = await pg.query(query, values);
      
      return res.status(200).json({
        success: true,
        data: result.rows,
        metadata: {
          count: result.rows.length,
          table: tableName,
          appliedFilters: conditions.length > 0 ? conditions : ['none'],
          message: `Successfully retrieved ${result.rows.length} records from ${tableName}`
        }
      });
  
    } catch (error) {
      console.error('Database query error:', {
        error: error.message,
        table: tableName,
        query: req.query
      });
  
      return res.status(500).json({
        error: 'Internal server error',
        message: error.message,
        details: {
          table: tableName,
          timestamp: new Date().toISOString(),
          errorType: error.name
        }
      });
  
    } finally {
      if (client) {
        try {
          await client.release();
        } catch (releaseError) {
          console.error('Error releasing client:', releaseError);
        }
      }
    }
  }
   
  // Join Club
async joinClub(req, res) {
    const { userId, clubId, agentCode } = req.body;
    if (!userId || (!clubId && !agentCode)) {
      return res.status(400).json({ error: 'userId and either clubId or agentCode are required' });
    }
    try {
      await pg.query('BEGIN');

      let targetClubId, targetAgentId = null, parentClubId = null;

      // Check if joining via Club ID
      if (clubId) {
        const clubQuery = 'SELECT id, owner_id FROM clubs WHERE unique_club_id = $1';
        const clubResult = await pg.query(clubQuery, [clubId]);

        if (clubResult.rows.length === 0) {
          return res.status(404).json({ error: 'Club not found' });
        }
        targetClubId = clubResult.rows[0].id;
        parentClubId = targetClubId;
      }

      // Check if joining via Agent Code
      if (agentCode) {
        const agentQuery = `
          SELECT 
            cm.club_id, 
            cm.user_id as agent_id, 
            c.id as parent_club_id
          FROM club_memberships cm
          JOIN clubs c ON cm.club_id = c.id
          WHERE cm.unique_agent_code = $1 AND cm.role = 'agent'
        `;
        const agentResult = await pg.query(agentQuery, [agentCode]);
        
        if (agentResult.rows.length === 0) {
          return res.status(404).json({ error: 'Agent not found' });
        }
        
        targetClubId = agentResult.rows[0].club_id;
        targetAgentId = agentResult.rows[0].agent_id;
        parentClubId = agentResult.rows[0].parent_club_id;
      }
      if (!targetClubId) {
        return res.status(400).json({ error: 'Club ID could not be determined' });
      }
  
      // Check if user is already a member
    const existingMemberQuery = `
    SELECT 
        id,
        user_id,
        club_id,
        role,
        agent_id,
        unique_agent_code,
        joined_at,
        last_active_at,
        player_rake_percentage,
        total_wager,
        parent_club_id
    FROM club_memberships 
    WHERE user_id = $1 AND club_id = $2
`;
      const existingMember = await pg.query(existingMemberQuery, [userId, targetClubId]);
      
      if (existingMember.rows.length > 0) {
        return res.status(400).json({ error: 'Already a club member' });
      }

      // Insert membership
      const membershipQuery = `
        INSERT INTO club_memberships 
        (user_id, club_id, role, agent_id, parent_club_id, joined_at)
        VALUES ($1, $2, $3, $4, $5, CURRENT_TIMESTAMP)
      `;
      await pg.query(membershipQuery, [
        userId, 
        targetClubId, 
        'member', 
        targetAgentId,
        parentClubId
      ]);

      await pg.query('COMMIT');

      res.status(201).json({
        message: 'Successfully joined club',
        clubId: targetClubId
      });
    } catch (error) {
      await pg.query('ROLLBACK');
      res.status(500).json({ error: error.message });
    } finally {
      client.release();
    }
  }

  // Change User Role
  async changeUserRole(req, res) {
    const { userId, clubId, newRole } = req.body;
    try {
      await pg.query('BEGIN');

      // Validate allowed role changes
      const allowedRoleChanges = {
        'member': ['agent'],
        'agent': ['member']
      };

      // Get current role
      const currentRoleQuery = `
        SELECT role 
        FROM club_memberships 
        WHERE user_id = $1 AND club_id = $2
      `;
      const currentRoleResult = await pg.query(currentRoleQuery, [userId, clubId]);

      if (currentRoleResult.rows.length === 0) {
        return res.status(404).json({ error: 'Membership not found' });
      }

      const currentRole = currentRoleResult.rows[0].role;

      // Check if role change is allowed
      if (!allowedRoleChanges[currentRole]?.includes(newRole)) {
        return res.status(400).json({ error: 'Invalid role change' });
      }

      // Generate new agent code if becoming an agent
      const uniqueAgentCode = newRole === 'agent' 
        ? IDGenerator.generateAgentCode()
        : null;

      // Update role and agent code
      const updateQuery = `
        UPDATE club_memberships 
        SET 
          role = $3, 
          unique_agent_code = $4
        WHERE user_id = $1 AND club_id = $2
        RETURNING *
      `;
      const updateResult = await pg.query(updateQuery, [
        userId, 
        clubId, 
        newRole, 
        uniqueAgentCode
      ]);

      await pg.query('COMMIT');

      res.status(200).json({
        message: `Successfully changed role to ${newRole}`,
        membership: updateResult.rows[0],
        newAgentCode: uniqueAgentCode
      });
    } catch (error) {
      await pg.query('ROLLBACK');
      res.status(500).json({ error: error.message });
    } finally {
      client.release();
    }
  }


  async getClubHierarchy(req, res) {
    const { clubId } = req.params;
    
    try {
      const numericClubId = Number(clubId.replace(/[^0-9]/g, ''));
      
      if (!Number.isInteger(numericClubId) || numericClubId <= 0) {
        return res.status(400).json({
          error: 'Invalid club ID format. Must be a positive integer.'
        });
      }
  
      const hierarchyQuery = `
 WITH RECURSIVE club_hierarchy AS (
    -- Base case: Start with owner and direct members/agents
    SELECT 
        cm.user_id,
        cm.role,
        cm.unique_agent_code,
        u.name AS user_name,
        cm.agent_id,
        CASE 
            WHEN cm.role = 'owner' THEN 0
            WHEN cm.agent_id IS NULL THEN 1
            ELSE array_length(string_to_array(cm.agent_id::TEXT, ','), 1) + 1
        END AS level,
        cm.agent_id AS parent_user_id,
        ARRAY[cm.user_id::TEXT] AS path,
        ARRAY[cm.user_id::TEXT] AS full_hierarchy_path
    FROM club_memberships cm
    JOIN users u ON cm.user_id = u.id
    WHERE cm.club_id = $1

    UNION ALL

    -- Recursive case: Find members under agents
    SELECT 
        cm.user_id,
        cm.role,
        cm.unique_agent_code,
        u.name AS user_name,
        cm.agent_id,
        ch.level + 1, -- Increment level for each level of depth
        cm.agent_id AS parent_user_id,
        ch.path || cm.user_id::TEXT,
        ch.full_hierarchy_path || cm.user_id::TEXT
    FROM club_memberships cm
    JOIN users u ON cm.user_id = u.id
    JOIN club_hierarchy ch ON cm.agent_id = ch.user_id
    WHERE 
        cm.club_id = $1 
        AND NOT (cm.user_id::TEXT = ANY(ch.path)) -- Prevent cycles
)
SELECT DISTINCT ON (h.user_id)
    h.user_id,
    h.role,
    h.unique_agent_code,
    h.user_name,
    h.agent_id,
    h.level,
    h.parent_user_id,
    p.name AS parent_name,
    pm.role AS parent_role,
    h.full_hierarchy_path
FROM club_hierarchy h
LEFT JOIN club_memberships pm ON h.parent_user_id = pm.user_id AND pm.club_id = $1
LEFT JOIN users p ON h.parent_user_id = p.id
ORDER BY h.user_id, array_length(h.full_hierarchy_path, 1) DESC;

      `;
  
      const hierarchyResult = await pg.query(hierarchyQuery, [numericClubId]);
      
      if (hierarchyResult.rows.length === 0) {
        return res.status(404).json({
          message: 'No hierarchy found for this club ID'
        });
      }
  
      res.status(200).json({
        message: 'Club hierarchy retrieved',
        hierarchy: hierarchyResult.rows
      });
      
    } catch (error) {
      console.error('Error in getClubHierarchy:', error);
      res.status(500).json({
        error: 'Internal server error while retrieving club hierarchy'
      });
    }
  }
  // Update Earnings Configuration
  async updateEarningsConfiguration(req, res) {
    const { 
      clubId, 
      ownerEarningsPercentage, 
      agentEarningsPercentage, 
      memberEarningsPercentage,
      activePlayerThreshold,
      activePlayerWagerThreshold
    } = req.body;

    

    try {
      await pg.query('BEGIN');

      const updateQuery = `
        UPDATE clubs 
        SET 
          owner_earnings_percentage = COALESCE($2, owner_earnings_percentage),
          agent_earnings_percentage = COALESCE($3, agent_earnings_percentage),
          member_earnings_percentage = COALESCE($4, member_earnings_percentage),
          min_active_players_threshold = COALESCE($5, min_active_players_threshold),
          active_player_wager_threshold = COALESCE($6, active_player_wager_threshold)
        WHERE id = $1
        RETURNING *
      `;

      const result = await pg.query(updateQuery, [
        clubId,
        ownerEarningsPercentage,
        agentEarningsPercentage,
        memberEarningsPercentage,
        activePlayerThreshold,
        activePlayerWagerThreshold
      ]);

      await pg.query('COMMIT');

      res.status(200).json({
        message: 'Earnings configuration updated',
        updatedClub: result.rows[0]
      });
    } catch (error) {
      await pg.query('ROLLBACK');
      res.status(500).json({ error: error.message });
    } finally {
      client.release();
    }
  }
  // Add to clubController.js

async getUserClubAffiliations(req, res) {
  const { userId } = req.params;
  
  if (!userId) {
    return res.status(400).json({
      error: 'User ID is required'
    });
  }

  try {
    // Query to get all club affiliations for the user
    const query = `
      WITH user_affiliations AS (
        -- Get clubs owned by user
        SELECT 
          c.id as club_id,
          c.name as club_name,
          c.unique_club_id,
          'owner' as role,
          c.created_at,
          c.is_active,
          NULL as agent_id,
          NULL as agent_name,
          cm.unique_agent_code,
          cm.joined_at
        FROM clubs c
        JOIN club_memberships cm ON c.id = cm.club_id
        WHERE c.owner_id = $1 AND cm.role = 'owner'

        UNION ALL

        -- Get clubs where user is member or agent
        SELECT 
          c.id as club_id,
          c.name as club_name,
          c.unique_club_id,
          cm.role,
          c.created_at,
          c.is_active,
          cm.agent_id,
          ag_user.name as agent_name,
          cm.unique_agent_code,
          cm.joined_at
        FROM club_memberships cm
        JOIN clubs c ON cm.club_id = c.id
        LEFT JOIN users ag_user ON cm.agent_id = ag_user.id
        WHERE cm.user_id = $1 AND cm.role != 'owner'
      )
      SELECT * FROM user_affiliations
      ORDER BY role, joined_at DESC
    `;

    const result = await pg.query(query, [userId]);

    // Organize the results
    const affiliations = {
      isClubOwner: false,
      ownedClubs: [],
      isMember: false,
      memberships: [],
      isAgent: false,
      agentClubs: [],
      totalAffiliations: result.rows.length
    };

    // Process results
    result.rows.forEach(row => {
      const clubInfo = {
        clubId: row.club_id,
        clubName: row.club_name,
        uniqueClubId: row.unique_club_id,
        isActive: row.is_active,
        joinedAt: row.joined_at,
        uniqueAgentCode: row.unique_agent_code
      };

      if (row.role === 'owner') {
        affiliations.isClubOwner = true;
        affiliations.ownedClubs.push({
          ...clubInfo,
          createdAt: row.created_at
        });
      } else if (row.role === 'agent') {
        affiliations.isAgent = true;
        affiliations.agentClubs.push(clubInfo);
      } else if (row.role === 'member') {
        affiliations.isMember = true;
        affiliations.memberships.push({
          ...clubInfo,
          agentId: row.agent_id,
          agentName: row.agent_name
        });
      }
    });

    return res.status(200).json({
      success: true,
      data: affiliations,
      metadata: {
        userId: userId,
        timestamp: new Date().toISOString()
      }
    });

  } catch (error) {
    console.error('Error fetching user club affiliations:', error);
    return res.status(500).json({
      error: 'Internal server error',
      message: error.message
    });
  }
}
}

module.exports = new ClubController();