const express = require('express');
const router = express.Router();
const pg = require('../General/Model');

// Get user's crypto deposits with pagination
router.get('/user/crypto/deposits/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        const {
            page = 1,
            limit = 10,
            status,
            startDate,
            endDate,
            chain
        } = req.query;

        let countQuery = `
            SELECT COUNT(*) 
            FROM ccdeposit
            WHERE userid = $1
        `;
        
        const values = [userId];
        let valueIndex = 2;

        if (status) {
            countQuery += ` AND status = $${valueIndex}`;
            values.push(status);
            valueIndex++;
        }

        if (startDate) {
            countQuery += ` AND created_at >= $${valueIndex}`;
            values.push(startDate);
            valueIndex++;
        }

        if (endDate) {
            countQuery += ` AND created_at <= $${valueIndex}`;
            values.push(endDate);
            valueIndex++;
        }

        if (chain) {
            countQuery += ` AND chain = $${valueIndex}`;
            values.push(chain);
            valueIndex++;
        }

        const countResult = await pg.query(countQuery, values);
        const totalItems = parseInt(countResult.rows[0].count);

        let query = `
            SELECT 
                id,
                coinid,
                price,
                orderid,
                chain,
                deposit_address,
                amount,
                memo,
                checkout_url,
                confirms_needed,
                created_at,
                updated_at,
                status
            FROM ccdeposit
            WHERE userid = $1
        `;

        if (status) query += ` AND status = $2`;
        if (startDate) query += ` AND created_at >= $3`;
        if (endDate) query += ` AND created_at <= $4`;
        if (chain) query += ` AND chain = $5`;

        const offset = (parseInt(page) - 1) * parseInt(limit);
        query += ` ORDER BY created_at DESC LIMIT $${valueIndex} OFFSET $${valueIndex + 1}`;
        values.push(parseInt(limit), offset);

        const result = await pg.query(query, values);

        res.json({
            success: true,
            data: result.rows,
            page: parseInt(page),
            limit: parseInt(limit),
            total: totalItems,
            totalPages: Math.ceil(totalItems / parseInt(limit))
        });
    } catch (error) {
        console.error('Error fetching user crypto deposits:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

// Get user's crypto deposit stats
router.get('/user/crypto/stats/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        const stats = await pg.query(`
            SELECT 
                COUNT(*)::integer as total_transactions,
                COALESCE(SUM(amount), 0)::float as total_amount,
                COUNT(CASE WHEN LOWER(status) = 'success' THEN 1 END)::integer as successful_transactions,
                COUNT(CASE WHEN LOWER(status) = 'processing' THEN 1 END)::integer as processing_transactions,
                COUNT(CASE WHEN LOWER(status) = 'failed' THEN 1 END)::integer as failed_transactions
            FROM ccdeposit
            WHERE userid = $1
        `, [userId]);

        res.json({
            success: true,
            data: stats.rows[0]
        });
    } catch (error) {
        console.error('Error fetching user crypto deposit stats:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

// Get user's fiat deposits with pagination
router.get('/user/fiat/deposits/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        const {
            page = 1,
            limit = 10,
            status,
            startDate,
            endDate,
            currency
        } = req.query;

        let countQuery = `
            SELECT COUNT(*) 
            FROM pay_in_transactions
            WHERE user_id = $1
        `;
        
        const values = [parseInt(userId)];
        let valueIndex = 2;

        if (status !== undefined) {
            countQuery += ` AND status = $${valueIndex}`;
            values.push(parseInt(status));
            valueIndex++;
        }

        if (startDate) {
            countQuery += ` AND created_at >= $${valueIndex}`;
            values.push(startDate);
            valueIndex++;
        }

        if (endDate) {
            countQuery += ` AND created_at <= $${valueIndex}`;
            values.push(endDate);
            valueIndex++;
        }

        if (currency) {
            countQuery += ` AND currency = $${valueIndex}`;
            values.push(currency);
            valueIndex++;
        }

        const countResult = await pg.query(countQuery, values);
        const totalItems = parseInt(countResult.rows[0].count);

        let query = `
            SELECT 
                id,
                transaction_id,
                out_trade_no,
                currency,
                amount,
                pay_amount,
                merchant_ratio,
                real_amount,
                status,
                pay_type,
                created_at,
                updated_at
            FROM pay_in_transactions
            WHERE user_id = $1
        `;

        if (status !== undefined) query += ` AND status = $2`;
        if (startDate) query += ` AND created_at >= $3`;
        if (endDate) query += ` AND created_at <= $4`;
        if (currency) query += ` AND currency = $5`;

        const offset = (parseInt(page) - 1) * parseInt(limit);
        query += ` ORDER BY created_at DESC LIMIT $${valueIndex} OFFSET $${valueIndex + 1}`;
        values.push(parseInt(limit), offset);

        const result = await pg.query(query, values);

        res.json({
            success: true,
            data: result.rows,
            page: parseInt(page),
            limit: parseInt(limit),
            total: totalItems,
            totalPages: Math.ceil(totalItems / parseInt(limit))
        });
    } catch (error) {
        console.error('Error fetching user fiat deposits:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

// Get user's fiat deposit stats
router.get('/user/fiat/stats/:userId', async (req, res) => {
    try {
        const userId = req.params.userId;
        const query = `
            SELECT
                COUNT(*)::integer as total_transactions,
                COALESCE(SUM(amount), 0)::float as total_amount,
                COUNT(CASE WHEN status = 1 THEN 1 END)::integer as successful_transactions,
                COUNT(CASE WHEN status = 0 THEN 1 END)::integer as processing_transactions,
                COUNT(CASE WHEN status = 2 THEN 1 END)::integer as failed_transactions
            FROM pay_in_transactions
            WHERE user_id = $1
        `;
        
        const result = await pg.query(query, [parseInt(userId)]);
        res.json({
            success: true,
            data: result.rows[0]
        });
    } catch (error) {
        console.error('Error fetching user fiat stats:', error);
        res.status(500).json({
            success: false,
            error: 'Internal server error'
        });
    }
});

module.exports = router;