// otp.service.js
const crypto = require('crypto');
const bcrypt = require('bcrypt');
const pg = require('../General/Model');
const EmailTemplateService = require('./templet');

class OTPService {
    constructor() {
        this.createOTPTable();
    }

    async createOTPTable() {
        const query = `
            CREATE TABLE IF NOT EXISTS user_otps (
                id SERIAL PRIMARY KEY,
                email TEXT NOT NULL,
                otp_hash TEXT NOT NULL,
                purpose TEXT NOT NULL,
                attempts INTEGER DEFAULT 0,
                created_at TIMESTAMPTZ DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMPTZ,
                is_verified BOOLEAN DEFAULT FALSE,
                user_id BIGINT,  -- Added to link with specific user
                CONSTRAINT fk_email
                    FOREIGN KEY(email) 
                    REFERENCES users(email) 
                    ON DELETE CASCADE
            )
        `;
        await pg.query(query);
    }

    async getUserByEmail(email) {
        const result = await pg.query(
            'SELECT id FROM users WHERE email = $1',
            [email]
        );
        return result.rows[0];
    }

    generateOTP() {
        return crypto.randomBytes(3).toString('hex').toUpperCase().slice(0, 6);
    }

    async generateAndSaveOTP(email, purpose) {
        let userId = null;

        // For non-registration purposes, verify user exists
        if (purpose !== 'register') {
            const user = await this.getUserByEmail(email);
            if (!user) {
                throw new Error('User not found');
            }
            userId = user.id;
        }

        // Clear any existing unverified OTPs
        await pg.query(
            `DELETE FROM user_otps 
             WHERE email = $1 
             AND purpose = $2 
             AND is_verified = FALSE`,
            [email, purpose]
        );

        const otp = this.generateOTP();
        const hashedOTP = await bcrypt.hash(otp, 10);

        // Save new OTP with user_id if available
        await pg.query(
            `INSERT INTO user_otps 
             (email, otp_hash, purpose, expires_at, user_id)
             VALUES ($1, $2, $3, CURRENT_TIMESTAMP + INTERVAL '2 minutes', $4)`,
            [email, hashedOTP, purpose, userId]
        );

        return otp;
    }

    async verifyOTP(email, otp, purpose) {
        const result = await pg.query(
            `SELECT uo.*, u.id as actual_user_id, 
                    EXTRACT(EPOCH FROM (CURRENT_TIMESTAMP - uo.created_at))/60 as minutes_passed
             FROM user_otps uo
             LEFT JOIN users u ON u.email = uo.email
             WHERE uo.email = $1 
             AND uo.purpose = $2 
             AND uo.is_verified = FALSE 
             AND uo.attempts < 3
             ORDER BY uo.created_at DESC 
             LIMIT 1`,
            [email, purpose]
        );

        if (!result.rows.length) {
            throw new Error('No OTP found');
        }

        const otpRecord = result.rows[0];

        // Check if OTP is expired (over 2 minutes)
        if (otpRecord.minutes_passed >= 2) {
            throw new Error('OTP has expired. Please request a new one.');
        }

        // Verify if OTP belongs to the correct user
        if (purpose !== 'register' && otpRecord.user_id !== otpRecord.actual_user_id) {
            throw new Error('Invalid OTP');
        }

        const isValid = await bcrypt.compare(otp, otpRecord.otp_hash);

        if (!isValid) {
            await pg.query(
                'UPDATE user_otps SET attempts = attempts + 1 WHERE id = $1',
                [otpRecord.id]
            );
            throw new Error('Invalid OTP');
        }

        await pg.query(
            'UPDATE user_otps SET is_verified = TRUE WHERE id = $1',
            [otpRecord.id]
        );

        return true;
    }

    async canResendOTP(email, purpose) {
        const result = await pg.query(
            `SELECT created_at 
             FROM user_otps 
             WHERE email = $1 
             AND purpose = $2 
             ORDER BY created_at DESC 
             LIMIT 1`,
            [email, purpose]
        );

        if (!result.rows.length) {
            return true;
        }

        const lastOTPTime = new Date(result.rows[0].created_at);
        const timeDiff = (new Date() - lastOTPTime) / 1000; // in seconds
        return timeDiff >= 120; // 2 minutes
    }
}

module.exports = new OTPService();