// controllers/exchangeRateController.js

const pg = require('../General/Model');

const exchangeRateController = {
    // Get all exchange rates
    getAllRates: async (req, res) => {
        try {
            const query = 'SELECT currency, usd_rate, last_updated FROM exchangerate ORDER BY currency';
            const result = await pg.query(query);
            
            res.status(200).json({
                success: true,
                data: result.rows
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error fetching exchange rates',
                error: error.message
            });
        }
    },

    // Get rate by currency
    getRateByCurrency: async (req, res) => {
        try {
            const { currency } = req.params;
            const query = 'SELECT currency, usd_rate, last_updated FROM exchangerate WHERE UPPER(currency) = UPPER($1)';
            const result = await pg.query(query, [currency]);

            if (result.rows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Currency not found'
                });
            }

            res.status(200).json({
                success: true,
                data: result.rows[0]
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error fetching exchange rate',
                error: error.message
            });
        }
    },

    // Add new currency and rate
    addRate: async (req, res) => {
        try {
            const { currency, usd_rate } = req.body;

            // Validation
            if (!currency || !usd_rate) {
                return res.status(400).json({
                    success: false,
                    message: 'Currency and rate are required'
                });
            }

            if (usd_rate <= 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Rate must be greater than 0'
                });
            }

            const query = `
                INSERT INTO exchangerate (currency, usd_rate)
                VALUES (UPPER($1), $2)
                RETURNING currency, usd_rate, last_updated
            `;
            const result = await pg.query(query, [currency, usd_rate]);

            res.status(201).json({
                success: true,
                message: 'Exchange rate added successfully',
                data: result.rows[0]
            });
        } catch (error) {
            if (error.constraint === 'unique_currency') {
                return res.status(409).json({
                    success: false,
                    message: 'Currency already exists'
                });
            }

            res.status(500).json({
                success: false,
                message: 'Error adding exchange rate',
                error: error.message
            });
        }
    },

    // Update rate for a currency
    updateRate: async (req, res) => {
        try {
            const { currency } = req.params;
            const { usd_rate } = req.body;

            // Validation
            if (!usd_rate) {
                return res.status(400).json({
                    success: false,
                    message: 'Rate is required'
                });
            }

            if (usd_rate <= 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Rate must be greater than 0'
                });
            }

            const query = `
                UPDATE exchangerate 
                SET usd_rate = $1, last_updated = CURRENT_TIMESTAMP
                WHERE UPPER(currency) = UPPER($2)
                RETURNING currency, usd_rate, last_updated
            `;
            const result = await pg.query(query, [usd_rate, currency]);

            if (result.rows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Currency not found'
                });
            }

            res.status(200).json({
                success: true,
                message: 'Exchange rate updated successfully',
                data: result.rows[0]
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error updating exchange rate',
                error: error.message
            });
        }
    },

    // Delete a currency
    deleteRate: async (req, res) => {
        try {
            const { currency } = req.params;
            const query = 'DELETE FROM exchangerate WHERE UPPER(currency) = UPPER($1) RETURNING currency';
            const result = await pg.query(query, [currency]);

            if (result.rows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Currency not found'
                });
            }

            res.status(200).json({
                success: true,
                message: 'Exchange rate deleted successfully',
                data: result.rows[0]
            });
        } catch (error) {
            res.status(500).json({
                success: false,
                message: 'Error deleting exchange rate',
                error: error.message
            });
        }
    }
};

module.exports = exchangeRateController;