const pg = require('../General/Model');
const fs = require('fs');
const path = require('path');

module.exports = {
  // User: Create Deposit
  async createDeposit(req, res) {
    try {
      const { userId, amount, paymentMethod } = req.body;
      const screenshot = req.file?.filename;

      if (!userId || !amount || !paymentMethod || !screenshot) {
        return res.status(400).json({ success: false, message: 'Missing required fields' });
      }

      const result = await pg.query(
        `INSERT INTO deposits (user_id, amount, payment_method, screenshot, status)
         VALUES ($1, $2, $3, $4, 'pending') RETURNING *`,
        [userId, amount, paymentMethod, screenshot]
      );

      res.json({ success: true, data: result.rows[0] });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // User: Get all deposits
  async getUserDeposits(req, res) {
    try {
      const { userId } = req.params;

      const result = await pg.query(
        `SELECT * FROM deposits WHERE user_id = $1 ORDER BY created_at DESC`,
        [userId]
      );

      res.json({ success: true, data: result.rows });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // User: Get deposit by ID
  async getDepositById(req, res) {
    try {
      const { depositId } = req.params;
      const { userId } = req.query;

      const result = await pg.query(
        `SELECT * FROM deposits WHERE id = $1 AND user_id = $2`,
        [depositId, userId]
      );

      if (result.rowCount === 0) {
        return res.status(404).json({ success: false, message: 'Deposit not found' });
      }

      res.json({ success: true, data: result.rows[0] });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // User: Get screenshot
  async getDepositScreenshot(req, res) {
    try {
      const { filename } = req.params;
      const filePath = path.join(__dirname, '../../uploads/deposits', filename);

      if (!fs.existsSync(filePath)) {
        return res.status(404).json({ success: false, message: 'Screenshot not found' });
      }

      res.sendFile(filePath);
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // Admin: Get all deposits
  async getAllDeposits(req, res) {
    try {
      const result = await pg.query(`
        SELECT d.*, u.email
        FROM deposits d
        JOIN users u ON d.user_id = u.id
        ORDER BY d.created_at DESC
      `);

      res.json({ success: true, data: result.rows });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // Admin: Get pending deposits
  async getPendingDeposits(_req, res) {
    try {
      const result = await pg.query(
        `SELECT d.*, u.email
         FROM deposits d
         JOIN users u ON d.user_id = u.id
         WHERE d.status = 'pending'
         ORDER BY d.created_at DESC`
      );

      res.json({ success: true, data: result.rows });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  },

  // Admin: Approve deposit
  async approveDeposit(req, res) {
    const client = await pg.connect();
    try {
      const { depositId } = req.body;

      await client.query('BEGIN');

      const { rows: depositRows } = await client.query(
        `UPDATE deposits
         SET status = 'approved', approved_at = CURRENT_TIMESTAMP
         WHERE id = $1 AND status = 'pending'
         RETURNING *`,
        [depositId]
      );

      if (depositRows.length === 0) {
        throw new Error('Deposit not found or already processed');
      }

      const deposit = depositRows[0];

      await client.query(
        `UPDATE users
         SET balance = balance + $1
         WHERE id = $2`,
        [deposit.amount, deposit.user_id]
      );

      await client.query('COMMIT');
      res.json({ success: true, message: 'Deposit approved and balance updated' });
    } catch (error) {
      await client.query('ROLLBACK');
      res.status(500).json({ success: false, message: error.message });
    } finally {
      client.release();
    }
  },

  // Admin: Reject deposit
  async rejectDeposit(req, res) {
    try {
      const { depositId, comment } = req.body;

      const result = await pg.query(
        `UPDATE deposits
         SET status = 'rejected', admin_comment = $2, rejected_at = CURRENT_TIMESTAMP
         WHERE id = $1 AND status = 'pending'
         RETURNING *`,
        [depositId, comment]
      );

      if (result.rowCount === 0) {
        return res.status(404).json({ success: false, message: 'Deposit not found or already processed' });
      }

      res.json({ success: true, message: 'Deposit rejected' });
    } catch (error) {
      res.status(500).json({ success: false, message: error.message });
    }
  }
};
