const admin = require('./config');
const pg = require('../General/Model');

class NotificationService {
    async registerToken(userId, token, deviceType, platform) {
        // Start a transaction
      
        try {
            await pg.query('BEGIN');
            
            // Check if user already has a record
            const existingRecord = await pg.query(`
    SELECT 
        id,
        user_id,
        token,
        device_type,
        platform,
        is_active,
        last_used,
        created_at
    FROM user_fcm_tokens 
    WHERE user_id = $1
`, [userId]);
    
            let result;
            
            if (existingRecord.rows.length > 0) {
                // User exists - check if token is different
                if (existingRecord.rows[0].token !== token) {
                    // Update with new token
                    result = await pg.query(
                        `UPDATE user_fcm_tokens 
                         SET token = $2,
                             device_type = $3,
                             platform = $4,
                             last_used = CURRENT_TIMESTAMP,
                             is_active = true
                         WHERE user_id = $1
                         RETURNING *`,
                        [userId, token, deviceType, platform]
                    );
                } else {
                    // Just update the last_used timestamp
                    result = await pg.query(
                        `UPDATE user_fcm_tokens 
                         SET last_used = CURRENT_TIMESTAMP
                         WHERE user_id = $1
                         RETURNING *`,
                        [userId]
                    );
                }
            } else {
                // No existing record - insert new one
                result = await pg.query(
                    `INSERT INTO user_fcm_tokens 
                     (user_id, token, device_type, platform, is_active)
                     VALUES ($1, $2, $3, $4, true)
                     RETURNING *`,
                    [userId, token, deviceType, platform]
                );
            }
    
            await pg.query('COMMIT');
            return result.rows[0];
            
        } catch (error) {
            await pg.query('ROLLBACK');
            throw error;
        } finally {
            console.log("registration done for fcm")
        }
    }
    // async sendToUser(userId, notification) {
    //     const tokenQuery = `
    //         SELECT token FROM user_fcm_tokens 
    //         WHERE user_id = $1 AND is_active = true
    //     `;
        
    //     const tokens = await pg.query(tokenQuery, [userId]);
    //     if (!tokens.rows.length) {
    //         throw new Error('No active tokens found');
    //     }

    //     const notifQuery = `
    //         INSERT INTO user_notifications 
    //         (user_id, title, body, type, additional_data)
    //         VALUES ($1, $2, $3, $4, $5)
    //         RETURNING id
    //     `;

    //     const notifResult = await pg.query(notifQuery, [
    //         userId,
    //         notification.title,
    //         notification.body,
    //         notification.type || 'general',
    //         notification.data || {}
    //     ]);

    //     const notifId = notifResult.rows[0].id;

    //     const messages = tokens.rows.map(row => ({
    //         token: row.token,
    //         notification: {
    //             title: notification.title,
    //             body: notification.body
    //         },
    //         data: {
    //             ...notification.data,
    //             notificationId: notifId.toString()
    //         }
    //     }));

    //     await Promise.all(messages.map(msg => admin.messaging().send(msg)));
    //     return notifId;
    // }
    async sendToUser(userId, notification) {
        console.log('[NotificationService] Starting sendToUser:', {
            userId,
            notificationType: notification.type,
            hasData: !!notification.data
        });
    
        const tokenQuery = `
            SELECT token FROM user_fcm_tokens 
            WHERE user_id = $1 AND is_active = true
        `;
        
        console.log('[NotificationService] Executing token query for userId:', userId);
        const tokens = await pg.query(tokenQuery, [userId]);
        console.log('[NotificationService] Token query results:', {
            foundTokens: tokens.rows.length,
            tokens: tokens.rows.map(r => r.token.substring(0, 20) + '...')  // Log partial tokens for security
        });
    
        if (!tokens.rows.length) {
            console.warn('[NotificationService] No active tokens found for userId:', userId);
            throw new Error('No active tokens found');
        }
    
        const notifQuery = `
            INSERT INTO user_notifications 
            (user_id, title, body, type, additional_data)
            VALUES ($1, $2, $3, $4, $5)
            RETURNING id
        `;
    
        console.log('[NotificationService] Inserting notification into database');
        const notifResult = await pg.query(notifQuery, [
            userId,
            notification.title,
            notification.body,
            notification.type || 'general',
            notification.data || {}
        ]);
        console.log('[NotificationService] Notification inserted with ID:', notifResult.rows[0].id);
    
        const notifId = notifResult.rows[0].id;
    
        const messages = tokens.rows.map(row => ({
            token: row.token,
            notification: {
                title: notification.title,
                body: notification.body
            },
            data: {
                ...notification.data,
                notificationId: notifId.toString()
            }
        }));
    
        console.log('[NotificationService] Preparing to send messages:', {
            messageCount: messages.length,
            notificationId: notifId,
            sampleToken: messages[0]?.token.substring(0, 20) + '...'
        });
    
        try {
            const results = await Promise.all(
                messages.map(async (msg, index) => {
                    try {
                        const result = await admin.messaging().send(msg);
                        console.log(`[NotificationService] Successfully sent message ${index + 1}/${messages.length}:`, {
                            result,
                            token: msg.token.substring(0, 20) + '...'
                        });
                        return { success: true, result };
                    } catch (error) {
                        console.error(`[NotificationService] Failed to send message ${index + 1}/${messages.length}:`, {
                            error: {
                                code: error.code,
                                message: error.message,
                                stack: error.stack
                            },
                            token: msg.token.substring(0, 20) + '...'
                        });
                        return { success: false, error };
                    }
                })
            );
    
            const failedCount = results.filter(r => !r.success).length;
            if (failedCount > 0) {
                console.warn('[NotificationService] Some messages failed to send:', {
                    total: messages.length,
                    failed: failedCount,
                    notificationId: notifId
                });
            }
    
            return notifId;
        } catch (error) {
            console.error('[NotificationService] Critical error while sending messages:', {
                error: {
                    code: error.code,
                    message: error.message,
                    stack: error.stack
                },
                notificationId: notifId
            });
            throw error;
        }
    }
    
    async sendBulk(notification) {
        const query = `
            SELECT DISTINCT user_id, token 
            FROM user_fcm_tokens 
            WHERE is_active = true
        `;

        const tokens = await pg.query(query);
        return Promise.allSettled(
            tokens.rows.map(row => this.sendToUser(row.user_id, notification))
        );
    }

    async getUnreadCount(userId) {
        const query = `
            SELECT COUNT(*) 
            FROM user_notifications 
            WHERE user_id = $1 AND is_read = false
        `;
        const result = await pg.query(query, [userId]);
        return parseInt(result.rows[0].count);
    }

    async markAsRead(userId, notificationIds) {
        const query = `
            UPDATE user_notifications 
            SET is_read = true 
            WHERE user_id = $1 
            AND id = ANY($2)
            RETURNING id, title, body, type, is_read, created_at
        `;
        const result = await pg.query(query, [userId, notificationIds]);
        return result.rows;
    }

    async getNotificationHistory(userId) {
        const query = `
            SELECT 
                id,
                title,
                body,
                type,
                is_read,
                additional_data,
                created_at
            FROM user_notifications 
            WHERE user_id = $1
            ORDER BY created_at DESC
        `;
        const result = await pg.query(query, [userId]);
        
        // Separate notifications into read and unread
        const unread = result.rows.filter(notification => !notification.is_read);
        const read = result.rows.filter(notification => notification.is_read);
        
        return { unread, read };
    }
    async getAllTokens() {
        const query = `
            SELECT 
                u.id AS user_id,
                u.name,
                u.email,
                t.*
            FROM user_fcm_tokens t
            JOIN users u ON u.id = t.user_id
            ORDER BY t.created_at DESC
        `;
        const result = await pg.query(query);
        return result.rows;
    }
    
}

module.exports = new NotificationService();
