// routes/admin.js
const express = require('express');
const router  = express.Router();
const pg      = require('../General/Model');
const { protectStaff } = require('../system/middleware/auth');

// router.use(protectStaff);

/**
 * GET  /admin/users/game-toggles
 *   → return every user with their blacklist/whitelist flags
 */
router.get('/admin/users/game-toggles',protectStaff,async (req, res) => {
  try {
    const { rows } = await pg.query(
      `SELECT
         u.id    AS uid,
         u.name  AS name,
         COALESCE(t.blacklist, FALSE)  AS blacklist,
         COALESCE(t.whitelist, FALSE)  AS whitelist
       FROM users u
  LEFT JOIN user_game_toggle t ON t.uid = u.id
     ORDER BY u.id`
    );
    res.json(rows);
  } catch (err) {
    console.error('GET all toggles error', err);
    res.status(500).json({ error: 'Internal server error' });
  }
});

/**
 * PUT /admin/user/:uid/game-toggle
 * body: { blacklist: bool, whitelist: bool }
 */
router.put('/admin/user/:uid/game-toggle',protectStaff,async (req, res) => {
  const { uid } = req.params;
  let   { blacklist = false, whitelist = false } = req.body;
  blacklist = !!blacklist;
  whitelist = !!whitelist;

  if (blacklist && whitelist) {
    return res.status(400).json({ error: 'Cannot enable BOTH flags' });
  }

  try {
    await pg.query(
      `INSERT INTO user_game_toggle(uid, blacklist, whitelist)
         VALUES ($1,$2,$3)
         ON CONFLICT (uid) DO UPDATE
           SET blacklist = EXCLUDED.blacklist,
               whitelist = EXCLUDED.whitelist`,
      [uid, blacklist, whitelist]
    );
    res.json({ uid:+uid, blacklist, whitelist });
  } catch (err) {
    console.error('PUT single toggle error', err);
    res.status(500).json({ error: 'Internal server error' });
  }
});

/**
 * PUT /admin/users/game-toggles
 * body: { action: 'blacklist-all' | 'whitelist-all' | 'disable-all' }
 */
router.put('/admin/users/game-toggles',protectStaff,async (req, res) => {
  const { action } = req.body;
  let sql;

  switch (action) {
    case 'blacklist-all':
      sql = `UPDATE user_game_toggle SET blacklist = TRUE, whitelist = FALSE`;
      break;
    case 'whitelist-all':
      sql = `UPDATE user_game_toggle SET blacklist = FALSE, whitelist = TRUE`;
      break;
    case 'disable-all':
      sql = `UPDATE user_game_toggle SET blacklist = FALSE, whitelist = FALSE`;
      break;
    default:
      return res.status(400).json({ error: 'Invalid action' });
  }

  try {
    const result = await pg.query(sql);
    res.json({ action, rowCount: result.rowCount });
  } catch (err) {
    console.error('PUT bulk toggle error', err);
    res.status(500).json({ error: 'Internal server error' });
  }
});

module.exports = router;
