
const axios       = require('axios');
const crypto      = require('crypto');
const NodeCache   = require('node-cache');
const { v4: uuid, v5: uuidv5} = require('uuid');
const db          = require('../General/Model');          // ← pg client
const Big         = require('big.js');

/* ────────── basic logger – prints ISO time + any args ─────────── */
const log = (...a) => console.log(new Date().toISOString(), ...a);
const paramsOf = req => (req.method === 'POST' ? req.body : req.query);

/* ────────── Slotegrator credentials (env in prod!) ────────────── */
const BASE  = 'https://staging.slotegrator.com/api/index.php/v1';
const M_ID  = '15e9a66a8fb380a47848ca6fc1db0081';
const M_KEY = 'b75997f9f5c4da6a9c8e76d63f086ab4a58f3d48';

/* ────────── RAM caches (games 24 h, jackpots 60 s, providers 24 h) ─ */
const cacheGames     = new NodeCache({ stdTTL: 86_400 });
const cacheJackpots  = new NodeCache({ stdTTL: 60     });
const cacheProviders = new NodeCache({ stdTTL: 86_400 });

/* ────────── helpers ────────────────────────────────────────────── */
const SUPPORTED = ['USDT','USD','INR','EUR'];
const EXT2INT   = { USD:'USDT' };
const INT2EXT   = { USDT:'USD' };

const sleep = ms => new Promise(r => setTimeout(r, ms));
const fs   = require('fs');
const path = require('path');
const winston = require('winston');

/* make sure the logs folder exists */
const logDir   = path.join(__dirname, '..', 'logs');   //  .. = project root
if (!fs.existsSync(logDir)) fs.mkdirSync(logDir, { recursive: true });

const fileLog  = winston.createLogger({
  level : 'info',
  format: winston.format.json(),
  transports: [
    new winston.transports.File({
      filename : path.join(logDir, 'gis-trx.log'), //  <── now in /logs/
      maxsize  : 1 * 1024 * 1024,   // 1 MB
      maxFiles : 3                  // keep last 3 files
    })
  ]
});

/* base headers for every call to Slotegrator */
const baseHdr = () => ({
  'X-Merchant-Id': M_ID,
  'X-Timestamp'  : Math.floor(Date.now() / 1000).toString(),
  'X-Nonce'      : crypto.randomBytes(16).toString('hex')
});

const canon = v =>
  v === null || v === undefined           ? '' :
  typeof v === 'number' || typeof v === 'bigint'
    ? Big(v).toString()                   // trims trailing zeros
    : String(v);                          // strings stay untouched

const flatten = (obj, prefix = '') => {
  const out = [];
  if (Array.isArray(obj)) {
    obj.forEach((v, i) => out.push(...flatten(v, `${prefix}[${i}]`)));
  } else if (obj && typeof obj === 'object') {
    Object.keys(obj).forEach(k =>
      out.push(...flatten(obj[k], prefix ? `${prefix}[${k}]` : k)));
  } else {
    out.push([prefix, canon(obj)]);
  }
  return out;
};

const asciiSort = ([a], [b]) => (a < b ? -1 : a > b ? 1 : 0);

// --- signature -------------------------------------------------------------
const sign = (params, hdr) => {
  const merged = {
    'X-Merchant-Id': hdr['X-Merchant-Id'],
    'X-Timestamp'  : hdr['X-Timestamp'],
    'X-Nonce'      : hdr['X-Nonce'],
    ...params
  };

  const qs = flatten(merged)
              .sort(asciiSort)            // PHP-compatible ordering
              .map(([k, v]) =>
                   `${encodeURIComponent(k)}=${encodeURIComponent(v)}`)
              .join('&');

  return crypto.createHmac('sha1', M_KEY).update(qs).digest('hex');
};

const verify = req =>
  sign(req.body, {
    'X-Merchant-Id': req.get('X-Merchant-Id'),
    'X-Timestamp'  : req.get('X-Timestamp'),
    'X-Nonce'      : req.get('X-Nonce')
  }) === req.get('X-Sign');
/* HTTP shortcuts */
/* HTTP helpers – ALWAYS 200 (Slotegrator rule) --------------------- */
const ok   = (r, o)                => r.status(200).json(o);
const err  = (r, c, m)             => ok(r, { error_code: c, error_description: m });
const created = (r,d)=>r.status(201).json(d);
const fail    = (r,s,m)=>r.status(s).json({ name:'Error', message:m, code:0, status:s });

/* common axios → response mapper */
const axiosErr = (res, e) => {
  const s = e.response?.status || 500;
  const m = e.response?.data?.message || e.message;
  log('axios-error', s, m);
  fail(res, s, m);
};

/* mini helpers */
const has = (o,f)=>o[f]!==undefined && o[f]!==null && o[f]!=='';  
const num = v=>!Number.isNaN(Number(v));
/* one constant namespace → NEVER change once in production */
const UUID_NS   = 'b58d9c74-80bb-46cb-8e0d-57458f25c23c';
const intId     = ext => ext ? uuidv5(ext, UUID_NS) : uuid();
function rspId(ext) {               // id we RETURN to provider
  return ext ? uuidv5(String(ext), UUID_NS) : uuid();
}
const b4 = n => Big(n).toFixed(4);
async function playerBalance (uid, col) {
  /* last cached value in gis_transactions first */
  const last = await db.query(
    `SELECT balance_after
       FROM gis_transactions
      WHERE user_id=$1
  ORDER BY id DESC
      LIMIT 1`, [uid]);

  if (last.rowCount) return Big(last.rows[0].balance_after);

  /* fallback – row in credits table */
  const cr = await db.query(`SELECT ${col} FROM credits WHERE uid=$1`, [uid]);
  if (!cr.rowCount) return null;          // unknown player
  return Big(cr.rows[0][col]);
}
/* ------------------------------------------------------------------ */
/* ─────────── rounding & logging helpers ─────────────────────────── */
const round4 = n => Number((Math.round(n * 1e4) / 1e4).toFixed(4));

const logTx = async (tx, bal) => {
  const txId = tx.transaction_id || uuid();           // unique
  await db.query(`
    INSERT INTO gis_transactions (
      user_id, session_id, transaction_id, action, amount, currency,
      game_uuid, type, freespin_id, quantity, round_id, finished,
      transaction_datetime, casino_request_retry_count, bet_transaction_id,
      rollback_transactions, provider_round_id, balance_after
    ) VALUES (
      $1,$2,$3,$4,$5,$6,
      $7,$8,$9,$10,$11,$12,
      $13,$14,$15,$16,$17,$18
    )
    ON CONFLICT (transaction_id) DO NOTHING
  `, [
    tx.user_id,
    tx.session_id,
    txId,
    tx.action,
    round4(tx.amount || 0),
    tx.currency,
    tx.game_uuid,
    tx.type,
    tx.freespin_id,
    tx.quantity,
    tx.round_id,
    tx.finished,
    tx.transaction_datetime ? new Date(tx.transaction_datetime) : null,
    tx.casino_request_retry_count,
    tx.bet_transaction_id,
    tx.rollback_transactions
      ? JSON.stringify(tx.rollback_transactions)
      : null,
    tx.provider_round_id,
    round4(bal)
  ]);
};

/* high-precision helpers */

const ZERO = Big(0);

/* update balance **without** rounding – DB keeps full precision     */
const setBal = (uid, ccy, val) =>
  db.query(`UPDATE credits SET ${ccy.toLowerCase()} = $1 WHERE uid = $2`,
           [val, uid]);
/* ------------------------------------------------------------------ */
/* balance helpers – always return a Big value                        */
async function fetchOrCreateBalance (playerId, col /* e.g. 'usdt' */) {
  // ① try last cached value in gis_transactions
  const last = await db.query(
    `SELECT balance_after
       FROM gis_transactions
      WHERE user_id = $1
   ORDER BY id DESC
      LIMIT 1`, [playerId]);

  if (last.rowCount) return Big(last.rows[0].balance_after);

  // ② fall back to credits table
  const cred = await db.query(
    `SELECT ${col} FROM credits WHERE uid = $1`,
    [playerId]);

  if (cred.rowCount) return Big(cred.rows[0][col]);

  // ③ first-time player – create zero row so we never get here again
  await db.query(
    `INSERT INTO credits (uid, ${col}) VALUES ($1, 0)
     ON CONFLICT (uid) DO NOTHING`,
    [playerId]);

  return Big(0);   // start with 0 balance
}

const gisOk   = (res, payload)                        => res.status(200).json(payload);
const gisFail = (res, code, description = '')        =>
  gisOk(res, { error_code: code, error_description: description }); // #1


module.exports = {

/* ═══════════════════════════════
   Public endpoints (Slotegrator)
   ═══════════════════════════════ */

/* ---------- GET /games ---------------------------------------------- */
async getGames(req,res){
  const { page=1, per_page=50 }=req.query;
  const k=`games_${page}_${per_page}`;
  if(cacheGames.has(k)) { log('cache hit /games'); return ok(res, cacheGames.get(k)); }

  try{
    const hdr = baseHdr();
    const prm = { expand:'tags,parameters,images,related_games', page, per_page };
    hdr['X-Sign'] = sign(prm, hdr);
    if(BASE.includes('staging')) await sleep(1000);
    log('GET /games', prm);

    const r = await axios.get(`${BASE}/games`, { headers:hdr, params:prm });
    const out = { items:r.data.items||r.data, _pagination:pag(r.headers) };

    cacheGames.set(k, out);
    ok(res, out);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /game-tags ------------------------------------------ */
async getGameTags(req,res){
  const { page=1, per_page=20 } = req.query;
  try{
    const hdr = baseHdr();
    const prm = { expand:'category', page, per_page };
    hdr['X-Sign'] = sign(prm, hdr);
    if(BASE.includes('staging')) await sleep(1000);
    log('GET /game-tags', prm);

    const r = await axios.get(`${BASE}/game-tags`, { headers:hdr, params:prm });
    ok(res, { items:r.data.items||r.data, _pagination:pag(r.headers) });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /games/lobby ---------------------------------------- */
async getGameLobby(req,res){
  const { game_uuid,currency,technology } = req.query;
  if(!game_uuid || !SUPPORTED.includes(currency))
    return fail(res,400,'Invalid game_uuid or currency');

  try{
    const hdr = baseHdr();
    const prm = { game_uuid, currency, ...(technology?{technology}:{}) };
    hdr['X-Sign'] = sign(prm, hdr);
    log('GET /games/lobby', prm);

    const r = await axios.get(`${BASE}/games/lobby`, { headers:hdr, params:prm });
    ok(res, r.data);
  }catch(e){ axiosErr(res,e); }
},
/* ------------------------------------------------------------------ */
/*  POST /games/init                                                  */
/* ------------------------------------------------------------------ */
async  initGame (req, res) {
  /* 1. Payload-level validation ------------------------------------ */
  const body   = { device: 'desktop', ...req.body };     // desktop is default
  const miss   = ['game_uuid', 'player_id', 'player_name', 'currency']
                   .filter(f => body[f] === undefined || body[f] === null || body[f] === '');

  if (miss.length)            return fail(res, 422, `Missing fields: ${miss.join(', ')}`);
  if (Number.isNaN(Number(body.player_id)))
                              return fail(res, 422, 'player_id must be a number');
  if (!SUPPORTED.includes(body.currency))
                              return fail(res, 400,  'Unsupported currency');
  if (body.device && !['desktop', 'mobile'].includes(body.device))
                              return fail(res, 422, 'device invalid');

  try {
    /* 2. Make sure the user exists --------------------------------- */
    const u = await db.query('SELECT id FROM users WHERE id = $1', [body.player_id]);
    if (!u.rowCount)          return fail(res, 404, 'User not found');

    /* 3. Persist session locally ----------------------------------- */
    const session_id = uuid();
    await db.query(`
      INSERT INTO gis_sessions (
        user_id, session_id, game_uuid, currency,
        device, return_url, language, lobby_data
      ) VALUES ($1,$2,$3,$4,$5,$6,$7,$8)
    `, [
      body.player_id,
      session_id,
      body.game_uuid,
      body.currency,
      body.device,
      body.return_url,
      body.language,
      body.lobby_data
    ]);

    /* 4. Build request to Slotegrator ------------------------------ */
    const hdr = {
      ...baseHdr(),
      'Content-Type': 'application/x-www-form-urlencoded'
    };

    /*  ◉  Keep scalars **raw** – NO String(player_id), NO pre-encoding  */
    const prm = {
      game_uuid   : body.game_uuid,
      player_id   : body.player_id,                   // ← number, not string
      player_name : body.player_name,
      currency    : INT2EXT[body.currency] || body.currency,
      session_id,
      device      : body.device,
      return_url  : body.return_url,
      language    : body.language,
      lobby_data  : body.lobby_data
    };

    /*  Remove empty optionals – spec wants them omitted completely    */
    Object.keys(prm).forEach(k => {
      if (prm[k] === undefined || prm[k] === null || prm[k] === '') delete prm[k];
    });

    /*  Calculate signature on the **un-encoded** payload             */
    hdr['X-Sign'] = sign(prm, hdr);

    /*  Log the exact data we’ll send                                 */
    log('POST /games/init →', prm);

    /* 5. Fire & relay ---------------------------------------------- */
    const sgRes = await axios.post(
      `${BASE}/games/init`,
      new URLSearchParams(prm),           // this performs the single encoding
      { headers: hdr }
    );

    /* Slotegrator responds with { url: "<launch-url>" }              */
    created(res, { url: sgRes.data.url, session_id });

  } catch (e) {
    axiosErr(res, e);                     // maps axios errors → fail()
  }
}


/* ---------- POST /games/init ---------------------------------------- */

,

/* ---------- POST /games/init-demo ----------------------------------- */
async initDemoGame(req,res){
  const { game_uuid } = req.body;
  if(!game_uuid) return fail(res,400,'game_uuid required');

  try{
    const prm = {
      game_uuid,
      device     : req.body.device||'desktop',
      return_url : req.body.return_url||'https://example.com',
      language   : req.body.language
    };
    const hdr = { ...baseHdr(), 'Content-Type':'application/x-www-form-urlencoded'};
    hdr['X-Sign'] = sign(prm, hdr);
    log('POST /games/init-demo', prm);

    const r = await axios.post(`${BASE}/games/init-demo`,
                               new URLSearchParams(prm), { headers:hdr });
    created(res, { url:r.data.url });
  }catch(e){ axiosErr(res,e); }
},
async handleTransaction (req, res) {
  try {
    /* ───────────── 1)  signature & params ────────────────────────── */
    log('[trx] ⇢', { ip: req.ip, params: paramsOf(req) });
    if (!verify(req))
      return gisFail(res, 'INTERNAL_ERROR', 'Invalid X-Sign');

    const p = paramsOf(req);
    const {
      action,
      player_id,
      currency,
      amount = '0',
      transaction_id: extId,             // provider’s id
      bet_transaction_id,
      rollback_transactions = [],
      round_id,
      game_uuid,
      provider_round_id
    } = p;

    if (!action || !player_id || !currency)
      return gisFail(res, 'INTERNAL_ERROR',
                     'action, player_id, currency are required');

   if (!/^\d+$/.test(String(player_id))) {
  log('[trx] non-numeric uid → unknown player', { player_id });   // console
  fileLog.info({ step: 'non-numeric-uid', player_id, action });   // file
  return gisFail(res, 'INTERNAL_ERROR', 'Player not found');
}
    if (!SUPPORTED.includes(currency))
      return gisFail(res, 'INTERNAL_ERROR', 'Unsupported currency');

const playerExists = await db.query(
  'SELECT 1 FROM credits WHERE uid::text = $1 LIMIT 1',   // ← cast to text
  [player_id]
);
if (!playerExists.rowCount)
  return gisFail(res, 'INTERNAL_ERROR', 'Player not found');


    const needAmt = ['bet', 'win', 'refund'].includes(action);
    const amtBig  = Big(amount);
    const retId   = rspId(extId);        // ← what we’ll send back

    /* ───────────── 2)  idempotency on provider id ────────────────── */
    const old = await db.query(
      `SELECT action, balance_after, rollback_transactions
         FROM gis_transactions
        WHERE transaction_id = $1
        LIMIT 1`, [extId]
    );
    if (old.rowCount) {
      const row = old.rows[0];
      const resp = {
        balance       : b4(row.balance_after),
        transaction_id: retId
      };
      if (row.action === 'rollback' && row.rollback_transactions)
        resp.rollback_transactions =
          row.rollback_transactions.map(t => t.transaction_id);
      return ok(res, resp);
    }

    /* extra: identical rollback array, new provider id -------------- */
    if (action === 'rollback') {
      const dup = await db.query(
        `SELECT balance_after
           FROM gis_transactions
          WHERE action='rollback'
            AND rollback_transactions::text = $1
          LIMIT 1`,
        [JSON.stringify(rollback_transactions)]
      );
      if (dup.rowCount)
        return ok(res, {
          balance             : b4(dup.rows[0].balance_after),
          transaction_id      : retId,
          rollback_transactions: rollback_transactions.map(t => t.transaction_id)
        });
    }

    /* extra: refund repeated by bet_transaction_id ------------------ */
    // if (action === 'refund') {
    //   const dup = await db.query(
    //     `SELECT balance_after
    //        FROM gis_transactions
    //       WHERE action='refund'
    //         AND bet_transaction_id=$1
    //       LIMIT 1`, [bet_transaction_id]);
    //   if (dup.rowCount)
    //     return ok(res, {
    //       balance       : b4(dup.rows[0].balance_after),
    //       transaction_id: retId
    //     });
    // }
if (action === 'refund') {
  const dup = await db.query(
    `SELECT balance_after, transaction_id          -- need provider id
       FROM gis_transactions
      WHERE action = 'refund'
        AND bet_transaction_id = $1
      LIMIT 1`,
    [bet_transaction_id]
  );
  if (dup.rowCount) {
    /* reproduce the exact id we sent in the first refund response */
    const oldRetId = rspId(dup.rows[0].transaction_id);   // uuid-v5 of FIRST ext-id
    return ok(res, {
      balance       : b4(dup.rows[0].balance_after),
      transaction_id: oldRetId
    });
  }
}
    /* ───────────── 3)  current balance or fail on unknown user ───── */
    const col   = (EXT2INT[currency] ?? currency).toLowerCase();
    const last  = await db.query(
      `SELECT balance_after
         FROM gis_transactions
        WHERE user_id = $1
     ORDER BY id DESC LIMIT 1`, [player_id]);

    let bal;
    if (last.rowCount) {
      bal = Big(last.rows[0].balance_after);
    }  else {
  /* first: does the uid exist at all?  (avoids selecting missing column) */
  const exists = await db.query(
    'SELECT 1 FROM credits WHERE uid=$1 LIMIT 1', [player_id]);
  if (!exists.rowCount)
    return gisFail(res, 'INTERNAL_ERROR', 'Player not found');

  /* uid exists → now read the specific currency column */
  const cr = await db.query(
    `SELECT ${col} FROM credits WHERE uid=$1`, [player_id]);
  bal = Big(cr.rows[0][col]);
}
    /*  balance probes with amount must error on insufficient funds    */
  /* balance probe – only error when an amount is supplied AND insufficient */
/* ───────── balance probe ────────────────────────────────────────── */
if (action === 'balance') {

  /* 1️⃣  unknown-player check (covers both numeric & hex ids) */
  if (!playerExists /* ← we created this a few lines earlier */) {
    fileLog.info({ step: 'balance-probe-unknown-player', player_id });
    return gisFail(res, 'INTERNAL_ERROR', 'Player not found');
  }

  /* 2️⃣  insufficient funds check – only when an amount is supplied   */
  if (p.hasOwnProperty('amount') &&
      amtBig.gt(0) &&
      bal.lt(amtBig)) {
    fileLog.info({
      step     : 'balance-probe-insufficient',
      player_id: player_id,
      balance  : bal.toString(),
      requested: amount
    });
    return gisFail(res, 'INSUFFICIENT_FUNDS',
                   'Not enough money to continue playing');
  }

  /* 3️⃣  simple balance probe → return current balance and exit       */
  fileLog.info({
    step     : 'balance-probe-ok',
    player_id: player_id,
    balance  : bal.toString()
  });
  return ok(res, { balance: b4(bal) });
}



    if (needAmt && amtBig.lt(0))
      return gisFail(res, 'INTERNAL_ERROR', 'amount must be non-negative');

    /* ───────────── 4)  business logic → delta ---------------------- */
    let delta = ZERO;

    switch (action) {
      case 'bet':
        if (bal.lt(amtBig))
          return gisFail(res, 'INSUFFICIENT_FUNDS',
                         'Not enough money to continue playing');
        delta = amtBig.neg();                      break;

      case 'win':     delta = amtBig;             break;

      case 'refund': {
        const bet = await db.query(
          `SELECT 1 FROM gis_transactions
            WHERE transaction_id=$1 AND action='bet'`,
          [bet_transaction_id]);
        if (bet.rowCount) delta = amtBig;
        break;
      }

      case 'rollback':
        for (const tx of rollback_transactions) {
          const a = Big(tx.amount || 0);
          if (a.lte(0)) continue;
          if (tx.action === 'bet')    delta = delta.plus(a);
          if (tx.action === 'win')    delta = delta.minus(a);
          if (tx.action === 'refund') delta = delta.minus(a);
        }
        break;

      case 'balance': /* probe – no change */    break;

      default:
        return gisFail(res, 'INTERNAL_ERROR', 'Invalid action');
    }

    /* ───────────── 5)  apply & persist ----------------------------- */
    if (!delta.eq(0)) {
      bal = bal.plus(delta);
      await setBal(player_id, col, bal.toString());
    }

    await logTx({
      ...p,
      user_id       : player_id,
      transaction_id: extId,              // keep provider’s id in DB
      amount        : amtBig,
      balance_after : bal.toString(),
      rollback_transactions
    }, bal);

    /* ───────────── 6)  build response ------------------------------ */
    if (action === 'balance')
      return ok(res, { balance: b4(bal) });

    if (action === 'rollback')
      return ok(res, {
        balance             : b4(bal),
        transaction_id      : retId,
        rollback_transactions: rollback_transactions.map(t => t.transaction_id)
      });

    /* bet | win | refund */
    return ok(res, { balance: b4(bal), transaction_id: retId });

  } catch (err) {
    console.error('handleTransaction()', err);
    return gisFail(res, 'INTERNAL_ERROR', 'Unhandled internal error');
  }
},
/* ---------- POST /callback/transactions & POST / -------------------- */

/* ---------- GET /limits & /limits/freespin --------------------------- */
async getLimits(_q,res){ try{
  const hdr = baseHdr(); hdr['X-Sign']=sign({},hdr);
  log('GET /limits');
  const r = await axios.get(`${BASE}/limits`,{ headers:hdr });
  ok(res,r.data);
}catch(e){axiosErr(res,e);}},

async getFreespinLimits(_q,res){ try{
  const hdr = baseHdr(); hdr['X-Sign']=sign({},hdr);
  log('GET /limits/freespin');
  const r = await axios.get(`${BASE}/limits/freespin`,{ headers:hdr });
  ok(res,r.data);
}catch(e){axiosErr(res,e);}},

/* ---------- GET /jackpots ------------------------------------------- */
async getJackpots(_q,res){
  if(cacheJackpots.has('j')) return ok(res,cacheJackpots.get('j'));
  try{
    const hdr=baseHdr(); hdr['X-Sign']=sign({},hdr);
    log('GET /jackpots');
    const r=await axios.get(`${BASE}/jackpots`,{ headers:hdr });
    cacheJackpots.set('j',r.data); ok(res,r.data);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /freespins/bets ------------------------------------- */
async getFreespinBets(req,res){
  const { game_uuid, currency } = req.query;
  if(!game_uuid || !['USDT','INR'].includes(currency))
    return fail(res,400,'Invalid params');

  try{
    const hdr = baseHdr();
    const prm = { game_uuid, currency };
    hdr['X-Sign'] = sign(prm,hdr);
    log('GET /freespins/bets', prm);

    const r = await axios.get(`${BASE}/freespins/bets`,{ headers:hdr, params:prm });
    ok(res,r.data);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- POST /freespins/set ------------------------------------- */
async setFreespin(req,res){
  const b=req.body;
  const miss=['player_id','player_name','currency','quantity','valid_from',
              'valid_until','freespin_id','game_uuid']
              .filter(f=>!has(b,f));
  if(miss.length) return fail(res,422,`Missing: ${miss.join(',')}`);
  if(!['USDT','INR'].includes(b.currency)) return fail(res,400,'currency must be USDT/INR');
  if(!( (b.bet_id && b.denomination) || b.total_bet_id))
    return fail(res,422,'Provide (bet_id+denomination) or total_bet_id');

  try{
    const u=await db.query('SELECT id FROM users WHERE id=$1',[b.player_id]);
    if(!u.rowCount) return fail(res,404,'User not found');

    await db.query(`
      INSERT INTO gis_freespins(
        user_id,freespin_id,game_uuid,currency,quantity,quantity_left,
        valid_from,valid_until,bet_id,total_bet_id,denomination,status)
      VALUES($1,$2,$3,$4,$5,$5,to_timestamp($6),to_timestamp($7),
             $8,$9,$10,'active')
      ON CONFLICT (freespin_id) DO NOTHING`,
      [b.player_id,b.freespin_id,b.game_uuid,b.currency,b.quantity,
       b.valid_from,b.valid_until,b.bet_id,b.total_bet_id,b.denomination]);

    const hdr={ ...baseHdr(),'Content-Type':'application/x-www-form-urlencoded'};
    hdr['X-Sign']=sign(b,hdr);
    log('POST /freespins/set', b);
    await axios.post(`${BASE}/freespins/set`, new URLSearchParams(b), { headers:hdr });

    created(res,{ message:'Freespin campaign set' });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /freespins/get -------------------------------------- */
async getFreespin(req,res){
  const { freespin_id } = req.query;
  if(!freespin_id) return fail(res,400,'freespin_id required');

  try{
   const loc = await db.query(`
    SELECT 
        id,
        user_id,
        freespin_id,
        game_uuid,
        currency,
        quantity,
        quantity_left,
        valid_from,
        valid_until,
        bet_id,
        total_bet_id,
        denomination,
        status,
        is_canceled,
        total_win,
        created_at
    FROM gis_freespins 
    WHERE freespin_id = $1
`, [freespin_id]);
    if(loc.rowCount) return ok(res,loc.rows[0]);

    const hdr=baseHdr(), prm={ freespin_id }; hdr['X-Sign']=sign(prm,hdr);
    log('GET /freespins/get', prm);
    const r=await axios.get(`${BASE}/freespins/get`,{ headers:hdr, params:prm });
    ok(res,r.data);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- POST /freespins/cancel ---------------------------------- */
async cancelFreespin(req,res){
  const { freespin_id }=req.body;
  if(!freespin_id) return fail(res,400,'freespin_id required');

  try{
    await db.query(`UPDATE gis_freespins SET status='canceled',is_canceled=1
                    WHERE freespin_id=$1`,[freespin_id]);
    const hdr={ ...baseHdr(),'Content-Type':'application/x-www-form-urlencoded'};
    hdr['X-Sign']=sign({ freespin_id },hdr);
    log('POST /freespins/cancel', { freespin_id });
    await axios.post(`${BASE}/freespins/cancel`,
                     new URLSearchParams({ freespin_id }),{ headers:hdr });
    ok(res,{ message:'Campaign canceled' });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- POST /freevouchers/set ---------------------------------- */
async setFreevoucher(req,res){
  const b=req.body;
  const need=['player_id','title','currency','initial_balance','max_winnings',
              'valid_until','voucher_id','table_ids'];
  const miss=need.filter(f=>!has(b,f));
  if(miss.length) return fail(res,422,`Missing: ${miss.join(',')}`);
  if(!Array.isArray(b.table_ids)||!b.table_ids.length)
    return fail(res,422,'table_ids must be array w/ at least 1 id');
  if(!['USDT','INR'].includes(b.currency)) return fail(res,400,'currency must be USDT/INR');

  try{
    const u=await db.query('SELECT id FROM users WHERE id=$1',[b.player_id]);
    if(!u.rowCount) return fail(res,404,'User not found');

    await db.query(`
      INSERT INTO gis_freevouchers(
        user_id,voucher_id,title,currency,initial_balance,max_winnings,
        valid_until,table_ids,short_terms,terms_and_conds,state,playable)
      VALUES($1,$2,$3,$4,$5,$6,to_timestamp($7),
             $8,$9,$10,'Active',$5)
      ON CONFLICT (voucher_id) DO NOTHING`,
      [b.player_id,b.voucher_id,b.title,b.currency,b.initial_balance,
       b.max_winnings,b.valid_until,b.table_ids,b.short_terms,b.terms_and_conds]);

    const hdr={ ...baseHdr(),'Content-Type':'application/x-www-form-urlencoded'};
    hdr['X-Sign']=sign(b,hdr);
    log('POST /freevouchers/set', b);
    await axios.post(`${BASE}/freevouchers/set`, new URLSearchParams(b), { headers:hdr });

    created(res,{ message:'Freevoucher campaign set' });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /freevouchers/get ----------------------------------- */
async getFreevoucher(req,res){
  const { voucher_id }=req.query;
  if(!voucher_id) return fail(res,400,'voucher_id required');

  try{
   const loc = await db.query(`
    SELECT 
        id,
        user_id,
        voucher_id,
        title,
        currency,
        initial_balance,
        max_winnings,
        valid_from,
        valid_until,
        table_ids,
        short_terms,
        terms_and_conds,
        state,
        playable,
        winnings,
        created_at
    FROM gis_freevouchers 
    WHERE voucher_id = $1
`, [voucher_id]);
    if(loc.rowCount) return ok(res,loc.rows[0]);

    const hdr=baseHdr(), prm={ voucher_id }; hdr['X-Sign']=sign(prm,hdr);
    log('GET /freevouchers/get', prm);
    const r=await axios.get(`${BASE}/freevouchers/get`,{ headers:hdr, params:prm });
    ok(res,r.data);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- POST /freevouchers/cancel ------------------------------- */
async cancelFreevoucher(req,res){
  const { voucher_id, reason }=req.body;
  if(!voucher_id||!['Canceled','Forfeited'].includes(reason))
    return fail(res,400,'voucher_id and valid reason required');

  try{
    await db.query(`UPDATE gis_freevouchers SET state=$1 WHERE voucher_id=$2`,
                   [reason,voucher_id]);
    const hdr={ ...baseHdr(),'Content-Type':'application/x-www-form-urlencoded'};
    const prm={ voucher_id,reason }; hdr['X-Sign']=sign(prm,hdr);
    log('POST /freevouchers/cancel', prm);
    await axios.post(`${BASE}/freevouchers/cancel`,
                     new URLSearchParams(prm),{ headers:hdr });
    ok(res,{ message:'Voucher canceled' });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- POST /self-validate ------------------------------------- */
async selfValidate(_req,res){
  try{
    const hdr={ ...baseHdr(),'Content-Type':'application/x-www-form-urlencoded'};
    hdr['X-Sign']=sign({},hdr);
    log('POST /self-validate');
    const r=await axios.post(`${BASE}/self-validate`,{}, { headers:hdr });
    ok(res,r.data);
  }catch(e){ axiosErr(res,e); }
},

/* ═══════════════════════════════
   Local helper endpoints (DB)
   ═══════════════════════════════ */

/* ---------- GET /sync – refresh gis_games table --------------------- */
async syncGames(_q,res){
  try{
    log('syncGames – start');
    await db.query('TRUNCATE TABLE gis_games RESTART IDENTITY');
    let page=1,total=0,skipped=[];
    for(;;){
      const hdr=baseHdr();
      const prm={ expand:'', per_page:50, page };
      hdr['X-Sign']=sign(prm,hdr);
      if(BASE.includes('staging')) await sleep(1000);
      const r=await axios.get(`${BASE}/games/index`,{ headers:hdr, params:prm });
      const games=r.data.items||r.data;

      for(const g of games){
        const up=parseInt(g.updated_at);
        if(Number.isNaN(up)){ skipped.push(g.uuid); continue; }

        await db.query(`
          INSERT INTO gis_games(
            uuid,name,provider,type,image,technology,has_lobby,is_mobile,
            has_freespins,freespin_valid_until_full_day,
            updated_at,api_sub_provider_id,updated_at_timestamp)
          VALUES($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,to_timestamp($11))
          ON CONFLICT (uuid) DO UPDATE SET
            name=EXCLUDED.name,provider=EXCLUDED.provider,type=EXCLUDED.type,
            image=EXCLUDED.image,technology=EXCLUDED.technology,
            has_lobby=EXCLUDED.has_lobby,is_mobile=EXCLUDED.is_mobile,
            has_freespins=EXCLUDED.has_freespins,
            freespin_valid_until_full_day=EXCLUDED.freespin_valid_until_full_day,
            updated_at=EXCLUDED.updated_at,
            api_sub_provider_id=EXCLUDED.api_sub_provider_id,
            updated_at_timestamp=EXCLUDED.updated_at_timestamp`,
          [g.uuid,g.name,g.provider,g.type,g.image,g.technology,
           g.has_lobby===1,g.is_mobile===1,g.has_freespins===1,
           g.freespin_valid_until_full_day===1,up,
           g.api_sub_provider_id?parseInt(g.api_sub_provider_id):null]);
        total++;
      }
      if(page >= (r.data._meta?.pageCount||1)) break;
      page++;
    }
    ok(res,{ message:`Synced ${total} games`, skipped });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /providersgis -------------------------------------- */
async getProvidersgis(req,res){
  const search=(req.query.search||'').trim().toLowerCase();
  try{
    const rows = search ?
    (await db.query(`
        SELECT 
            name
        FROM gis_providers 
        WHERE LOWER(name) LIKE $1 
        ORDER BY name
    `, [`%${search}%`])).rows :
    (await db.query(`
        SELECT 
            name
        FROM gis_providers 
        ORDER BY name
    `)).rows;
    ok(res,{ success:true, data:rows, total:rows.length });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /gamesgis (+ filters) ------------------------------- */
async getGamesgis(req,res){
  const page  = +req.query.page  || 1;
  const limit = Math.min(+req.query.limit || 10, 100);

  const f = {                                          // all filters in one obj
    provider     : req.query.provider,
    type         : req.query.type,
    search       : req.query.search,
    technology   : req.query.technology,
    has_lobby    : req.query.has_lobby==='true'?true:req.query.has_lobby==='false'?false:null,
    is_mobile    : req.query.is_mobile==='true'?true:req.query.is_mobile==='false'?false:null,
    has_freespins: req.query.has_freespins==='true'?true:req.query.has_freespins==='false'?false:null
  };

  const where=[],args=[], add=(sql,val)=>{ args.push(val); where.push(sql.replace('?',`$${args.length}`)); };
  if(f.provider)     add('LOWER(provider)=LOWER(?)',f.provider);
  if(f.type)         add('LOWER(type)=LOWER(?)',f.type);
  if(f.search)       add('LOWER(name) LIKE LOWER(?)',`%${f.search}%`);
  if(f.technology)   add('LOWER(technology)=LOWER(?)',f.technology);
  if(f.has_lobby!==null)    add('has_lobby=?',f.has_lobby);
  if(f.is_mobile!==null)    add('is_mobile=?',f.is_mobile);
  if(f.has_freespins!==null)add('has_freespins=?',f.has_freespins);
  const whereSQL = where.length ? 'WHERE '+where.join(' AND ') : '';

  try{
    const total = +(await db.query(`SELECT COUNT(*) FROM gis_games ${whereSQL}`,args)).rows[0].count;
    args.push(limit,(page-1)*limit);
    const data  = (await db.query(`
      SELECT uuid,name,provider,type,image,technology,has_lobby,is_mobile,
             has_freespins,freespin_valid_until_full_day,api_sub_provider_id,
             created_at,updated_at
      FROM gis_games ${whereSQL}
      ORDER BY name ASC LIMIT $${args.length-1} OFFSET $${args.length}`, args)).rows;

    ok(res,{
      success:true,
      data,
      pagination:{
        currentPage:page,
        totalPages :Math.ceil(total/limit),
        totalItems :total,
        itemsPerPage:limit
      },
      filters:f
    });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /gamesgis/provider/:provider ------------------------ */
async getGamesByProvidergis(req,res){
  req.query.provider = req.params.provider;
  return this.getGamesgis(req,res);
},

/* ---------- GET /gamesgis/stats ------------------------------------- */
async getGameStats(_q,res){
  try{
    const stats = (await db.query(`
      SELECT COUNT(*) AS total_games,
             COUNT(DISTINCT provider) AS total_providers,
             COUNT(DISTINCT type)     AS total_types,
             COUNT(*) FILTER (WHERE has_lobby)        AS games_with_lobby,
             COUNT(*) FILTER (WHERE is_mobile)        AS mobile_games,
             COUNT(*) FILTER (WHERE has_freespins)    AS games_with_freespins
      FROM gis_games`)).rows[0];

    const topProviders = (await db.query(`
      SELECT provider,COUNT(*) AS game_count
      FROM gis_games
      GROUP BY provider
      ORDER BY game_count DESC LIMIT 10`)).rows;

    const gameTypes = (await db.query(`
      SELECT type,COUNT(*) AS game_count
      FROM gis_games
      GROUP BY type
      ORDER BY game_count DESC`)).rows;

    ok(res,{ success:true, stats, topProviders, gameTypes });
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /games/provider?provider=X -------------------------- */
async getGamesByProvider(req,res){
  const { provider, page=1, per_page=50 } = req.query;
  if(!provider) return fail(res,400,'provider query-param required');

  const key=`prov_${provider}_${page}_${per_page}`;
  if(cacheGames.has(key)) return ok(res, cacheGames.get(key));

  try{
    const hdr=baseHdr();
    const prm={ expand:'tags,parameters,images,related_games', page, per_page };
    hdr['X-Sign']=sign(prm,hdr);
    log('GET /games/provider', prm);

    const r=await axios.get(`${BASE}/games/index`,{ headers:hdr, params:prm });
    const items=(r.data.items||r.data)
                 .filter(g => g.provider.toLowerCase()===provider.toLowerCase());

    if(!items.length) return fail(res,404,`No games for provider ${provider}`);

    const out={ items, _pagination:pag(r.headers) };
    cacheGames.set(key,out);
    ok(res,out);
  }catch(e){ axiosErr(res,e); }
},

/* ---------- GET /providers ----------------------------------------- */
async getProviders(_req,res){
  if(cacheProviders.has('list')) return ok(res,{ providers:cacheProviders.get('list') });
  try{
    let page=1, more=true, names=new Set();
    while(more){
      const hdr=baseHdr();
      const prm={ expand:'', per_page:50, page };
      hdr['X-Sign']=sign(prm,hdr);
      const r=await axios.get(`${BASE}/games/index`,{ headers:hdr, params:prm });
      (r.data.items||r.data).forEach(g=>g.provider && names.add(g.provider));
      more = page < (r.data._meta?.pageCount || 1);
      page++;
      if(BASE.includes('staging')) await sleep(500);
    }
    const list=[...names].sort();
    cacheProviders.set('list',list);
    ok(res,{ providers:list });
  }catch(e){ axiosErr(res,e); }
}

}; // ───── end module.exports ─────
