const crypto = require('crypto');
const pg = require('../General/Model');
const GameResponseHandler = require('./gameResposneHandler');
const CryptoJS = require('crypto-js');
const NodeCache = require('node-cache');
const { check, query, validationResult } = require('express-validator');

const cache = new NodeCache({ 
    stdTTL: 300,  // 5 minutes default TTL
    checkperiod: 60 // check for expired keys every minute
});

class GameController {
    // Configuration constants
    static AGENCY_CONFIG = {
        agency_uid: 'b96581ad0785ff9f86c960def63aee4b',
        aes_key: '8ce9295ab6786ef6e4bd8d07eda4ce81',
        player_prefix: 'h24e9e',
        server_url: 'https://huidu.bet'
    };


    // AES Encryption Utility
    static encryptPayload(payload, key) {
        try {
            const keyBuffer = Buffer.from(key, 'utf8');
            const cipher = crypto.createCipheriv('aes-256-ecb', keyBuffer, Buffer.alloc(0));
            let encrypted = cipher.update(JSON.stringify(payload), 'utf8', 'base64');
            encrypted += cipher.final('base64');
            return encrypted;
        } catch (error) {
            console.error('Encryption error:', error);
            throw new Error('Payload encryption failed');
        }
    }
  
    static decryptPayload(encryptedPayload, key) {
        try {
            // Convert key to WordArray for CryptoJS compatibility
            const keyWordArray = CryptoJS.enc.Utf8.parse(key);

            // Decrypt using CryptoJS with ECB mode and PKCS7 padding
            const decryptedWordArray = CryptoJS.AES.decrypt(
                encryptedPayload, 
                keyWordArray, 
                {
                    mode: CryptoJS.mode.ECB,
                    padding: CryptoJS.pad.Pkcs7
                }
            );

            // Convert decrypted WordArray to UTF-8 string
            const decryptedString = decryptedWordArray.toString(CryptoJS.enc.Utf8);

            // Parse the decrypted JSON
            return JSON.parse(decryptedString);
        } catch (error) {
            console.error('Decryption error:', error);
            throw new Error('Payload decryption failed');
        }
    }


    // 1. Get Game Launch URL (SEAMLESS)
    async getGameLaunchURL(req, res) {
        const { 
            game_uid, 
            user_id, 
            credit_amount = '50', 
            currency_code, 
            language = 'en', 
            home_url = 'https://api.boss707.com', 
            callback_url = 'https://api.boss707.com/jsGames/game/bet-callback' 
        } = req.body;

        // Validate currency
        const allowedCurrencies = ['INR', 'BDT', 'USDT'];
        if (!allowedCurrencies.includes(currency_code)) {
            return res.status(400).json(
                GameResponseHandler.handleErrorResponse(10009, 'Invalid currency. Allowed currencies are INR, BDT, USDT')
            );
        }

        try {
            // Validate game exists
          const gameQuery = `
    SELECT 
        id,
        game_name,
        game_uid,
        game_type,
        game_icon,
        is_active,
        created_at,
        updated_at,
        vendor
    FROM js_games 
    WHERE game_uid = $1 AND is_active = TRUE
`;
            const gameResult = await pg.query(gameQuery, [game_uid]);
            
            if (gameResult.rows.length === 0) {
                return res.status(404).json(
                    GameResponseHandler.handleErrorResponse(10008)
                );
            }

            // Prepare payload
            const payload = {
                agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                member_account: `${GameController.AGENCY_CONFIG.player_prefix}_${currency_code}_${user_id}`,
                game_uid,
                timestamp: Date.now().toString(),
                credit_amount,
                currency_code,
                language,
                home_url,
                callback_url
            };

            // Encrypt payload
            const encryptedPayload = GameController.encryptPayload(payload, GameController.AGENCY_CONFIG.aes_key);

            // Make API call to get game launch URL
            const apiResponse = await fetch(`${GameController.AGENCY_CONFIG.server_url}/game/v1`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                    timestamp: Date.now().toString(),
                    payload: encryptedPayload
                })
            });

            const responseData = await apiResponse.json();

            // Process game server response
            const processedResponse = GameResponseHandler.processGameServerResponse(responseData);

            // If not successful, return early
            if (!processedResponse.success) {
                return res.status(400).json(processedResponse);
            }

            // Create game session
            const sessionQuery = `
                INSERT INTO js_game_sessions 
                (user_id, game_uid, session_token, launch_url) 
                VALUES ($1, $2, $3, $4) 
                RETURNING *
            `;
            const sessionToken = crypto.randomBytes(16).toString('hex');
            await pg.query(sessionQuery, [
                user_id, 
                game_uid, 
                sessionToken,
                processedResponse.payload.game_launch_url
            ]);

            // Return standardized success response
            res.json(
                GameResponseHandler.handleSuccessResponse(responseData, {
                    game_launch_url: processedResponse.payload.game_launch_url,
                    session_token: sessionToken
                })
            );
        } catch (error) {
            // Log and handle unexpected errors
            GameResponseHandler.logErrorDetails('getGameLaunchURL', error, { game_uid, user_id });
            res.status(500).json(
                GameResponseHandler.handleErrorResponse(10004)
            );
        }
    }

    // 2. Retrieve Bet Information (SEAMLESS) - Callback
   
    // 3. Game Transfer (Deposit/Withdrawal)
   
    async processBetCallback(req, res) {
        const { payload, agency_uid, timestamp } = req.body;
    
        try {
            // Decrypt payload
            const decryptedData = GameController.decryptPayload(
                payload, 
                GameController.AGENCY_CONFIG.aes_key
            );
            console.log('Decrypted ',decryptedData);
            // Extract user ID from member account
            const userId = decryptedData.member_account.split('_')[2];
            let transactionType;
            if (parseFloat(decryptedData.bet_amount) > 0 && parseFloat(decryptedData.win_amount) === 0) {
                // Bet placement
                transactionType = 'bet';
               
            } else if (parseFloat(decryptedData.bet_amount) === 0 && parseFloat(decryptedData.win_amount) > 0) {
                // Win
                transactionType = 'win';
               
            } else if (parseFloat(decryptedData.bet_amount) == 0 && parseFloat(decryptedData.win_amount) == 0) {
                // Loss (bet amount deducted, win amount is less than bet)
                transactionType = 'loss';
                
            } 
            // Begin transaction
            await pg.query('BEGIN');
    
            try {
                // Determine transaction type and amount
                const betAmount = parseFloat(decryptedData.bet_amount);
                const winAmount = parseFloat(decryptedData.win_amount);
                const currency = decryptedData.currency_code.toLowerCase();
    
                // Insert transaction record
                const transactionQuery = `
                    INSERT INTO js_game_transactions 
                    (user_id, game_uid, transaction_type, amount, currency, 
                     transaction_status,external_transaction_id,serial_number, additional_data)
                    VALUES ($1, $2, $3, $4, $5, $6, $7, $8,$9)
                    RETURNING *
                `;
                
               
                if (parseFloat(decryptedData.bet_amount) > 0 && parseFloat(decryptedData.win_amount) === 0) {
                    // Bet placement
                    transactionType = 'bet';
                    const transactionResult = await pg.query(transactionQuery, [
                        userId,
                        decryptedData.game_uid,
                        transactionType,
                        betAmount,
                        currency,
                        'processed',
                        decryptedData.game_round,
                        decryptedData.serial_number,
                        JSON.stringify(decryptedData)
                    ]);
                } else if (parseFloat(decryptedData.bet_amount) === 0 && parseFloat(decryptedData.win_amount) > 0) {
                    // Win
                    transactionType = 'win';
                    const transactionResult = await pg.query(transactionQuery, [
                        userId,
                        decryptedData.game_uid,
                        transactionType,
                        winAmount,
                        currency,
                        'processed',
                        decryptedData.game_round,
                        decryptedData.serial_number,
                        JSON.stringify(decryptedData)
                    ]);
                } else if (parseFloat(decryptedData.bet_amount) == 0 && parseFloat(decryptedData.win_amount) == 0) {
                    // Loss (bet amount deducted, win amount is less than bet)
                    transactionType = 'loss';
                    const transactionResult = await pg.query(transactionQuery, [
                        userId,
                        decryptedData.game_uid,
                        transactionType,
                        winAmount,
                        currency,
                        'processed',
                        decryptedData.game_round,
                        decryptedData.serial_number,
                        JSON.stringify(decryptedData)
                    ]);
                } 
                // Update user credits
                const updateCreditQuery = `
                    UPDATE credits 
                    SET ${currency} = ${currency} - $1 + $2
                    WHERE uid = $3
                    RETURNING ${currency} as new_balance
                `;
    
                const creditResult = await pg.query(updateCreditQuery, [
                    betAmount, 
                    winAmount, 
                    userId
                ]);
    
                // Commit the transaction
                await pg.query('COMMIT');
    
                // Prepare response payload
                const responsePayload = {
                    credit_amount: creditResult.rows[0].new_balance.toString(),
                    timestamp: Date.now().toString()
                };
    
                // Encrypt response payload
                const encryptedResponsePayload = GameController.encryptPayload(
                    responsePayload, 
                    GameController.AGENCY_CONFIG.aes_key
                );
    
                // Respond with standardized success
                res.json({
                    code: 0,
                    msg: "",
                    payload: encryptedResponsePayload
                });
            } catch (error) {
                // Rollback the transaction in case of error
                await pg.query('ROLLBACK');
                throw error;
            }
        } catch (error) {
            // Log and handle unexpected errors
            GameResponseHandler.logErrorDetails('processBetCallback', error);
            res.status(500).json(
                GameResponseHandler.handleErrorResponse(1, 'Bet processing failed')
            );
        }
    }
  
    async processGameTransfer(req, res) {
        const { user_id, game_uid, credit_amount, transfer_type,currency_code } = req.body;

        try {
            // Prepare transfer payload
            const payload = {
                agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                member_account: `${GameController.AGENCY_CONFIG.player_prefix}_${currency_code}_${user_id}`,
                game_uid,
                timestamp: Date.now().toString(),
                credit_amount: transfer_type === 'deposit' ? credit_amount : `-${credit_amount}`,
                transfer_id: crypto.randomBytes(8).toString('hex')
            };

            // Encrypt payload
            const encryptedPayload = GameController.encryptPayload(payload, GameController.AGENCY_CONFIG.aes_key);

            // Make API call to process transfer
            const apiResponse = await fetch(`${GameController.AGENCY_CONFIG.server_url}/game/v2`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                    timestamp: Date.now().toString(),
                    payload: encryptedPayload
                })
            });

            const responseData = await apiResponse.json();

            // Process game server response
            const processedResponse = GameResponseHandler.processGameServerResponse(responseData);

            // Record transfer transaction
            const transactionQuery = `
                INSERT INTO js_game_transactions 
                (user_id, game_uid, transaction_type, amount, currency, 
                 transaction_status, external_transaction_id)
                VALUES ($1, $2, $3, $4, $5, $6, $7)
                RETURNING *
            `;

            await pg.query(transactionQuery, [
                user_id,
                game_uid,
                transfer_type,
                credit_amount,
                'USD',
                processedResponse.success ? 'completed' : 'failed',
                payload.transfer_id
            ]);

            // Return processed response
            res.json(processedResponse);
        } catch (error) {
            // Log and handle unexpected errors
            GameResponseHandler.logErrorDetails('processGameTransfer', error, { user_id, game_uid });
            res.status(500).json(
                GameResponseHandler.handleErrorResponse(10004, 'Transfer processing failed')
            );
        }
    }

    // 4. Get Transaction Records
    async getTransactionRecords(req, res) {
        const { 
            from_date, 
            to_date, 
            page_no = 1, 
            page_size = 30 
        } = req.body;

        try {
            // Prepare payload for API
            const payload = {
                agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                timestamp: Date.now().toString(),
                from_date: new Date(from_date).getTime(),
                to_date: new Date(to_date).getTime(),
                page_no,
                page_size
            };

            // Encrypt payload
            const encryptedPayload = GameController.encryptPayload(payload, GameController.AGENCY_CONFIG.aes_key);

            // Make API call to get transaction records
            const apiResponse = await fetch(`${GameController.AGENCY_CONFIG.server_url}/game/transaction/list`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    agency_uid: GameController.AGENCY_CONFIG.agency_uid,
                    timestamp: Date.now().toString(),
                    payload: encryptedPayload
                })
            });

            const responseData = await apiResponse.json();

            // Process game server response
            const processedResponse = GameResponseHandler.processGameServerResponse(responseData);

            // Store transaction records if successful
            if (processedResponse.success && processedResponse.payload.records) {
                const storeTransactionsQuery = `
                    INSERT INTO js_game_transactions 
                    (user_id, game_uid, transaction_type, amount, currency, 
                     transaction_status, external_transaction_id, serial_number, additional_data)
                    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
                    ON CONFLICT (external_transaction_id) DO NOTHING
                `;
                for (const record of processedResponse.payload.records) {
                    await pg.query(storeTransactionsQuery, [
                        record.member_account.split('_')[2],
                        record.game_uid,
                        'transaction',
                        record.transfer_amount,
                        record.currency,
                        record.transfer_status === 1 ? 'completed' : 'failed',
                        record.transaction_id,
                        record.serial_number,
                        JSON.stringify(record)
                    ]);
                }
            }

            // Return processed response
            res.json(processedResponse);
        } catch (error) {
            // Log and handle unexpected errors
            GameResponseHandler.logErrorDetails('getTransactionRecords', error);
            res.status(500).json(
                GameResponseHandler.handleErrorResponse(10004, 'Error fetching transaction records')
            );
        }
    }

    /**
 * Get active games with pagination, optional vendor filtering, and prioritized games
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 */
async getActiveGames(req, res) {
    try {
        // Validate request parameters
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(422).json(
                GameResponseHandler.handleErrorResponse(10002, 'Validation failed', errors.array())
            );
        }

        // Extract parameters with defaults
        const { 
            vendor,
            page = 1, 
            per_page = 20
        } = req.query;

        // Validate pagination parameters
        const pageNum = Math.max(1, parseInt(page));
        const perPageNum = Math.min(100, Math.max(1, parseInt(per_page)));
        const offset = (pageNum - 1) * perPageNum;

        // Generate cache key
        const cacheKey = `active_games_${vendor || 'all'}_${pageNum}_${perPageNum}`;
        
        // Check cache
        const cachedResult = cache.get(cacheKey);
        if (cachedResult) {
            return res.json(
                GameResponseHandler.handleSuccessResponse({
                    payload: cachedResult,
                    meta: { cached: true }
                })
            );
        }

        // Count all active games for the vendor (or all vendors)
        let countQuery = 'SELECT COUNT(*) as count FROM js_games WHERE is_active = TRUE';
        let countParams = [];

        if (vendor) {
            countQuery += ' AND vendor = $1';
            countParams.push(vendor);
        }

        // Execute count query to get total
        const countResult = await pg.query(countQuery, countParams);
        const total = parseInt(countResult.rows[0].count);
        const totalPages = Math.ceil(total / perPageNum);

        // Variables to be set based on prioritization logic
        let games = [];
        let prioritizedGameIds = [];

        // Check if we need to prioritize games for the requested vendor
        if (vendor) {
            // Fetch prioritized games for this vendor from the database
            const prioritizedQuery = 'SELECT game_ids FROM prioritized_games WHERE vendor = $1';
            const prioritizedResult = await pg.query(prioritizedQuery, [vendor]);
            
            if (prioritizedResult.rows.length > 0) {
                // Convert comma-separated list to array of numbers
                prioritizedGameIds = prioritizedResult.rows[0].game_ids.split(',').map(id => parseInt(id.trim()));
            }
        }
        
        // Determine how to fetch and paginate data based on page number and prioritized games
        if (vendor && prioritizedGameIds.length > 0) {
            // If we're on page 1 and have prioritized games for this vendor, handle differently
            if (pageNum === 1) {
                // First, fetch the prioritized games that exist and are active
                const prioritizedQuery = `
                    SELECT   id,
        game_name,
        game_uid,
        game_type,
        game_icon,
        is_active,
        created_at,
        updated_at,
        vendor
        FROM js_games 
                    WHERE is_active = TRUE 
                    AND vendor = $1 
                    AND id IN (${prioritizedGameIds.map((_, i) => `$${i + 2}`).join(',')})
                    ORDER BY CASE ${prioritizedGameIds.map((id, i) => `WHEN id = $${i + 2} THEN ${i}`).join(' ')} ELSE ${prioritizedGameIds.length} END
                `;
                
                const prioritizedParams = [vendor, ...prioritizedGameIds];
                const prioritizedResult = await pg.query(prioritizedQuery, prioritizedParams);
                const prioritizedGames = prioritizedResult.rows;

                // Calculate how many additional regular games we need
                const remainingSlots = perPageNum - prioritizedGames.length;

                if (remainingSlots > 0) {
                    // Fetch additional games, excluding the prioritized ones
                    const regularQuery = `
                        SELECT   id,
        game_name,
        game_uid,
        game_type,
        game_icon,
        is_active,
        created_at,
        updated_at,
        vendor FROM js_games 
                        WHERE is_active = TRUE 
                        AND vendor = $1 
                        AND id NOT IN (${prioritizedGameIds.map((_, i) => `$${i + 2}`).join(',')})
                        ORDER BY id
                        LIMIT $${prioritizedGameIds.length + 2}
                    `;
                    
                    const regularParams = [vendor, ...prioritizedGameIds, remainingSlots];
                    const regularResult = await pg.query(regularQuery, regularParams);
                    
                    // Combine prioritized games with regular games
                    games = [...prioritizedGames, ...regularResult.rows];
                } else {
                    // If we have enough or more prioritized games than per_page, just use those
                    games = prioritizedGames.slice(0, perPageNum);
                }
            } else {
                // For pages after the first, we need to skip prioritized games and adjust pagination
                // Get the IDs of all prioritized games that exist in the database
                const existingPrioritizedQuery = `
                    SELECT id FROM js_games 
                    WHERE is_active = TRUE 
                    AND vendor = $1 
                    AND id IN (${prioritizedGameIds.map((_, i) => `$${i + 2}`).join(',')})
                `;
                
                const existingPrioritizedParams = [vendor, ...prioritizedGameIds];
                const existingPrioritizedResult = await pg.query(existingPrioritizedQuery, existingPrioritizedParams);
                const existingPrioritizedIds = existingPrioritizedResult.rows.map(row => row.id);

                // Calculate adjusted offset - we need to skip all prioritized games plus regular pagination
                const prioritizedCount = existingPrioritizedIds.length;
                const adjustedOffset = offset + prioritizedCount - perPageNum;
                
                // Fetch paginated games excluding the prioritized ones
                const paginatedQuery = `
                    SELECT   id,
        game_name,
        game_uid,
        game_type,
        game_icon,
        is_active,
        created_at,
        updated_at,
        vendor FROM js_games 
                    WHERE is_active = TRUE 
                    AND vendor = $1 
                    AND id NOT IN (${existingPrioritizedIds.map((_, i) => `$${i + 2}`).join(',')})
                    ORDER BY id
                    LIMIT $${existingPrioritizedIds.length + 2} OFFSET $${existingPrioritizedIds.length + 3}
                `;
                
                const paginatedParams = [vendor, ...existingPrioritizedIds, perPageNum, Math.max(0, adjustedOffset)];
                const paginatedResult = await pg.query(paginatedQuery, paginatedParams);
                
                games = paginatedResult.rows;
            }
        } else {
            // Standard query without prioritization
          let dataQuery = `
    SELECT 
        id,
        game_name,
        game_uid,
        game_type,
        game_icon,
        is_active,
        created_at,
        updated_at,
        vendor
    FROM js_games 
    WHERE is_active = TRUE
`;
            let queryParams = [];

            // If vendor is provided, add it to the query
            if (vendor) {
                dataQuery += ' AND vendor = $1';
                queryParams.push(vendor);
            }

            // Add pagination to data query
            dataQuery += ' ORDER BY id LIMIT $' + (queryParams.length + 1) + ' OFFSET $' + (queryParams.length + 2);
            queryParams.push(perPageNum, offset);

            // Execute data query
            const dataResult = await pg.query(dataQuery, queryParams);
            games = dataResult.rows;
        }

        // Prepare response
        const response = {
            games: games,
            pagination: {
                current_page: pageNum,
                per_page: perPageNum,
                total: total,
                total_pages: totalPages
            }
        };

        // Cache the results
        cache.set(cacheKey, response);

        // Send response
        return res.json(
            GameResponseHandler.handleSuccessResponse({
                payload: response,
                meta: { cached: false }
            })
        );
    } catch (error) {
        GameResponseHandler.logErrorDetails('getActiveGames', error);
        res.status(500).json(
            GameResponseHandler.handleErrorResponse(10004, 'Error fetching game list')
        );
    }
}

    /**
     * Search games with relevance scoring and pagination
     * @param {Object} req - Express request object
     * @param {Object} res - Express response object
     */
    async searchGames(req, res) {
        try {
            // Extract and validate parameters
            let {
                keyword = '',
                page = 1,
                per_page = 20,
                vendor = null
            } = req.query;
    
            // Clean and prepare the search keyword
            keyword = keyword.trim().toLowerCase();
            
            // Validate pagination parameters
            const pageNum = Math.max(1, parseInt(page));
            const perPageNum = Math.min(50, Math.max(1, parseInt(per_page)));
            const offset = (pageNum - 1) * perPageNum;
    
            // Generate cache key
            const cacheKey = `games_search_${keyword}_${vendor || 'all'}_${pageNum}_${perPageNum}`;
            
            // Check cache
            const cachedResult = cache.get(cacheKey);
            if (cachedResult) {
                return res.json(
                    GameResponseHandler.handleSuccessResponse({
                        payload: cachedResult,
                        meta: { cached: true }
                    })
                );
            }
    
            // Base conditions
            let conditions = ['is_active = TRUE'];
            let params = [];
            let paramIndex = 1;
    
            // Add vendor filter if provided
            if (vendor) {
                conditions.push(`vendor = $${paramIndex}`);
                params.push(vendor);
                paramIndex++;
            }
    
            // Add search conditions if keyword provided
            if (keyword) {
                conditions.push(`(
                    game_name ILIKE $${paramIndex} OR 
                    vendor ILIKE $${paramIndex} OR 
                    game_type ILIKE $${paramIndex}
                )`);
                params.push(`%${keyword}%`);
                paramIndex++;
            }
    
            // Build where clause
            const whereClause = conditions.length ? 'WHERE ' + conditions.join(' AND ') : '';
    
            // Count query
            const countQuery = `
                SELECT COUNT(*) as count 
                FROM js_games 
                ${whereClause}
            `;
    
            // Search query with relevance scoring
            const searchQuery = `
                WITH search_results AS (
                    SELECT 
                        *,
                        CASE
                            WHEN game_name ILIKE $${paramIndex} THEN 100  -- Exact start match (game_name)
                            WHEN game_name ILIKE $${paramIndex+1} THEN 80  -- Contains match (game_name)
                            WHEN vendor ILIKE $${paramIndex} THEN 70   -- Exact start match (vendor)
                            WHEN vendor ILIKE $${paramIndex+1} THEN 60  -- Contains match (vendor)
                            WHEN game_type ILIKE $${paramIndex} THEN 50     -- Exact start match (game_type)
                            WHEN game_type ILIKE $${paramIndex+1} THEN 40    -- Contains match (game_type)
                            ELSE 0
                        END as relevance_score
                    FROM js_games
                    ${whereClause}
                )
                SELECT * FROM search_results
                ${keyword ? 'WHERE relevance_score > 0' : ''}
                ORDER BY ${keyword ? 'relevance_score DESC, ' : ''} id ASC
                LIMIT $${paramIndex+2} OFFSET $${paramIndex+3}
            `;
    
            // Add search pattern parameters if keyword is provided
            if (keyword) {
                params.push(`${keyword}%`);  // Exact start match
                params.push(`%${keyword}%`); // Contains match
            } else {
                // If no keyword, add placeholder parameters to maintain indexing
                params.push('');
                params.push('');
            }
    
            // Add pagination parameters
            params.push(perPageNum, offset);
    
            // Execute queries in parallel
            const [countResult, searchResult] = await Promise.all([
                pg.query(countQuery, params.slice(0, paramIndex-1)),  // Only use the necessary params
                pg.query(searchQuery, params)
            ]);
    
            // Calculate pagination metadata
            const total = parseInt(countResult.rows[0].count);
            const totalPages = Math.ceil(total / perPageNum);
    
            // Prepare response
            const response = {
                games: searchResult.rows,
                pagination: {
                    current_page: pageNum,
                    per_page: perPageNum,
                    total: total,
                    total_pages: totalPages
                },
                meta: {
                    query: keyword,
                    searchFields: ['game_name', 'vendor', 'game_type']
                }
            };
    
            // Cache the results
            cache.set(cacheKey, response);
    
            // Send response
            res.json(
                GameResponseHandler.handleSuccessResponse({
                    payload: response,
                    meta: { cached: false }
                })
            );
        } catch (error) {
            GameResponseHandler.logErrorDetails('searchGames', error);
            res.status(500).json(
                GameResponseHandler.handleErrorResponse(10005, 'Error searching games')
            );
        }
    }
    /**
     * Get validation rules for pagination endpoint
     * @returns {Array} Array of validation rules
     */
    static getActiveGamesValidationRules() {
        return [
            query('vendor').optional(),
            query('page').optional().isInt({ min: 1 }).withMessage('Page must be a positive integer'),
            query('per_page').optional().isInt({ min: 1, max: 100 }).withMessage('Per page must be between 1 and 100')
        ];
    }

    /**
     * Get validation rules for search endpoint
     * @returns {Array} Array of validation rules
     */
    static getSearchValidationRules() {
        return [
            query('keyword').optional(),
            query('vendor').optional(),
            query('page').optional().isInt({ min: 1 }).withMessage('Page must be a positive integer'),
            query('per_page').optional().isInt({ min: 1, max: 50 }).withMessage('Per page must be between 1 and 50')
        ];
    }

   
}

module.exports = new GameController();