// middlewares/hierarchy.js
const pg = require('../../General/Model');

/** ------------------------------------------------------------------
 *  Build an array with every user-id that is inside the caller’s
 *  down-line (recursive CTE –  ≈ 5 ms for 50 k users)
 * ----------------------------------------------------------------- */

/* ────────────────────────────────────────────────────────────────────
 *  ① check a single USER-ID
 *     (= for approve / reject / list filtering)
 *  ② exported helper requireStaffForUsername()
 * ──────────────────────────────────────────────────────────────────── */
async function flatDownline(staffId) {
  /* 1) staff descendants (includes self) ------------------------- */
  const { rows: staffRows } = await pg.query(
    `SELECT descendant_id
       FROM staff_hierarchy
      WHERE ancestor_id = $1`,
    [staffId]
  );
  const staffIds = staffRows.map(r => r.descendant_id);
  if (!staffIds.includes(staffId)) staffIds.push(staffId); // safety

  /* 2) every player under *any* of those staff ------------------- */
  const { rows: users } = await pg.query(
    `SELECT id
       FROM users
      WHERE parent_staff_id = ANY($1)`,
    [staffIds]
  );

  return users.map(r => r.id);
}

/* ------------------------------------------------------------------ */
exports.assertInHierarchy = async (staffId, userId) => {
  if (!staffId) throw new Error('staffId missing');
  const allowed = await flatDownline(staffId);
  if (!allowed.includes(userId)) throw new Error('outside hierarchy');
};
/* middleware: make sure :username or body.username is inside tree */
exports.requireStaffForUsername = () => async (req,res,next) => {
  if (!req.user?.is_staff)
    return res.status(403).json({error:'staff only'});

  const username =
        req.params.username || req.body.username || req.query.username;
  if (!username)
    return res.status(400).json({error:'username required'});

  // self is always ok
  if (username === req.user.username) return next();

  const { rows:[u] } = await pg.query(
    'SELECT id FROM users WHERE username = $1', [username]);
  if (!u) return res.status(404).json({error:'user not found'});

  try { await exports.assertInHierarchy(req.user.id, u.id); }
  catch { return res.status(403).json({error:'outside hierarchy'}); }

  next();
};
