const cron = require('node-cron');
const axios = require('axios');

const pg = require("./General/Model");

const logger = {
  info: (message) => console.log(`[INFO] ${message}`),
  error: (message, error) => console.error(`[ERROR] ${message}`, error)
};


async function fetchCryptoRates() {
  try {
    const response = await axios.get(
      'https://api.coingecko.com/api/v3/simple/price', {
        params: {
          ids: 'bitcoin,ethereum,litecoin,bitcoin-cash,tether,tron,dogecoin,cardano,ripple,binancecoin,shiba-inu,usd-coin,binance-usd,matic-network,maker,nexo,siacoin,bitbop,tether-usd',
          vs_currencies: 'usd'
        }
      }
    );

    const idToCurrencyMap = {
      'bitcoin': 'btc',
      'ethereum': 'eth',
      'litecoin': 'ltc',
      'bitcoin-cash': 'bch',
      'tether': 'usdt',
      'tron': 'trx',
      'dogecoin': 'doge',
      'cardano': 'ada',
      'ripple': 'xrp',
      'binancecoin': 'bnb',
      'shiba-inu': 'shib',
      'usd-coin': 'usdc',
      'binance-usd': 'busd',
      'matic-network': 'matic',
      'maker': 'mkr',
      'nexo': 'nexo',
      'siacoin': 'sc',
      'bitbop': 'sbc',
      'tether-usd': 'usdp',
      'true-usd': 'tusd'
    };

    const rates = {};

    for (const [id, value] of Object.entries(response.data)) {
      const currency = idToCurrencyMap[id];
      if (currency && value.usd) {
        rates[currency] = value.usd;
      }
    }

    // Set default values for stablecoins if they're missing or significantly off $1
    ['usdt', 'usdc', 'busd', 'usdp', 'tusd'].forEach(stablecoin => {
      if (!rates[stablecoin] || Math.abs(rates[stablecoin] - 1) > 0.05) {
        rates[stablecoin] = 1.0;
      }
    });

    return rates;
  } catch (error) {
    logger.error('Error fetching crypto rates:', error);
    throw error;
  }
}

async function fetchFiatRates() {
  try {
    const response = await axios.get(
      'https://open.er-api.com/v6/latest/USD'
    );
    
    const rates = {};
    
    if (response.data && response.data.rates) {
      // Extract fiat currencies from your database
      rates['inr'] = response.data.rates.INR ? (1 / response.data.rates.INR) : 0.01;
      rates['npr'] = response.data.rates.NPR ? (1 / response.data.rates.NPR) : 0.0075;
      rates['aed'] = response.data.rates.AED ? (1 / response.data.rates.AED) : 0.27;
      rates['mvr'] = response.data.rates.MVR ? (1 / response.data.rates.MVR) : 0.05;
      rates['pkr'] = response.data.rates.PKR ? (1 / response.data.rates.PKR) : 0.0035;
      
      // Add other fiat currencies if needed
    }
    
    return rates;
  } catch (error) {
    logger.error('Error fetching fiat rates:', error);
    throw error;
  }
}

async function updateExchangeRates() {
  try {
    logger.info('Starting exchange rate update job');
    
    const cryptoRates = await fetchCryptoRates();
    const fiatRates = await fetchFiatRates();
    
    const allRates = { ...cryptoRates, ...fiatRates };
    
    // Fetch all exchange rates from the database
    const { rows: dbRates } = await pg.query(`
    SELECT 
        id,
        currency,
        usd_rate,
        last_updated
    FROM exchangerate
`);
    
    const dbCurrencies = dbRates.map(rate => rate.currency.toLowerCase());
    
    // Update existing rates
    for (const rate of dbRates) {
      const currency = rate.currency.toLowerCase();
      
      if (allRates[currency]) {
        const newRate = parseFloat(allRates[currency].toFixed(8));
        
        if (parseFloat(rate.usd_rate) !== newRate) {
          await pg.query(
            'UPDATE exchangerate SET usd_rate = $1, last_updated = $2 WHERE currency = $3',
            [newRate, new Date(), rate.currency]
          );
          logger.info(`Updated ${rate.currency}: ${newRate}`);
        }
      }
    }
    
    // Insert any missing currencies from our fetched rates
    for (const [currency, rate] of Object.entries(allRates)) {
      if (!dbCurrencies.includes(currency.toLowerCase())) {
        const newRate = parseFloat(rate.toFixed(8));
        await pg.query(
          'INSERT INTO exchangerate (currency, usd_rate, last_updated) VALUES ($1, $2, $3)',
          [currency.toUpperCase(), newRate, new Date()]
        );
        logger.info(`Added new currency ${currency.toUpperCase()}: ${newRate}`);
      }
    }
    
    logger.info('Exchange rate update job completed successfully');
  } catch (error) {
    logger.error('Error updating exchange rates:', error);
  }
}

function initExchangeRateUpdateJob() {
  // Schedule to run every 10 minutes
  cron.schedule('*/10 * * * *', async () => {
    try {
      await updateExchangeRates();
    } catch (error) {
      logger.error('Scheduled exchange rate update failed:', error);
    }
  });
  
  logger.info('Exchange rate update job scheduled (running every 10 minutes)');
  
  // Run immediately on startup
  updateExchangeRates().catch(err => {
    logger.error('Initial exchange rate update failed:', err);
  });
}

module.exports = {
  initExchangeRateUpdateJob,
  updateExchangeRates
};