const GameService = require('./service');
const { validationResult, check } = require('express-validator');

// Initialize game service
const gameService = new GameService();

// Game controller with pagination methods
class GameController {
  /**
   * Get games by vendor with pagination
   * @param {Object} req - Express request object
   * @param {Object} res - Express response object
   */
  async getGamesByVendor(req, res) {
    try {
      // Validate request parameters
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(422).json({
          success: false,
          message: 'Validation failed',
          errors: errors.array()
        });
      }

      // Extract validated parameters
      const { vendor, page, per_page } = req.query;

      // Get paginated games data
      const result = await gameService.getGamesByVendorPaginated(
        vendor,
        parseInt(page),
        parseInt(per_page)
      );

      // Return formatted response
      return res.status(200).json({
        success: true,
        data: result.data,
        pagination: {
          current_page: result.currentPage,
          per_page: result.perPage,
          total: result.total,
          total_pages: result.totalPages
        }
      });
    } catch (error) {
      console.error('Error in getGamesByVendor:', error);
      return res.status(500).json({
        success: false,
        message: 'Failed to fetch games',
        error: error.message
      });
    }
  }

/**
 * Get all vendors with their game counts
 * Can filter by specific vendor if provided and calculate pages if pageSize provided
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 */
async getVendors(req, res) {
    try {
      const { vendor, pageSize } = req.query;
      const pageSizeNumber = pageSize ? parseInt(pageSize, 10) : null;
      
      // Validate pageSize if provided
      if (pageSize !== undefined && (isNaN(pageSizeNumber) || pageSizeNumber <= 0)) {
        return res.status(422).json({
          success: false,
          message: 'Validation failed',
          errors: [{ param: 'pageSize', msg: 'Page size must be a positive integer' }]
        });
      }
      
      // Get vendors with game counts (filtered by vendor name if provided)
      const vendors = await gameService.getVendorsWithGameCount(vendor, pageSizeNumber);
      
      // Return successful response
      return res.status(200).json({
        success: true,
        data: vendors
      });
    } catch (error) {
      console.error('Error in getVendors:', error);
      return res.status(500).json({
        success: false,
        message: 'Failed to fetch vendors',
        error: error.message
      });
    }
  }

  /**
   * Get validation rules for pagination endpoint
   * @returns {Array} Array of validation rules
   */
  static getValidationRules() {
    return [
      check('vendor').notEmpty().withMessage('Vendor is required'),
      check('page').isInt({ min: 1 }).withMessage('Page must be a positive integer'),
      check('per_page').isInt({ min: 1, max: 100 }).withMessage('Per page must be between 1 and 100')
    ];
  }
}

module.exports = new GameController();