const db  = require('../General/Model');
const NodeCache = require('node-cache');

// Initialize cache with TTL defaults
const cache = new NodeCache({ 
  stdTTL: 600, // 10 minutes default TTL
  checkperiod: 120 // check for expired keys every 2 minutes
});

class GameService {
/**
 * Get paginated games by vendor
 * @param {string} vendor - Vendor name
 * @param {number} page - Page number
 * @param {number} perPage - Items per page
 * @returns {Object} - Paginated results with metadata
 */
async getGamesByVendorPaginated(vendor, page, perPage) {
    try {
      // Calculate offset for pagination
      const offset = (page - 1) * perPage;
  
      // Get total count (with caching for performance)
      const cacheKey = `games_count_vendor_${vendor}`;
      let total = cache.get(cacheKey);
  
      if (total === undefined) {
        const totalResult = await db.query(
          'SELECT COUNT(*) as count FROM apigames WHERE vendor = $1 AND enabled = 1',
          [vendor]
        );
        
        // In PostgreSQL, count returns a string, so we parse it
        total = parseInt(totalResult.rows[0].count, 10);
        cache.set(cacheKey, total);
      }
  
      // Calculate total pages
      const totalPages = Math.ceil(total / perPage);
  
      // Get paginated results
      const gamesResult = await db.query(
        'SELECT id, title, platform, type, subtype, vendor, details_description_en, details_thumbnails_300x300 ' +
        'FROM apigames WHERE vendor = $1 AND enabled = 1 ORDER BY id LIMIT $2 OFFSET $3',
        [vendor, perPage, offset]
      );
  
      const games = gamesResult.rows || [];
  
      // Return formatted result
      return {
        data: games,
        currentPage: page,
        perPage: perPage,
        total: total,
        totalPages: totalPages
      };
    } catch (error) {
      console.error('Error in getGamesByVendorPaginated:', error);
      throw new Error(`Failed to get games: ${error.message}`);
    }
  }

/**
 * Get vendors with their game counts and optional page count calculation
 * Can filter by specific vendor if vendorName is provided
 * @param {string} [vendorName] - Optional vendor name to filter
 * @param {number} [pageSize] - Optional page size for calculating total pages
 * @returns {Array} - Array of vendor objects with counts and optional page info
 */
async getVendorsWithGameCount(vendorName = null, pageSize = null) {
    try {
      // Create cache key based on parameters
      const pageParam = pageSize ? `_pageSize${pageSize}` : '';
      const cacheKey = vendorName 
        ? `vendor_game_count_${vendorName}${pageParam}` 
        : `vendors_with_game_count${pageParam}`;
      
      let vendors = cache.get(cacheKey);
      
      if (vendors === undefined) {
        let query, result;
        
        // For PostgreSQL, we need to use $1, $2, etc. for parameterized queries
        if (vendorName && vendorName !== 'all') {
          query = 'SELECT vendor, COUNT(*) as game_count FROM apigames WHERE enabled = 1 AND vendor = $1 GROUP BY vendor ORDER BY vendor';
          result = await db.query(query, [vendorName]);
        } else {
          query = 'SELECT vendor, COUNT(*) as game_count FROM apigames WHERE enabled = 1 GROUP BY vendor ORDER BY vendor';
          result = await db.query(query);
        }
        
        vendors = result.rows || []; // For PostgreSQL client, results are in the rows property
        
        // Calculate total pages if pageSize is provided
        if (pageSize && pageSize > 0) {
          vendors = vendors.map(vendor => {
            const gameCount = parseInt(vendor.game_count, 10);
            const totalPages = Math.ceil(gameCount / pageSize);
            return {
              ...vendor,
              total_pages: totalPages,
              page_size: pageSize
            };
          });
        }
        
        cache.set(cacheKey, vendors, 1800); // Cache for 30 minutes
      }
      
      return vendors;
    } catch (error) {
      console.error('Error in getVendorsWithGameCount:', error);
      throw new Error(`Failed to get vendors: ${error.message}`);
    }
  }
}

module.exports = GameService;