import React, { useState, useEffect, useRef } from 'react';
import styled from 'styled-components';
import { isMobile } from '../../../Helper';

// Styled Components
const Container = styled.div`
  width: 100%;
  padding: 0.5rem 0rem;
  @media (max-width: 768px) {
    margin-top: -20px;
  }
`;

const Header = styled.div`
  display: flex;
  align-items: center;
  margin-left: 16px;
`;

const Title = styled.h2`
  color: white;
  font-size: 16px;
  font-weight: bold;
  display: flex;
  align-items: center;
`;

const GreenDot = styled.div`
  width: 0.5rem;
  height: 0.5rem;
  background-color: #f09c06;
  border-radius: 50%;
  margin-right: 0.5rem;
  box-shadow: 0 0 0 0 rgba(36, 238, 137, 0.4);
  animation: pulse 0.9s infinite;

  @keyframes pulse {
    0% {
      box-shadow: 0 0 0 0 rgba(36, 238, 137, 0.4);
    }
    70% {
      box-shadow: 0 0 0 10px rgba(36, 238, 137, 0);
    }
    100% {
      box-shadow: 0 0 0 0 rgba(36, 238, 137, 0);
    }
  }
`;

const SliderContainerOuter = styled.div`
  background: #03030F;
  padding: 0 12px;
  border-radius: 12px;
`;

const SliderContainer = styled.div`
  position: relative;
  overflow: hidden;
  background: #03030F;
  padding: 12px;
  border-radius: 12px;
  contain: content;
  min-height: 100px;
  width: 100%; /* Ensure full width */
`;

const MarqueeContainer = styled.div`
  width: 100%;
  overflow: hidden;
`;

const MarqueeContent = styled.div`
  display: flex;
  gap: 0.5rem; /* Keeping original gap */
  white-space: nowrap;
  animation: marquee ${props => props.duration}s linear infinite; /* Much slower duration */
  will-change: transform;

  @keyframes marquee {
    0% {
      transform: translateX(0);
    }
    100% {
      transform: translateX(-${props => props.cardWidth * props.cardCount}px); /* Full sequence width */
    }
  }

  @media (max-width: 768px) {
    animation: marquee ${props => props.mobileDuration}s linear infinite; /* Much slower mobile duration */
  }

  &:hover {
    animation-play-state: paused; /* Ensure slide stops on hover */
  }
`;

const GameCard = styled.div`
  flex-shrink: 0;
  border-radius: 10px;
  overflow: hidden;
  margin: 0 2px;
  display: inline-block;
`;

const ImageContainer = styled.div`
  position: relative;
  height: 5rem;
  width: 3.5rem;
  border-radius: 10px;
  
  img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: 10px;
  }
`;

const UserContainer = styled.div`
  text-align: center;
  margin-top: 4px;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
`;

const UserInfo = styled.div`
  display: flex;
  align-items: center;
  gap: 4px;
  justify-content: center;
`;

const UserBadge = styled.img`
  height: 15px;
  width: 15px;
  object-fit: contain;
`;

const Username = styled.div`
  font-size: 9px;
`;

const ProfitAmount = styled.div`
  color: #f09c06;
  font-size: 9px;
`;

// Utility Functions and Config
const cryptoConfig = {
  USDT: {
    icon: '/assets/images/USDT.png',
    minBet: 50000,
    maxBet: 500000,
    decimals: 0
  }
};

const formatProfit = (amount) => {
  if (amount >= 1000) {
    return `$${(amount / 1000).toFixed(1)}k`;
  }
  return `$${amount.toFixed(0)}`;
};

const generateRandomProfit = (min, max, decimals) => {
  const factor = 10 ** decimals;
  return ((Math.random() * (max - min) + min) * factor).toFixed(decimals) / factor;
};

const shuffleArray = (array) => {
  const shuffled = [...array];
  for (let i = shuffled.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]];
  }
  return shuffled;
};

// Main Component
const LatestWins = React.memo(() => {
  const marqueeRef = useRef(null);

  const gameSample = [
    { id: 1, game_icon: '/assets/images/bigwins/pgsoft/Alchemy Gold.avif' },
    { id: 2, game_icon: '/assets/images/bigwins/pgsoft/Asgardian Rising.avif' },
    { id: 3, game_icon: '/assets/images/bigwins/evolution/Bac Bo Ao Vivo.webp' },
    { id: 4, game_icon: '/assets/images/bigwins/evolution/Baccarat A.webp' },
    { id: 5, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 28.webp' },
    { id: 6, game_icon: '/assets/images/bigwins/ezugi/Andar Bahar.webp' },
    { id: 7, game_icon: '/assets/images/bigwins/ezugi/Bet on Teen Patti.webp' },
    { id: 8, game_icon: '/assets/images/bigwins/pgsoft/Butterfly Blossom.avif' },
    { id: 9, game_icon: '/assets/images/bigwins/pragmatic/5 Lions Megaways.avif' },
    { id: 10, game_icon: '/assets/images/bigwins/pgsoft/Battleground Royale.avif' },
    { id: 11, game_icon: '/assets/images/bigwins/pgsoft/Baccarat Deluxe.avif' },
    { id: 12, game_icon: '/assets/images/bigwins/ezugi/Casino Marina Andar Bahar.webp' },
    { id: 13, game_icon: '/assets/images/bigwins/ezugi/Casino Marina Roulette 1.webp' },
    { id: 14, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 29.webp' },
    { id: 15, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 30.webp' },
    { id: 16, game_icon: '/assets/images/bigwins/pgsoft/Bali Vacation.avif' },
    { id: 17, game_icon: '/assets/images/bigwins/pgsoft/Battleground Royale.avif' },
    { id: 18, game_icon: '/assets/images/bigwins/pgsoft/Bikini Paradise.avif' },
    { id: 19, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 32.webp' },
    { id: 20, game_icon: '/assets/images/bigwins/pragmatic/Big Bass Splash.avif' },
    { id: 21, game_icon: '/assets/images/bigwins/pragmatic/Big Juan.avif' },
    { id: 22, game_icon: '/assets/images/bigwins/pragmatic/Bigger Bass Bonanza.avif' },
    { id: 23, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 32.webp' },
    { id: 24, game_icon: '/assets/images/bigwins/evolution/Blackjack VIP 32.webp' },
    { id: 25, game_icon: '/assets/images/bigwins/ezugi/EZ Dealer Roulette Thai.webp' },
    { id: 26, game_icon: '/assets/images/bigwins/ezugi/Fortune Baccarat.webp' },
    { id: 27, game_icon: '/assets/images/bigwins/ezugi/Gold Blackjack 1.webp' },
    { id: 28, game_icon: '/assets/images/bigwins/ezugi/Gold Blackjack 3.webp' },
    { id: 29, game_icon: '/assets/images/bigwins/pgsoft/Candy Bonanza.avif' },
    { id: 30, game_icon: '/assets/images/bigwins/pgsoft/Candy Burst.avif' },
    { id: 31, game_icon: '/assets/images/bigwins/pgsoft/Captain_s Bounty.avif' },
    { id: 32, game_icon: '/assets/images/bigwins/pragmatic/Big Juan.avif' },
    { id: 33, game_icon: '/assets/images/bigwins/pragmatic/Black Bull.avif' },
  ];

  const processedGames = gameSample.map((game) => {
    const currency = 'USDT';
    const { minBet, maxBet, decimals } = cryptoConfig[currency];
    const profitAmount = generateRandomProfit(minBet, maxBet, decimals);
    const profit = formatProfit(profitAmount);
    return {
      ...game,
      profit,
      username: 'Hidden'
    };
  });

  // Duplicate the array for seamless looping
  const seamlessGames = [...processedGames, ...processedGames];

  // Calculate card width (3.5rem + 0.5rem gap, converted to px)
  const cardWidth = 56 + 8; // 3.5rem ≈ 56px, 0.5rem ≈ 8px
  const cardCount = processedGames.length; // 33 cards

  useEffect(() => {
    const marquee = marqueeRef.current;
    if (marquee) {
      marquee.style.animation = 'none';
      marquee.offsetHeight; // Trigger reflow
      marquee.style.animation = isMobile()
        ? 'marquee 35s linear infinite' // Much slower mobile duration
        : 'marquee 40s linear infinite'; // Much slower desktop duration
    }

    return () => {
      if (marquee) marquee.style.animation = 'none';
    };
  }, []);

  return (
    <Container>
      <Header>
        <Title>
          <GreenDot />
          Recent Big Wins
        </Title>
      </Header>
      <SliderContainerOuter>
        <SliderContainer>
          <MarqueeContainer>
            <MarqueeContent
              ref={marqueeRef}
              id={`marquee-1`}
              cardWidth={cardWidth}
              cardCount={cardCount}
              duration={40} // Much slower desktop duration
              mobileDuration={35} // Much slower mobile duration
            >
              {seamlessGames.length > 0 ? (
                seamlessGames.map((game, index) => (
                  <GameCard key={`${game.id}-${index}`}>
                    <ImageContainer>
                      <img
                        src={game.game_icon || '/assets/images/placeholder.png'}
                        alt={game.game_name || 'Game'}
                        onError={(e) => (e.target.src = '/assets/images/placeholder.png')}
                      />
                    </ImageContainer>
                    <UserContainer>
                      <UserInfo>
                        <UserBadge
                          src='/assets/images/bcgames/bannertags/tags.png'
                          alt="VIP badge"
                        />
                        <Username>{game.username}</Username>
                      </UserInfo>
                      <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', gap: '4px', marginLeft: '4px', width: '100%' }}>
                        <img
                          src="/assets/images/USDT.png"
                          style={{ height: '10px', width: '10px', objectFit: 'contain' }}
                          alt="USDT"
                        />
                        <ProfitAmount>{game.profit}</ProfitAmount>
                      </div>
                    </UserContainer>
                  </GameCard>
                ))
              ) : (
                <div style={{ color: 'white', display: 'inline-block' }}>No games available</div>
              )}
            </MarqueeContent>
          </MarqueeContainer>
        </SliderContainer>
      </SliderContainerOuter>
    </Container>
  );
});

export default LatestWins;