import { useState, useRef, useEffect } from "react"
import styled from "styled-components"
import { Circle, Loader2 } from 'lucide-react'
import NotificationsSkyla from "../../../Icons/Notifications"
import { getUID } from "../../../Helper"
import { notificationService } from "../../../utils/Services"

const Container = styled.div`
  position: relative;
  display: inline-block;
`

const IconWrapper = styled.div`
  cursor: pointer;
  border-radius: 50%;
  transition: all 0.3s ease;
  // padding: 8px;

  &:hover {
    background-color: rgba(115, 198, 196, 0.15);
    transform: scale(1.05);
  }
`
const UnreadBadge = styled.div`
  position: absolute;
  top: -8px;
  right: -8px;
  min-width: 20px;
  height: 20px;
  background-color: #FFD700; // Yellow background
  color: #FF0000; // Red text
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  font-size: 12px;
  font-weight: bold;
  padding: 0 4px;
  box-sizing: border-box;
`;

const DropdownMenu = styled.div`
  position: absolute;
  right: 0;
  margin-top: 12px;
  width: 360px;
//  background: rgb(30, 35, 40);
background:rgb(30, 35, 40);
   border-radius: 16px;
  box-shadow: 0 12px 28px rgba(0, 0, 0, 0.2);
  overflow: hidden;
  z-index: 50;
  max-height: 85vh;
  display: flex;
  flex-direction: column;
  backdrop-filter: blur(8px);
  border: 1px solid rgba(255, 255, 255, 0.1);
`

const Header = styled.div`
  display: flex;
  align-items: center;
  justify-content: space-between;
  padding: 20px;
  background: rgba(0, 0, 0, 0.1);
  border-bottom: 1px solid rgba(255, 255, 255, 0.1);
`

const Title = styled.div`
  font-size: 20px;
  font-weight: 700;
  color: #ffffff;
  letter-spacing: 0.5px;
`

const MarkAllButton = styled.div`
  font-size: 13px;
  font-weight: 600;
  color: #ffffff;
  background: #f09c06;
  padding: 8px 16px;
  border-radius: 24px;
  transition: all 0.2s ease;
  cursor: pointer;

  &:hover {
    background: rgba(255, 255, 255, 0.25);
    transform: translateY(-1px);
  }

  &:active {
    transform: translateY(0);
  }
`

const TabContainer = styled.div`
  display: flex;
  padding: 8px;
  background: rgba(0, 0, 0, 0.05);
  gap: 8px;
`

const TabButton = styled.div`
  flex: 1;
  padding: 12px;
  font-size: 14px;
  font-weight: 600;
  color: #ffffff;
  text-align: center;
  background: ${(props) => (props.active ? "#f09c06" : "transparent")};
  border-radius: 12px;
  transition: all 0.2s ease;
  cursor: pointer;

  &:hover {
    background: ${(props) => (props.active ? "#f09c06" : "rgba(255, 255, 255, 0.1)")};
  }
`

const NotificationList = styled.div`
  overflow-y: auto;
  padding: 8px;

  &::-webkit-scrollbar {
    width: 8px;
  }

  &::-webkit-scrollbar-track {
    background: rgba(0, 0, 0, 0.1);
    border-radius: 4px;
  }

  &::-webkit-scrollbar-thumb {
    background: rgba(255, 255, 255, 0.2);
    border-radius: 4px;
    
    &:hover {
      background: rgba(255, 255, 255, 0.3);
    }
  }
`
const NotificationItemWrapper = styled.div`
  padding: 16px;
  margin-bottom: 8px;
  border-radius: 12px;
  cursor: pointer;
  transition: all 0.2s ease;

  background: ${(props) =>
    props.unread
      ? `rgba(255, 255, 255, 0.18)`
      : `rgba(255, 255, 255, 0.18)`};

  &:hover {
    background: rgba(255, 255, 255, 0.18);
    transform: translateY(-1px);
  }

  &:active {
    transform: translateY(0);
  }
`;

const NotificationHeader = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: flex-start;
  margin-bottom: 6px;
`

const TitleContainer = styled.div`
  display: flex;
  align-items: center;
  gap: 8px;
`

const NotificationTitle = styled.div`
  font-size: 15px;
  font-weight: 600;
  color: #ffffff;
`

const UnreadDot = styled(Circle)`
  width: 6px;
  height: 6px;
  fill: #FF6B8B;
`

const Timestamp = styled.span`
  font-size: 12px;
  color: rgba(255, 255, 255, 0.7);
  font-weight: 500;
`

const NotificationContent = styled.div`
  font-size: 14px;
  color: rgba(255, 255, 255, 0.9);
  line-height: 1.5;
`

const EmptyState = styled.div`
  padding: 32px 16px;
  text-align: center;
  color: rgba(255, 255, 255, 0.7);
  font-size: 15px;
  font-weight: 500;
`

const LoadingSpinner = styled(Loader2)`
  animation: spin 1s linear infinite;
  margin: 32px auto;
  display: block;
  color: rgba(255, 255, 255, 0.9);
  
  @keyframes spin {
    from {
      transform: rotate(0deg);
    }
    to {
      transform: rotate(360deg);
    }
  }
`

const IconContainer = styled.div`
  position: relative;
  display: inline-flex;
`

const NotificationDropdown = ({ setCount, notifications: externalNotifications, onMarkAsRead, onMarkAllAsRead, unreadCount: externalUnreadCount, isLoading: externalIsLoading }) => {
  const [isOpen, setIsOpen] = useState(false);
  const [activeTab, setActiveTab] = useState("unread");
  const [notifications, setNotifications] = useState({ unread: [], read: [] });
  const [isLoading, setIsLoading] = useState(false);
  const [unreadCount, setUnreadCount] = useState(0);
  const menuRef = useRef(null);
  const iconRef = useRef(null);

  // Use external props if provided, otherwise use internal state
  const finalNotifications = externalNotifications || notifications;
  const finalUnreadCount = externalUnreadCount !== undefined ? externalUnreadCount : unreadCount;
  const finalIsLoading = externalIsLoading !== undefined ? externalIsLoading : isLoading;

  const fetchUnreadCount = async () => {
    // Only fetch if setCount is provided (internal mode)
    if (!setCount) return;
    
    try {
      const data = await notificationService.getUnreadCount(getUID());
      if (data.success) {
        setUnreadCount(data.unreadCount);
        setCount(data.unreadCount); // Update parent component count
      }
    } catch (error) {
      console.error('Failed to fetch unread count:', error);
    }
  };

  const fetchNotifications = async () => {
    // Only fetch if external props are not provided (internal mode)
    if (externalNotifications) return;
    
    setIsLoading(true);
    try {
      const data = await notificationService.getNotificationHistory(getUID());
      if (data.success) {
        setNotifications({
          unread: data.notifications.unread || [],
          read: data.notifications.read || []
        });
        // Update unread count after fetching notifications
        await fetchUnreadCount();
      }
    } catch (error) {
      console.error('Failed to fetch notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const markAsRead = async (id) => {
    // Use external handler if provided, otherwise use internal logic
    if (onMarkAsRead) {
      onMarkAsRead(id);
      return;
    }
    
    try {
      const response = await notificationService.markAsRead({
        userId: getUID(),
        notificationIds: [id]
      });

      if (response.success) {
        // Update local state
        setNotifications(prev => {
          const updatedUnread = prev.unread.filter(n => n.id !== id);
          const movedNotification = prev.unread.find(n => n.id === id);
          return {
            unread: updatedUnread,
            read: movedNotification ? [...prev.read, { ...movedNotification, is_read: true }] : prev.read
          };
        });
        // Refresh unread count
        await fetchUnreadCount();
      }
    } catch (error) {
      console.error('Failed to mark notification as read:', error);
    }
  };

  const markAllAsRead = async () => {
    // Use external handler if provided, otherwise use internal logic
    if (onMarkAllAsRead) {
      onMarkAllAsRead();
      return;
    }
    
    try {
      const unreadIds = notifications.unread.map(notification => notification.id);
      const response = await notificationService.markMultipleAsRead({
        userId: getUID(),
        notificationIds: unreadIds
      });

      if (response.success) {
        setNotifications(prev => ({
          unread: [],
          read: [...prev.read, ...prev.unread.map(n => ({ ...n, is_read: true }))]
        }));
        // Refresh unread count
        await fetchUnreadCount();
      }
    } catch (error) {
      console.error('Failed to mark all notifications as read:', error);
    }
  };

  // Handle clicks outside the dropdown
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (
        isOpen &&
        menuRef.current &&
        !menuRef.current.contains(event.target) &&
        !iconRef.current.contains(event.target)
      ) {
        setIsOpen(false);
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => {
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, [isOpen]);

  // Fetch notifications when dropdown opens
  useEffect(() => {
    if (isOpen) {
      fetchNotifications();
    }
  }, [isOpen]);

  // Initial fetch of unread count
  useEffect(() => {
    fetchUnreadCount();
  }, []);

  const formatTimestamp = (timestamp) => {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now.getTime() - date.getTime();
    
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);

    if (days > 0) return `${days}d ago`;
    if (hours > 0) return `${hours}h ago`;
    if (minutes > 0) return `${minutes}m ago`;
    return 'Just now';
  };

  const NotificationItem = ({ notification }) => (
    <NotificationItemWrapper 
      onClick={() => markAsRead(notification.id)}
      unread={!notification.is_read}
    >
      <NotificationHeader>
        <TitleContainer>
          <NotificationTitle>{notification.title}</NotificationTitle>
          {!notification.is_read && <UnreadDot />}
        </TitleContainer>
        <Timestamp>{formatTimestamp(notification.created_at)}</Timestamp>
      </NotificationHeader>
      <NotificationContent>{notification.body}</NotificationContent>
    </NotificationItemWrapper>
  );

  return (
    <Container>
      <IconWrapper onClick={() => setIsOpen(!isOpen)} ref={iconRef}>
        <IconContainer>
          <NotificationsSkyla />
          <UnreadBadge>{finalUnreadCount}</UnreadBadge>
        </IconContainer>
      </IconWrapper>

      {isOpen && (
        <DropdownMenu ref={menuRef}>
          <Header>
            <Title>Notifications</Title>
            <MarkAllButton 
              onClick={markAllAsRead} 
              disabled={finalNotifications.unread.length === 0}
            >
              Mark all as read
            </MarkAllButton>
          </Header>

          <TabContainer>
            <TabButton 
              active={activeTab === "unread"} 
              onClick={() => setActiveTab("unread")}
            >
              Unread ({finalNotifications.unread.length})
            </TabButton>
            <TabButton 
              active={activeTab === "read"} 
              onClick={() => setActiveTab("read")}
            >
              Read ({finalNotifications.read.length})
            </TabButton>
          </TabContainer>

          <NotificationList>
            {finalIsLoading ? (
              <LoadingSpinner size={24} />
            ) : (
              activeTab === "unread" ? (
                finalNotifications.unread.length > 0 ? (
                  finalNotifications.unread.map(notification => (
                    <NotificationItem key={notification.id} notification={notification} />
                  ))
                ) : (
                  <EmptyState>No unread notifications</EmptyState>
                )
              ) : (
                finalNotifications.read.length > 0 ? (
                  finalNotifications.read.map(notification => (
                    <NotificationItem key={notification.id} notification={notification} />
                  ))
                ) : (
                  <EmptyState>No read notifications</EmptyState>
                )
              )
            )}
          </NotificationList>
        </DropdownMenu>
      )}
    </Container>
  );
};
export default NotificationDropdown