import React, { useState, useRef, useEffect } from 'react';
import styled, { keyframes } from 'styled-components';
import { useHistory } from 'react-router-dom';

const SportsBanner = () => {
  const [scrollPosition, setScrollPosition] = useState(0);
  const containerRef = useRef(null);
  const history = useHistory();
  const maxScroll = useRef(0);

  const sportsData = [
    { id: 1, name: 'In Play', icon: '🎯', tab: 'all-sports' },
    { id: 2, name: 'Cricket', icon: '🏏', tab: 'cricket' },
    { id: 3, name: 'Soccer', icon: '⚽', tab: 'soccer' },
    { id: 4, name: 'Tennis', icon: '🎾', tab: 'tennis' },
    // { id: 5, name: "Mixed Martial Arts", icon: "🥊", tab: 'all-sportsw' },
    { id: 6, name: "Golf", icon: "⛳", tab: 'golf' },
    { id: 7, name: "Boxing", icon: "👊", tab: 'boxing' },
    { id: 8, name: "American Football", icon: "🏈", tab: 'american-football' },
    { id: 9, name: "Baseball", icon: "⚾", tab: 'baseball' },
    { id: 10, name: "Basketball", icon: "🏀", tab: 'basketball' },
    { id: 11, name: "Rugby League", icon: "🏉", tab: 'rugby-league' },
  ];

  useEffect(() => {
    const container = containerRef.current;
    if (!container) return;

    const updateMaxScroll = () => {
      const containerWidth = container.offsetWidth;
      const scrollContent = container.querySelector('[data-scroll-track]');
      if (scrollContent) {
        const scrollWidth = scrollContent.scrollWidth;
        maxScroll.current = Math.max(0, scrollWidth - containerWidth);
      }
    };
    updateMaxScroll();
    window.addEventListener('resize', updateMaxScroll);

    const handleWheel = (e) => {
      e.preventDefault();
      e.stopPropagation();

      const scrollSensitivity = 1;
      const delta = (e.deltaX !== 0 ? e.deltaX : e.deltaY) * scrollSensitivity;

      setScrollPosition((prev) => {
        const newPosition = prev + delta;
        return Math.max(0, Math.min(newPosition, maxScroll.current || 0));
      });
    };

    container.addEventListener('wheel', handleWheel, { passive: false });

    return () => {
      container.removeEventListener('wheel', handleWheel);
      window.removeEventListener('resize', updateMaxScroll);
    };
  }, []);

  const [isLoading, setIsLoading] = useState(true);
  const [mobileMatches, setMobileMatches] = useState([]);
  const [activeTab, setActiveTab] = useState('in-play');
  const [sporttab, setSportTab] = useState('all-sports');
  const [isMobile, setIsMobile] = useState(false);

  useEffect(() => {
    const updateIsMobile = () => setIsMobile(window.innerWidth <= 420);
    updateIsMobile();
    window.addEventListener('resize', updateIsMobile);
    return () => window.removeEventListener('resize', updateIsMobile);
  }, []);

  const handleTabClick = (tab) => {
    setActiveTab(tab);
    // setIsLoading(true);
  };

  // Add this click handler function with loader
  const handleSportTabClick = (tabValue) => {

    setIsLoading(true);
    setSportTab(tabValue);

  };
  const useResponsiveStyle = () => {
    const [style, setStyle] = useState(getSportCardStyle(window.innerWidth));

    useEffect(() => {
      const handler = () => {
        setStyle(getSportCardStyle(window.innerWidth));
      };

      window.addEventListener('resize', handler);
      return () => window.removeEventListener('resize', handler);
    }, []);

    return style;
  };

  // useEffect(() => {
  //   if (props.tab) {
  //     setIsLoading(true)
  //   }
  // }, [props.tab])
  const fetchMatchOdds = async (matchId) => {
    try {
      const token = localStorage.getItem('token');
      const res = await fetch(
        `https://api.boss707.com/sports/market-odds?market_id=${matchId}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );
      const [marketOdds] = await res.json();
      if (!marketOdds?.runners) return null;

      const runners = marketOdds.runners;
      const formatted = {
        one: { back: '--', lay: '--' },
        x: { back: '--', lay: '--' },
        two: { back: '--', lay: '--' }
      };

      if (runners.length === 2) {
        formatted.one = {
          back: runners[0].ex.availableToBack[0]?.price || '--',
          lay: runners[0].ex.availableToLay[0]?.price || '--'
        };
        formatted.two = {
          back: runners[1].ex.availableToBack[0]?.price || '--',
          lay: runners[1].ex.availableToLay[0]?.price || '--'
        };
      } else if (runners.length === 3) {
        formatted.one = {
          back: runners[0].ex.availableToBack[0]?.price || '--',
          lay: runners[0].ex.availableToLay[0]?.price || '--'
        };
        formatted.x = {
          back: runners[1].ex.availableToBack[0]?.price || '--',
          lay: runners[1].ex.availableToLay[0]?.price || '--'
        };
        formatted.two = {
          back: runners[2].ex.availableToBack[0]?.price || '--',
          lay: runners[2].ex.availableToLay[0]?.price || '--'
        };
      }

      return formatted;
    } catch (err) {
      console.error('Error fetching odds:', err);
      return null;
    }
  };

  useEffect(() => {
    const fetchAllMatches = async () => {
      setIsLoading(true);
      try {
        const token = localStorage.getItem('token');
        let rawMarkets = [];

        if (activeTab === 'in-play') {
          // in-play
          if (sporttab === 'all-sports') {
            // For all-sports + in-play: call only the two aggregate endpoints
            const [inPlayRes, allMatchesRes] = await Promise.all([
              fetch('https://api.boss707.com/sports/allinplay', {
                headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' }
              }),
              fetch('https://api.boss707.com/sports/all-matches', {
                headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' }
              })
            ]);
            const inPlayData = await inPlayRes.json();
            const allMatchesData = await allMatchesRes.json();

            const collectMarkets = (arr) => Array.isArray(arr)
              ? arr.flatMap(s => s?.markets || [])
              : [];

            const inPlayMarkets = collectMarkets(inPlayData).map(m => ({ ...m, inplay: true }));
            const allMarkets = collectMarkets(allMatchesData);

            const map = new Map();
            inPlayMarkets.forEach(m => m?.marketId && map.set(m.marketId, m));
            allMarkets.forEach(m => {
              if (m?.marketId && !map.has(m.marketId)) map.set(m.marketId, m);
            });
            rawMarkets = Array.from(map.values());
          } else {
            // Per-sport in-play using sport IDs
            const tabToSportId = {
              'cricket': '4',
              'tennis': '2',
              'soccer': '1',
              'american-football': '6423',
              'baseball': '7511',
              'basketball': '7522',
              'boxing': '6',
              'darts': '3503',
              'gaelic-games': '2152880',
              'golf': '3',
              'greyhound-racing': '4339',
              'horse-racing': '7',
              'mma': '26420387',
              'motorsport': '8',
              'rugby-league': '1477',
              'rugby-union': '5',
              'snooker': '6422'
            };

            const fetchInplayById = async (sportId) => {
              const res = await fetch(`https://api.boss707.com/sports/inplayGameId/${sportId}`, {
                headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' }
              });
              const json = await res.json();
              const games = json?.games || json || [];
              const markets = Array.isArray(games)
                ? games.flatMap(g => g?.markets || [])
                : [];
              // tag as in-play explicitly
              return markets.map(m => ({ ...m, inplay: true }));
            };
            const fetchMatchesById = async (sportId) => {
              const res = await fetch(`https://api.boss707.com/sports/matches/${sportId}`, {
                headers: { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' }
              });
              const json = await res.json();
              const games = json?.games || json || [];

              const markets = Array.isArray(games)
                ? games.flatMap(g => g?.markets || [])
                : [];

              return markets.map(m => ({ ...m, inplay: false })); // Optional: tag these as not in-play
            };

            const sportId = tabToSportId[sporttab];
            const markets = sportId ? await fetchInplayById(sportId) : [];
            rawMarkets = markets || [];
          }
        }

        // map rawMarkets → our UI shape, then fetch odds
        const withOdds = await Promise.all(
          rawMarkets.map(async m => {
            const utcDate = new Date(m.marketStartTime);
            const timeStr = new Intl.DateTimeFormat('en-IN', {
              hour: '2-digit', minute: '2-digit', hour12: true
            }).format(utcDate);

            const match = {
              id: m.marketId,
              runners: m.runners,
              team1: m.runners[0]?.runnerName || '',
              team2: m.runners[1]?.runnerName || '',
              status: (m?.inplay ? 'In-Play' : timeStr),
              isLive: Boolean(m?.inplay),
              time: timeStr,
              date: utcDate,
              isSuspended: m.stopBet,
              eventid: m.version,
              odds: { one: { back: '--', lay: '--' }, x: { back: '--', lay: '--' }, two: { back: '--', lay: '--' } }
            };

            const odds = await fetchMatchOdds(match.id);
            if (odds) match.odds = odds;
            return match;
          })
        );

        // sort: in-play first, then by start time
        withOdds.sort((a, b) => {
          if (a.isLive && !b.isLive) return -1;
          if (!a.isLive && b.isLive) return 1;
          return new Date(a.date) - new Date(b.date);
        });

        setMobileMatches(withOdds);
      } catch (err) {
        console.error('Error fetching matches:', err);
      } finally {
        setIsLoading(false);
      }
    };

    fetchAllMatches();
  }, [sporttab, activeTab]);
  // Create dates for testing
  const today = new Date();
  const tomorrow = new Date(today);
  tomorrow.setDate(tomorrow.getDate() + 1);

  console.log("macthes", mobileMatches)
  console.log("is loading", isLoading)

  const mobileMatchesMap = mobileMatches?.map((data) => ({
    id: data?.id,
    team1: data?.team1,
    team2: data?.team2,
    status: data?.status,
    isLive: true,
    time: '14:30',
    date: new Date(),
    isSuspended: data?.isSuspended,
    competition: 'International',
    eventId: data.version,
    odds: data.odds || {
      one: { back: '--', lay: '--' },
      x: { back: '--', lay: '--' },
      two: { back: '--', lay: '--' }
    }
  }))

  console.log("mobile matches map", mobileMatchesMap)

  const getMatchStatus = (match) => (
    <Status status={match.status}>
      {match.status}
    </Status>
  );

  const handleMatchClick = (matchOrId) => {
    const selectedMatch =
      typeof matchOrId === 'object'
        ? matchOrId
        : mobileMatches?.find((m) => String(m.id) === String(matchOrId));

    // If not found, still navigate by id to let the route fetch its own data
    if (!selectedMatch) {
      history.push(`/match/fullmarket/event/${matchOrId}`);
      return;
    }

    history.push({
      pathname: `/match/fullmarket/event/${selectedMatch.id}`,
      state: {
        matchData: {
          ...selectedMatch,
          matchId: selectedMatch.id,
          eventId: selectedMatch?.eventid ?? selectedMatch?.version,
          matchTitle: `${selectedMatch.team1} v ${selectedMatch.team2}`,
          competition: selectedMatch?.competition,
          status: selectedMatch?.status,
          isLive: selectedMatch?.isLive,
          isSuspended: selectedMatch?.isSuspended,
        },
      },
    });
  };

  const bannerContainerStyle = {
    width: '100%',
    padding: isMobile ? '6px 4px' : '10px 6px',
    overflow: 'hidden',
    margintop: '10px',
    position: 'relative',
    zIndex: 1,
  };

  const scrollContainerStyle = {
    position: 'relative',
    height: isMobile ? '68px' : '80px',
    cursor: 'grab',
    userSelect: 'none',
  };

  const scrollTrackStyle = {
    display: 'flex',
    gap: '15px',
    transform: `translateX(-${scrollPosition}px)`,
    transition: 'transform 0.1s ease-out',
    width: 'max-content',
    willChange: 'transform',
  };

  const sportCardStyle = {
    flexShrink: 0,
    background: 'rgba(255, 255, 255, 0.1)',
    backdropFilter: 'blur(10px)',
    borderRadius: '12px',
    padding: isMobile ? '8px 8px' : '12px 10px',
    border: '1px solid rgba(255, 255, 255, 0.2)',
    cursor: 'pointer',
    transition: 'all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275)',
  };

  const sportCardHoverStyle = {
    background: 'rgba(255, 255, 255, 0.2)',
    transform: 'scale(1.05)',
  };

  const sportContentStyle = {
    display: 'flex',
    alignItems: 'center',
    gap: '12px',
    whiteSpace: 'nowrap',
  };

  const sportIconStyle = {
    fontSize: isMobile ? '20px' : '24px',
    transition: 'transform 0.3s ease',
  };

  const sportNameStyle = {
    color: 'white',
    fontWeight: '500',
    fontSize: isMobile ? '14px' : '16px',
    fontFamily: 'Montserrat Alternates, sans-serif',
  };

  const progressBarContainerStyle = {
    position: 'absolute',
    bottom: '2px',
    left: '50%',
    transform: 'translateX(-50%)',
    display: 'flex',
    gap: '4px',
  };

  const progressBarStyle = {
    width: '64px',
    height: '4px',
    background: 'rgba(255, 255, 255, 0.2)',
    borderRadius: '2px',
    overflow: 'hidden',
  };

  const progressFillStyle = {
    height: '100%',
    background: 'rgba(255, 255, 255, 0.6)',
    transition: 'width 0.1s ease',
    borderRadius: '2px',
    width: `${maxScroll.current > 0 ? Math.min(100, (scrollPosition / maxScroll.current) * 100) : 0}%`,
  };

  const scrollHintStyle = {
    position: 'absolute',
    top: '50%',
    right: '16px',
    transform: 'translateY(-50%)',
    color: 'rgba(255, 255, 255, 0.6)',
    fontSize: '12px',
    animation: 'pulse 2s infinite',
  };

  return (
    <>
      <div style={bannerContainerStyle}>
        <div
          ref={containerRef}
          style={scrollContainerStyle}
          onMouseDown={(e) => (e.currentTarget.style.cursor = 'grabbing')}
          onMouseUp={(e) => (e.currentTarget.style.cursor = 'grab')}
          onMouseLeave={(e) => (e.currentTarget.style.cursor = 'grab')}
        >
          <div style={scrollTrackStyle} data-scroll-track>
            {sportsData.map((sport) => {
              const isActive = sporttab === sport.tab;
              const activeStyle = isActive ? {
                background: 'linear-gradient(135deg, rgba(240, 156, 6, 0.4), rgba(255, 193, 7, 0.2))',
                border: '2px solid rgb(240, 156, 6)',
                transform: 'scale(1.05)',
                boxShadow: '0 8px 25px rgba(240, 156, 6, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.2)',
                backdropFilter: 'blur(15px)',
                position: 'relative',
                overflow: 'hidden',
                '::before': {
                  content: '""',
                  position: 'absolute',
                  top: 0,
                  left: '-100%',
                  width: '100%',
                  height: '100%',
                  background: 'linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent)',
                  animation: 'shimmer 2s infinite'
                }
              } : {};

              return (
                <div
                  key={sport.id}
                  style={{ ...sportCardStyle, ...activeStyle }}
                  onClick={() => handleSportTabClick(sport.tab)}
                  onMouseEnter={(e) => {
                    if (!isActive) {
                      Object.assign(e.currentTarget.style, {
                        ...sportCardStyle,
                        ...sportCardHoverStyle
                      });
                    }
                    const icon = e.currentTarget.querySelector('.sport-icon');
                    if (icon) icon.style.transform = 'scale(1.1)';
                  }}
                  onMouseLeave={(e) => {
                    if (!isActive) {
                      Object.assign(e.currentTarget.style, sportCardStyle);
                    } else {
                      Object.assign(e.currentTarget.style, {
                        ...sportCardStyle,
                        ...activeStyle
                      });
                    }
                    const icon = e.currentTarget.querySelector('.sport-icon');
                    if (icon) icon.style.transform = isActive ? 'scale(1.05)' : 'scale(1)';
                  }}
                >
                  <div style={sportContentStyle}>
                    <span className="sport-icon" style={{
                      ...sportIconStyle,
                      transform: isActive ? 'scale(1.15)' : 'scale(1)',
                      filter: isActive ? 'drop-shadow(0 4px 8px rgba(240, 156, 6, 0.6)) brightness(1.2)' : 'none'
                    }}>
                      {sport.icon}
                    </span>
                    <span style={{
                      ...sportNameStyle,
                      color: isActive ? '#fff' : 'white',
                      fontWeight: isActive ? '700' : '500',
                      textShadow: isActive ? '0 2px 8px rgba(240, 156, 6, 0.5)' : 'none',
                      letterSpacing: isActive ? '0.5px' : 'normal'
                    }}>
                      {sport.name}
                    </span>
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        <style jsx>{`
          @keyframes pulse {
            0%,
            100% {
              opacity: 0.6;
            }
            50% {
              opacity: 1;
            }
          }
          
          @keyframes shimmer {
            0% {
              left: -100%;
            }
            100% {
              left: 100%;
            }
          }
          
          @keyframes glow {
            0%, 100% {
              box-shadow: 0 8px 25px rgba(240, 156, 6, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.2);
            }
            50% {
              box-shadow: 0 12px 35px rgba(240, 156, 6, 0.5), inset 0 1px 0 rgba(255, 255, 255, 0.3);
            }
          }
        `}</style>
      </div>

     <TabContent>
  {isLoading ? (
    Array.from({ length: isMobile ? 6 : 8 }).map((_, idx) => (
      <SkeletonRow key={idx}>
        <SkeletonLeft>
          <div style={{ display: 'flex', alignItems: 'center', gap: '6px' }}>
            <SkeletonDot />
            <SkeletonLine widthPercent={70} heightPx={isMobile ? 10 : 12} />
          </div>
          <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
            <SkeletonPill widthPx={isMobile ? 40 : 56} heightPx={isMobile ? 16 : 20} />
            <SkeletonPill widthPx={isMobile ? 16 : 20} heightPx={isMobile ? 16 : 20} />
            <SkeletonPill widthPx={isMobile ? 16 : 20} heightPx={isMobile ? 16 : 20} />
            <SkeletonPill widthPx={isMobile ? 16 : 20} heightPx={isMobile ? 16 : 20} />
            <SkeletonPill widthPx={isMobile ? 16 : 20} heightPx={isMobile ? 16 : 20} />
            <SkeletonLine widthPercent={20} heightPx={isMobile ? 10 : 12} style={{ marginLeft: 8 }} />
          </div>
        </SkeletonLeft>
        <SkeletonOddsGroup>
          <SkeletonOddsBox mobile={isMobile} />
          <SkeletonOddsBox mobile={isMobile} />
          <SkeletonOddsBox mobile={isMobile} />
          <SkeletonOddsBox mobile={isMobile} />
        </SkeletonOddsGroup>
      </SkeletonRow>
    ))
  ) : mobileMatchesMap.length === 0 ? (
    <div style={{ 
      display: 'flex', 
      flexDirection: 'column', 
      alignItems: 'center', 
      justifyContent: 'center', 
      padding: '20px', 
      textAlign: 'center',
      color: '#fff',
      fontFamily: 'Montserrat Alternates, sans-serif',
      fontSize: isMobile ? '14px' : '16px'
    }}>
      <img
        src="/assets/images/bcgames/affiliate/empty.webp" // Replace with your warning/error icon path
        style={{ 
          width: isMobile ? '60px' : '50px', 
          height: isMobile ? '50px' : '50px', 
          marginBottom: '10px',
          objectFit: 'contain'
        }}
        alt="warning icon"
      />
 <span>Oops! , No match found for this right now .</span>
    </div>
  ) : (
    mobileMatchesMap.map((match, index) => (
      <Match key={index} onClick={() => handleMatchClick(match)}>
        <MatchInfo>
          <div style={{ display: 'flex', gap: '0px', alignItems: 'center', justifyContent: 'space-between' }}>
            <div style={{ display: 'flex', flexDirection: 'column' }}>
              <div style={{ display: 'flex', gap: '5px' }}>
                <StatusDot isLive={match?.status} />
                <MatchTitle>
                  {`${match.team1} v ${match.team2}`}
                </MatchTitle>
              </div>
              <div style={{ display: 'flex', gap: '10px', alignItems: 'center' }}>
                <IconGroup>
                  <img
                    src="/assets/images/sports/cricket.png"
                    style={{ width: isMobile ? '30px' : '25px', height: isMobile ? '12px' : '18px', objectFit: 'contain' }}
                    alt="cricket icon"
                  />
                  <img
                    src="/assets/images/sports/livestream.png"
                    style={{ width: isMobile ? '12px' : '15px', height: isMobile ? '12px' : '14px', objectFit: 'contain' }}
                    alt="livestream icon"
                  />
                  <img
                    src="/assets/images/sports/watch.png"
                    style={{ width: isMobile ? '12px' : '15px', height: isMobile ? '12px' : '14px', objectFit: 'contain' }}
                    alt="watch icon"
                  />
                  <img
                    src="/assets/images/sports/fancy.png"
                    style={{ width: isMobile ? '12px' : '15px', height: isMobile ? '12px' : '14px', objectFit: 'contain' }}
                    alt="fancy icon"
                  />
                  <img
                    src="/assets/images/sports/book.png"
                    style={{ width: isMobile ? '12px' : '15px', height: isMobile ? '12px' : '14px', objectFit: 'contain' }}
                    alt="book icon"
                  />
                </IconGroup>
                {getMatchStatus(match)}
              </div>
            </div>
            <div style={{ display: 'flex', flexDirection: 'row', gap: '4px', alignItems: 'center' }}>
              <div style={{
                backgroundColor: '#72C0F4',
                padding: isMobile ? '3px 2px' : '4px 3px',
                borderRadius: '2px',
                fontWeight: 'bold',
                fontSize: isMobile ? '8px' : '9px',
                color: 'black',
                minWidth: isMobile ? '24px' : '30px',
                textAlign: 'center',
                fontFamily: 'Montserrat Alternates, sans-serif'
              }}>
                {match?.odds?.one?.back}
              </div>
              <div style={{
                backgroundColor: '#FFB6C1',
                padding: isMobile ? '3px 2px' : '4px 3px',
                borderRadius: '2px',
                fontWeight: 'bold',
                fontSize: isMobile ? '8px' : '9px',
                color: 'black',
                minWidth: isMobile ? '24px' : '30px',
                textAlign: 'center',
                fontFamily: 'Montserrat Alternates, sans-serif'
              }}>
                {match?.odds?.one?.lay}
              </div>
              <div style={{
                backgroundColor: '#72C0F4',
                marginLeft: "4px",
                padding: isMobile ? '3px 2px' : '4px 3px',
                borderRadius: '2px',
                fontWeight: 'bold',
                fontSize: isMobile ? '8px' : '9px',
                color: 'black',
                minWidth: isMobile ? '24px' : '30px',
                textAlign: 'center',
                fontFamily: 'Montserrat Alternates, sans-serif'
              }}>
                {match?.odds?.two?.back}
              </div>
              <div style={{
                backgroundColor: '#FFB6C1',
                padding: isMobile ? '3px 2px' : '4px 3px',
                borderRadius: '2px',
                fontWeight: 'bold',
                fontSize: isMobile ? '8px' : '9px',
                color: 'black',
                minWidth: isMobile ? '24px' : '30px',
                textAlign: 'center',
                fontFamily: 'Montserrat Alternates, sans-serif'
              }}>
                {match?.odds?.two?.lay}
              </div>
            </div>
          </div>
        </MatchInfo>
      </Match>
    ))
  )}
</TabContent>
    </>
  );
};

const Match = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  padding: 10px;
  border-bottom: 1px solid rgb(50, 55, 56);
  background: #000000;
  font-family: 'Montserrat Alternates', sans-serif;
  cursor: pointer;
  position: relative;
  z-index: 2; /* ensure rows are above any overlapping banner */
`;

const MatchInfo = styled.div`
  flex: 1;
  font-family: 'Montserrat Alternates', sans-serif;
`;

const MatchTitle = styled.div`
   color: #ffffff;
  font-weight: bold;
  margin-bottom: 0px;
  cursor: pointer;
  display: inline-block;
  font-size: 8px;
  font-family: 'Montserrat Alternates', sans-serif;
  max-width: 180px;
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;

  &:hover {
    text-decoration: underline;
    color: rgb(240, 156, 6);
  }
`;

const StatusDot = styled.span`
  display: inline-block;
  width: 8px;
  height: 8px;
  background-color: ${(props) => (props.isLive === 'In-Play' ? '#2ecc71' : '#95a5a6')};
  border-radius: 50%;
  margin-right: 8px;
  margin-bottom: -5px;
  animation: ${(props) => (props.isLive === 'In-Play' ? 'blink 1s ease-in-out infinite' : 'none')};

  @keyframes blink {
    0% {
      opacity: 0;
    }
    50% {
      opacity: 1;
    }
    100% {
      opacity: 0;
    }
  }
`;

const Status = styled.div`
  color: ${(props) => {
    switch (props.status) {
      case 'In-Play':
        return 'rgb(240, 156, 6)';
      case 'Not Started':
        return '#cccccc';
      case 'Suspended':
        return '#e74c3c';
      default:
        return '#cccccc';
    }
  }};
  font-size: 8px;
  font-weight: bold;
  margin-top: 2px;
  font-family: 'Montserrat Alternates', sans-serif;
`;

const IconGroup = styled.div`
  display: flex;
  gap: 2px;
  margin-bottom: -5px;
  font-family: 'Montserrat Alternates', sans-serif;
  margin-left: 20px;
`;

// Skeleton styles
const shimmerAnim = keyframes`
  0% { background-position: -200% 0; }
  100% { background-position: 200% 0; }
`;

const SkeletonBase = styled.div`
  background: linear-gradient(90deg, rgba(255,255,255,0.05), rgba(255,255,255,0.15), rgba(255,255,255,0.05));
  background-size: 200% 100%;
  animation: ${shimmerAnim} 1.2s ease-in-out infinite;
`;

const SkeletonRow = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  padding: 10px;
  border-bottom: 1px solid rgb(50, 55, 56);
  background: #000000;
`;

const SkeletonLeft = styled.div`
  flex: 1;
  display: flex;
  flex-direction: column;
  gap: 6px;
`;

const SkeletonDot = styled(SkeletonBase)`
  width: 8px;
  height: 8px;
  border-radius: 50%;
`;

const SkeletonLine = styled(SkeletonBase)`
  width: ${(p) => (p.widthPercent ? `${p.widthPercent}%` : '60%')};
  height: ${(p) => (p.heightPx ? `${p.heightPx}px` : '12px')};
  border-radius: 6px;
`;

const SkeletonPill = styled(SkeletonBase)`
  width: ${(p) => (p.widthPx ? `${p.widthPx}px` : '40px')};
  height: ${(p) => (p.heightPx ? `${p.heightPx}px` : '16px')};
  border-radius: 10px;
`;

const SkeletonOddsGroup = styled.div`
  display: flex;
  flex-direction: row;
  gap: 6px;
  align-items: center;
`;

const SkeletonOddsBox = styled(SkeletonBase)`
  width: ${(p) => (p.mobile ? '30px' : '36px')};
  height: ${(p) => (p.mobile ? '18px' : '22px')};
  border-radius: 6px;
`;

const TabContent = styled.div`
  margin-top: -30px;
  animation: fadeIn 0.3s ease;
  max-height: 235px; /* Set maximum height to 300px */
  overflow-y: auto; /* Enable vertical scrolling */
  background-color: #000000;
  font-family: 'Montserrat Alternates', sans-serif;
  

  &::-webkit-scrollbar {
    width: 8px;
  }

  &::-webkit-scrollbar-track {
    background: rgb(50, 55, 56);
  }

  &::-webkit-scrollbar-thumb {
    background: rgb(240, 156, 6);
    border-radius: 4px;
  }

  &::-webkit-scrollbar-thumb:hover {
    background: rgb(220, 136, 6);
  }

  @keyframes fadeIn {
    from {
      opacity: 0;
    }
    to {
      opacity: 1;
    }
  }
`;

export default SportsBanner;