import React from "react";
import { Col, Form, Row } from "react-bootstrap";
import storage from "../../../../Storage";
import socket from "../../../../Socket";
import { decode, encode, wait, sendNotfication, getUID } from "../../../../Helper";
import C from "../../../../Constant";
import { Icon } from "@mui/material";
import styled from "styled-components";
import ActiveSessions from "./Sessions";
import LockIcon from "../../../../Icons/Lock";
import EmailIcon from "../../../../Icons/EmailIcon";
import {
  Modal,
  Box,
  Typography,
  Card,
  CardContent,
  DialogTitle,
  DialogContent
} from '@mui/material';
import ChangePasswordModal from "./changePassword";

const modalStyle = {
  position: 'absolute',
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  width: 450,
  bgcolor: '#17181b',
  boxShadow: 24,
  p: 4,
  borderRadius: 12,
  border: '1px solid #ffffff15',
};

const ModalWrapper = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.5);
  display: flex;
  align-items: center;
  justify-content: center;
  z-index: 9999;
`;

const ModalContent = styled.div`
  width: 450px;
  border-radius: 12px;
  padding: 24px;
  box-shadow: 0 8px 16px rgba(0, 0, 0, 0.3);
  background-color: #17181b;
  position: relative;
`;

const ModalTitle = styled.div`
  font-size: 20px;
  font-weight: 700;
  color: #ffffff;
  margin-bottom: 20px;
`;

const ModalBody = styled.div`
  display: flex;
  flex-direction: column;
  gap: 16px;
`;

const ModalFooter = styled.div`
  display: flex;
  justify-content: flex-end;
  gap: 12px;
  margin-top: 20px;
`;

const Button = styled.button`
  background: ${(props) => (props.disabled ? '#f09c0680' : '#f09c06')};
  color: #000;
  border: none;
  border-radius: 12px;
  font-size: 14px;
  font-weight: 700;
  padding: 12px 24px;
  cursor: ${(props) => (props.disabled ? 'not-allowed' : 'pointer')};
  transition: background 0.3s ease;

  &:hover {
    background: ${(props) => (props.disabled ? '#f09c0680' : '#4AE54A')};
  }
`;

const CancelButton = styled(Button)`
  background: #ffffff15;
  color: #ffffff;

  &:hover {
    background: #ffffff30;
  }
`;

const TextField = styled.input`
  width: 100%;
  padding: 12px 16px;
  background-color: #02030E;
  border: 1px solid #ffffff15;
  border-radius: 8px;
  color: #ffffff;
  font-size: 14px;
  transition: border 0.3s ease;

  &:focus {
    outline: none;
    border-color: #f09c06;
  }
`;

const SecuritySetupWrapper = styled.div`
  display: flex;
  flex-direction: column;
  gap: 10px;
  padding: 20px;
  border-radius: 10px;
  background: #02030E;
`;

const SecuritySetupTitle = styled.div`
  font-size: 16px;
  font-weight: bold;
  color: #ffffff;
  padding-bottom: 10px;
  border-bottom: 1px solid #ffffff15;
  width: 100%;
`;

const SecuritySetupContainer = styled.div`
  display: flex;
  gap: 10px;
  flex-wrap: wrap;
  width: 100%;
  justify-content: space-between;
`;

const SecurityItem = styled.div`
  width: 50%;
  display: flex;
  flex-direction: column;
  gap: 15px;
  padding: 15px;
  border-radius: 10px;
  background: #111418;
  min-height: 240px;

  @media (max-width: 991px) {
    width: 100%;
  }
`;

const ItemHeader = styled.div`
  font-size: 14px;
  font-weight: bold;
  color: #ffffff;
  display: flex;
  align-items: center;
  gap: 10px;
  justify-content: space-between;
`;

const Status = styled.div`
  font-size: 12px;
  color: ${(props) => (props.isEnabled ? "#FFD700" : "#fff")};
`;

const ItemDetails = styled.div`
  display: flex;
  gap: 15px;
  align-items: center;
  flex-direction: column;
  width: 100%;
  height: 100%;
`;

const ItemDetailsTitle = styled.div`
  font-size: 16px;
  color: #ffffff;
  font-weight: 600;
  width: 100%;
`;

const ItemDetailsDescription = styled.div`
  font-size: 14px;
  font-weight: 500;
  color: #ffffff80;
  width: 100%;
`;

const EditButton = styled.button`
  color: #fff;
  background: #02030E;
  border: none;
  border-radius: 10px;
  padding: 10px 25px;
  font-size: 14px;
  font-weight: 700;
  letter-spacing: 1px;
  cursor: pointer;
  transition: all 0.3s ease;
  white-space: nowrap;
  width: 200px;
  margin-top: 15px;

  &:hover {
    background: #ffffff15;
  }

  @media (max-width: 768px) {
    font-size: 12px;
    padding: 8px 15px;
  }
`;

const ActiveSessionsWrapper = styled.div`
  display: flex;
  align-items: center;
  flex-direction: column;
  gap: 10px;
  margin-top: 15px;
  background: #02030E;
  padding: 15px;
  border-radius: 10px;
  width: 100%;
`;

const ActiveSessionsTitle = styled.div`
  font-size: 14px;
  color: #ffffff;
  font-weight: 600;
  width: 100%;
  padding-bottom: 15px;
  border-bottom: 1px solid #ffffff15;
`;

const CloseButton = styled.button`
  position: absolute;
  top: 16px;
  right: 16px;
  background: transparent;
  border: none;
  font-size: 1.5rem;
  cursor: pointer;
  color: #ffffff;
  transition: color 0.3s ease;

  &:hover {
    color: #ff0000;
  }
`;

const QRCodeImage = styled.img`
  margin: 16px auto;
  max-width: 200px;
  height: auto;
  display: block;
`;

const QRCodeWrapper = styled.div`
  background: #02030E;
  padding: 16px;
  border-radius: 8px;
  text-align: center;
`;

const QRCodeText = styled.p`
  color: #ffffff80;
  font-size: 14px;
  margin-top: 8px;
`;

class Security extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      loading: true,
      disabled: false,
      password: "",
      newPassword: "",
      confirmPassword: "",
      changePasswordModal: false,
      g2faModal: false,
      twoFaEnabled: false,
      twoFaQrCode: null,
      twoFaSecret: null,
      verifyCode: '',
    };
  }

  async componentDidMount() {
    const token = localStorage.getItem('token');
    try {
      const response = await fetch(`https://api.boss707.com/2fa/status/${getUID()}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });
      const data = await response.json();
      this.setState({
        twoFaEnabled: data.isEnabled,
        loading: false
      });
    } catch (error) {
      sendNotfication('Failed to get 2FA status', 'error', 'top-center');
      this.setState({ loading: false });
    }
  }

  initiate2FASetup = async () => {
    const token = localStorage.getItem('token');
    try {
      const response = await fetch('https://api.boss707.com/2fa/enable', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify({
          uid: getUID()
        })
      });

      const data = await response.json();

      if (data.qrCode && data.secret) {
        this.setState({
          twoFaQrCode: data.qrCode,
          twoFaSecret: data.secret
        });
      } else {
        sendNotfication('Failed to setup 2FA', 'error', 'top-center');
      }
    } catch (error) {
      sendNotfication('Failed to setup 2FA', 'error', 'top-center');
    }
  };

  verify2FA = async (e) => {
    e.preventDefault();
    const { verifyCode } = this.state;

    if (!verifyCode) {
      sendNotfication('Please enter verification code', 'error', 'top-center');
      return;
    }

    this.setState({ disabled: true });

    try {
      const response = await fetch('https://api.boss707.com/2fa/setup-verify', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          uid: getUID(),
          token: verifyCode
        })
      });

      const data = await response.json();

      if (data.success) {
        this.setState({
          twoFaEnabled: true,
          disabled: false,
          twoFaQrCode: null,
          twoFaSecret: null,
          verifyCode: '',
          g2faModal: false
        });
        sendNotfication('2FA enabled successfully', 'success', 'top-center');
      } else {
        this.setState({ disabled: false });
        sendNotfication('Invalid verification code', 'error', 'top-center');
      }
    } catch (error) {
      this.setState({ disabled: false });
      sendNotfication('Failed to verify code', 'error', 'top-center');
    }
  };

  disable2FA = async (e) => {
    e.preventDefault();
    const { verifyCode } = this.state;

    if (!verifyCode) {
      sendNotfication('Please enter verification code', 'error', 'top-center');
      return;
    }

    this.setState({ disabled: true });

    try {
      const response = await fetch('https://api.boss707.com/2fa/disable', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify({
          uid: getUID(),
          token: verifyCode
        })
      });

      const data = await response.json();

      if (data.success) {
        this.setState({
          twoFaEnabled: false,
          disabled: false,
          verifyCode: '',
          g2faModal: false
        });
        sendNotfication('2FA disabled successfully', 'success', 'top-center');
      } else {
        this.setState({ disabled: false });
        sendNotfication('Invalid verification code', 'error', 'top-center');
      }
    } catch (error) {
      this.setState({ disabled: false });
      sendNotfication('Failed to disable 2FA', 'error', 'top-center');
    }
  };

  changePassword = async (e) => {
    e.preventDefault();
    const { password, newPassword, confirmPassword } = this.state;
  
    if (!password || !newPassword || !confirmPassword) {
      sendNotfication('Please fill in all password fields', 'error', 'top-center');
      return;
    }
  
    if (newPassword !== confirmPassword) {
      sendNotfication('New passwords do not match', 'error', 'top-center');
      return;
    }
  
    this.setState({ disabled: true });
    const userid = getUID();
    const token = localStorage.getItem('token');
    const response = await fetch('https://api.boss707.com/user/change-password', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`,
      },
      body: JSON.stringify({
        uid: userid,
        oldPassword: password,
        newPassword: newPassword
      })
    });
  
    const data = await response.json();
  
    if (data.success) {
      this.setState({
        disabled: false,
        password: '',
        newPassword: '',
        confirmPassword: '',
        changePasswordModal: false
      });
      sendNotfication('Password changed successfully', 'success', 'top-center');
    } else {
      this.setState({ disabled: false });
      sendNotfication(data.message, 'error', 'top-center');
    }
  };
  
  // Handle input changes
  handleInputChange = (e) => {
    this.setState({ [e.target.name]: e.target.value });
  };
  
  // Toggle modal
  toggleModal = (modalName, value) => {
    this.setState({ [modalName]: value });
  };
  

  handleInputChange = (e) => {
    this.setState({ [e.target.name]: e.target.value });
  };

  toggleModal = (modalName, value) => {
    this.setState({ [modalName]: value });
  };

  render() {
    const {
      loading,
      disabled,
      changePasswordModal,
      g2faModal,
      twoFaEnabled,
      twoFaQrCode,
      verifyCode,
    } = this.state;

    if (loading) {
      return <div>Loading...</div>;
    }

    return (
      <>
        <SecuritySetupWrapper>
          <SecuritySetupTitle>Security Setup</SecuritySetupTitle>
          <SecuritySetupContainer>
            <SecurityItem>
              <ItemHeader>
                <LockIcon color="#FFEA00" hoverColor="#FFD700" />
                <Status isEnabled={true}>Enabled</Status>
              </ItemHeader>
              <ItemDetails>
                <ItemDetailsTitle>Change Password</ItemDetailsTitle>
                <ItemDetailsDescription>
                  Change to a strong password to protect your account from unauthorized access.
                </ItemDetailsDescription>
                <EditButton
                  type="button"
                  onClick={() => {
                    // Clear password fields before opening modal
                    this.setState({
                      password: '',
                      newPassword: '',
                      confirmPassword: ''
                    });
                    this.toggleModal("changePasswordModal", true);
                  }}
                >
                  Change Password
                </EditButton>
              </ItemDetails>
            </SecurityItem>

            <SecurityItem>
              <ItemHeader>
                <EmailIcon color="#FFEA00" hoverColor="#FFD700" />
                <Status isEnabled={twoFaEnabled}>
                  {twoFaEnabled ? "Enabled" : "Disabled"}
                </Status>
              </ItemHeader>
              <ItemDetails>
                <ItemDetailsTitle>Two-factor authentication</ItemDetailsTitle>
                <ItemDetailsDescription>
                  Enable Two-factor authentication to protect your account from unauthorized access.
                </ItemDetailsDescription>
                <EditButton
                  type="button"
                  onClick={() => {
                    this.toggleModal("g2faModal", true);
                    if (!twoFaEnabled) {
                      this.initiate2FASetup();
                    }
                  }}
                >
                  {twoFaEnabled ? 'Manage 2FA' : 'Enable 2FA'}
                </EditButton>
              </ItemDetails>
            </SecurityItem>
          </SecuritySetupContainer>
        </SecuritySetupWrapper>

        <ChangePasswordModal
          open={this.state.changePasswordModal}
          onClose={() => this.toggleModal("changePasswordModal", false)}
          password={this.state.password}
          newPassword={this.state.newPassword}
          oldPassword={""}
          confirmPassword={this.state.confirmPassword}
          handleInputChange={this.handleInputChange}
          changePassword={this.changePassword}
          disabled={this.state.isSubmitting}
        />
        {/* 2FA Modal */}
        {g2faModal && (
          <ModalWrapper>
            <ModalContent>
              <CloseButton onClick={() => this.toggleModal("g2faModal", false)}>
                ×
              </CloseButton>
              <ModalTitle>{twoFaEnabled ? 'Manage 2FA' : 'Enable 2FA'}</ModalTitle>
              <ModalBody>
                {!twoFaEnabled && twoFaQrCode && (
                  <QRCodeWrapper>
                    <QRCodeImage src={twoFaQrCode} alt="2FA QR Code" />
                    <QRCodeText>Scan this QR code with your authenticator app</QRCodeText>
                  </QRCodeWrapper>
                )}
                <TextField
                  type="text"
                  name="verifyCode"
                  value={verifyCode}
                  onChange={this.handleInputChange}
                  placeholder="Enter verification code"
                />
              </ModalBody>
              <ModalFooter>
                <CancelButton
                  type="button"
                  onClick={() => this.toggleModal("g2faModal", false)}
                >
                  Cancel
                </CancelButton>
                <Button
                  type="button"
                  onClick={twoFaEnabled ? this.disable2FA : this.verify2FA}
                  disabled={disabled}
                >
                  {twoFaEnabled
                    ? disabled
                      ? 'Disabling...'
                      : 'Disable 2FA'
                    : disabled
                      ? 'Enabling...'
                      : 'Enable 2FA'}
                </Button>
              </ModalFooter>
            </ModalContent>
          </ModalWrapper>
        )}

        <ActiveSessionsWrapper>
          <ActiveSessionsTitle>Active Sessions</ActiveSessionsTitle>
          <ActiveSessions />
        </ActiveSessionsWrapper>
      </>
    );
  }
}

export default Security;