import React, { Component } from 'react';
import styled from 'styled-components';
import { Play } from 'lucide-react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faArrowLeft, faSpinner } from '@fortawesome/free-solid-svg-icons';
import { getUID } from '../../../Helper';

const Grid = styled.div`
  display: grid;
  grid-template-columns: repeat(2, 1fr);
  gap: 1rem;
  
  @media (min-width: 768px) {
    grid-template-columns: repeat(4, 1fr);
  }
  
  @media (min-width: 1024px) {
    grid-template-columns: repeat(5, 1fr);
  }
`;

const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.4);
  backdrop-filter: blur(5px);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(255, 255, 255, 0.3);
  display: flex;
  justify-content: center;
  align-items: center;
`;

const FullScreenContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  z-index: 999;
  background-color: #1b2336;
`;

const FullScreenIframe = styled.iframe`
  width: 100%;
  height: 100%;
  border: none;
  display: ${(props) => (props.loading ? 'none' : 'block')};
`;

const PlayIcon = styled(Play)`
  color: #fff;
  font-size: 24px;
`;

const BackButton = styled.button`
  position: absolute;
  top: 10px;
  left: 10px;
  z-index: 1000;
  background: none;
  border: none;
  color: #fff;
  font-size: 24px;
  cursor: pointer;
  padding: 8px;
  
  &:hover {
    opacity: 0.8;
  }
`;

const GridWrapper = styled.div`
  display: grid;
  gap: 12px;
  padding: 12px;
  width: 100%;
  
  /* Mobile - 3 items per row */
  grid-template-columns: repeat(3, 1fr);
  
  /* Tablet - 4 items per row */
  @media (min-width: 640px) {
    grid-template-columns: repeat(4, 1fr);
  }
  
  /* Desktop - 6 items per row */
  @media (min-width: 1024px) {
    grid-template-columns: repeat(6, 1fr);
    gap: 16px;
    padding: 16px;
  }
  
  /* Large Desktop - 8 items per row */
  @media (min-width: 1280px) {
    grid-template-columns: repeat(8, 1fr);
    gap: 20px;
    padding: 20px;
  }
`;

const LotteryGameCard = styled.div`
  width: 100%;
  aspect-ratio: 1.32;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    ${GameOverlay} {
      opacity: 1;
    }
    transform: translateY(-5px);
  }
`;

const LoadMoreButton = styled.button`
 background-color: #03030F;
  color: white;
  border: none;
  padding: 12px 24px;
  border-radius: 8px;
  cursor: pointer;
  font-weight: bold;
  margin: 20px auto;
  display: block;
  transition: background-color 0.3s ease;

  &:hover {
    background-color: #4a5354;
  }

  &:disabled {
    background-color: #ccc;
    cursor: not-allowed;
  }
`;

const LoaderContainer = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 20px;
`;

const Spinner = styled.div`
  width: 40px;
  height: 40px;
  border: 4px solid #1c1e22;
  border-top: 4px solid rgb(18, 20, 24);
  border-radius: 50%;
  animation: spin 1s linear infinite;

  @keyframes spin {
    0% {
      transform: rotate(0deg);
    }
    100% {
      transform: rotate(360deg);
    }
  }
`;

const LotteryGameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
`;

const SkeletonLoader = styled.div`
  width: 100%;
  height: 100%;
  background: linear-gradient(
    90deg,
    rgba(190, 190, 190, 0.2) 25%,
    rgba(129, 129, 129, 0.24) 37%,
    rgba(190, 190, 190, 0.2) 63%
  );
  background-size: 400% 100%;
  animation: skeleton-loading 1.4s ease infinite;
  border-radius: 8px;

  @keyframes skeleton-loading {
    0% {
      background-position: 100% 50%;
    }
    100% {
      background-position: 0% 50%;
    }
  }
`;

class AllSlotsLiveGames extends Component {
  constructor(props) {
    super(props);
    this.state = {
      isLogged: localStorage.getItem('logged') === 'true',
      displayGamesCrashXgaming: [],
      message: null,
      iframeUrl: '',
      showIframe: false,
      iframeLoading: false,
      allPragmaticPlay: [],
      allPgGames: [],
      allNetentGames: [],
      showMoreAllSlots: false,
      isLoadingMoreSlots: false,
      pragmaticPage: 1,
      pgPage: 1,
      netentPage: 1,
      loadedImages: new Set(),
    };
  }
  handleImageLoad = (gameId) => {
    this.setState(prevState => ({
      loadedImages: new Set(prevState.loadedImages).add(gameId)
    }));
  };
  fetchVendorData = async ({ vendorName, page = 1, perPage = 10 }) => {
    const token = localStorage.getItem('token');
    const baseUrl = 'https://api.boss707.com/xGaming/by-vendor';
    const url = `${baseUrl}?vendor=${vendorName}&page=${page}&per_page=${perPage}`;

    try {
      const response = await fetch(url, {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      const data = await response.json();
      return data;
    } catch (error) {
      console.error(`Error fetching data for ${vendorName}:`, error);
      throw error;
    }
  };

  async componentDidMount() {
    await this.fetchInitialGames();
  }

  fetchInitialGames = async () => {
    try {
      const pragmatic = await this.fetchVendorData({
        vendorName: 'pragmaticplay',
        page: 1,
        perPage: 18
      });

      console.log("pragmetic", pragmatic?.data)
      const pg = await this.fetchVendorData({
        vendorName: 'pgsoft',
        page: 1,
        perPage: 8
      });
      const netent = await this.fetchVendorData({
        vendorName: 'netent',
        page: 1,
        perPage: 8
      });

      this.setState({
        allPragmaticPlay: pragmatic?.data || [],
        allPgGames: pg?.data || [],
        allNetentGames: netent?.data || [],
      });
    } catch (error) {
      console.error('Error fetching initial slots games:', error);
    }
  };



  loadMoreAllSlots = async () => {
    this.setState({ isLoadingMoreSlots: true });

    try {
      const nextPragmaticPage = this.state.pragmaticPage + 1;
      const nextPgPage = this.state.pgPage + 1;
      const nextNetentPage = this.state.netentPage + 1;

      const [pragmatic, pg, netent] = await Promise.all([
        this.fetchVendorData({
          vendorName: 'pragmaticplay',
          page: nextPragmaticPage,
          perPage: 18
        }),
        this.fetchVendorData({
          vendorName: 'pgsoft',
          page: nextPgPage,
          perPage: 8
        }),
        this.fetchVendorData({
          vendorName: 'netent',
          page: nextNetentPage,
          perPage: 8
        })
      ]);
      console.log("pragmetic", pragmatic)

      this.setState(prevState => ({
        allPragmaticPlay: [...prevState.allPragmaticPlay, ...(pragmatic.data || [])],
        allPgGames: [...prevState.allPgGames, ...(pg.data?.data || [])],
        allNetentGames: [...prevState.allNetentGames, ...(netent.data?.data || [])],
        pragmaticPage: nextPragmaticPage,
        pgPage: nextPgPage,
        netentPage: nextNetentPage,
        isLoadingMoreSlots: false
      }));
    } catch (error) {
      console.error('Error loading more slots:', error);
      this.setState({ isLoadingMoreSlots: false });
    }
  };

  handleGameLaunchXGaming = async (game) => {
    if (!this.state.isLogged) {
      this.setState({ message: { type: 'error', text: "Please login to play live game" } });
      return;
    }
    const token = localStorage.getItem('token');
    try {
      const response = await fetch('https://api.boss707.com/api/casino/gamerun', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}` },
        body: JSON.stringify({
          game_id: game.id,
          currency: "USD",
          mode: "real",
          language: "en",
          home_url: "https://api.boss707.com",
          device: "desktop",
          user_id: getUID,
          vendor: game.vendor,
          title: game.title,
          coin: "usdt"
        }),
      });
      const result = await response.json();
      if (result.gameRunUrl) {
        this.setState({ iframeUrl: result.gameRunUrl, showIframe: true, iframeLoading: true });
      } else {
        throw new Error("Failed to launch game");
      }
    } catch (error) {
      this.setState({ message: { type: 'error', text: error.message } });
    }
  };
  handleIframeLoad = () => {
    this.setState({ iframeLoading: false });
  };

  handleBackButton = () => {
    this.setState({ showIframe: false, iframeUrl: "", iframeLoading: true });
  };


  render() {
    const { showIframe, iframeUrl, iframeLoading, loadedImages } = this.state;
    console.log("all ages", this.state.allPgGames)
    const imageStyle = {
      position: 'absolute',
      top: '50%',
      left: '50%',
      transform: 'translate(-50%, -50%)',
      width: '250px',
      animation: 'fadeinout 2s infinite',
    };

    const animationStyle = `
            @keyframes fadeinout {
                0%, 100% { opacity: 0; }
                50% { opacity: 1; }
            }
        `;

    if (showIframe) {
      return (
        <FullScreenContainer>
          <style>{animationStyle}</style>
          {iframeLoading && (
            <img src="assets/images/./assets/boss707.png" alt="Loading" style={imageStyle} />
          )}
          <FullScreenIframe
            src={iframeUrl}
            title="Game Iframe"
            onLoad={this.handleIframeLoad}
            loading={iframeLoading}
          />
          <BackButton onClick={this.handleBackButton}>
            <FontAwesomeIcon icon={faArrowLeft} />
          </BackButton>
        </FullScreenContainer>
      );
    }

    return (


      <>
        <GridWrapper>
          {[...this.state.allNetentGames, ...this.state.allPgGames, ...this.state.allPragmaticPlay]?.map((game, index) => (
            <LotteryGameCard
              key={game.id}
              onClick={() => this.handleGameLaunchXGaming(game)}
            >
             {!loadedImages.has(game.id) && <SkeletonLoader />}
              <LotteryGameImage
                src={game.details_thumbnails_300x300}
                alt={game.title}
                onLoad={() => this.handleImageLoad(game.id)}
                style={{
                  display: loadedImages.has(game.id) ? 'block' : 'none'
                }}
              />
              <GameOverlay>
                <PlayIconContainer>
                  <PlayIcon />
                </PlayIconContainer>
              </GameOverlay>
            </LotteryGameCard>
          ))}
        </GridWrapper>
        {this.state.isLoadingMoreSlots ? (
          <LoaderContainer>
            <Spinner />
          </LoaderContainer>
        ) : (
          <LoadMoreButton
            onClick={this.loadMoreAllSlots}
            disabled={this.state.isLoadingMoreSlots}
          >
            Load More
          </LoadMoreButton>
        )}
      </>
    );
  }
}

export default AllSlotsLiveGames;