import React from 'react';
import styled from 'styled-components';
import { AreaChart, Area, ResponsiveContainer } from 'recharts';
import Draggable from 'react-draggable';
import C from '../../../Constant';
import { encode, decode, forceSatoshiFormat, __ } from "../../../Helper";
import storage from "../../../Storage";
import socket from '../../../Socket';

const PopupWrapper = styled.div`
  position: absolute;
  top: 10px;
  right: 10px;
  z-index: 1000;
`;

const PopupContainer = styled.div`
  background-color: #1a1a1a;
  border-radius: 8px;
  padding: 5px 10px;
  width: 270px;
  color: white;
  font-family: Arial, sans-serif;
  box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
  cursor: move;
`;

const Header = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-bottom: 16px;
`;

const Title = styled.h2`
  font-size: 18px;
  font-weight: bold;
  text-align: center;
  color: #49BC0D;
`;

const InnerContainer = styled.div`
  background-color: #1E2024;
  border-radius: 12px;
  padding: 10px;
`;

const CloseButton = styled.button`
  background: none;
  border: none;
  color: #777;
  font-size: 18px;
  cursor: pointer;
`;

const BetInfo = styled.div`
  display: flex;
  justify-content: space-between;
  margin-bottom: 16px;
  padding: 0 12px;
`;

const InfoColumn = styled.div`
  display: flex;
  flex-direction: column;
`;

const Label = styled.span`
  font-size: 14px;
  color: #777;
  margin-bottom: 4px;
`;

const Value = styled.span`
  font-size: 13px;
  font-weight: bold;
`;

const GraphContainer = styled.div`
  height: 150px;
  margin-bottom: 16px;
`;

const WinLoseInfo = styled.div`
  display: flex;
  justify-content: space-between;
  padding: 0 12px;
  margin-top: 4px;
`;

class BetsPopHilo extends React.Component {
  constructor(props) {
    super(props);
    this._isMounted = false;
    this.state = {
      profit: 0.00000000,
      engine: this.props.engine || null,
      wins: 0,
      losses: 0,
      data: [{ value: 0 }],
      userData: null,
      wagered: 0.00000000,
      roundWins: 0
    };
  }

  componentDidMount() {
    this._isMounted = true;
    const { engine } = this.state;
    
    if (engine) {
      engine.on(C.PLAY_HILO, this.handlePlay);
      engine.on(C.NEXT_HILO, this.handleNext);
      engine.on(C.HIGH_HILO, this.handleHighLow);
      engine.on(C.LOW_HILO, this.handleHighLow);
      engine.on(C.BUSTED_HILO, this.handleBusted);
      engine.on(C.ERROR_HILO, this.handleError);
      engine.on('win', this.handleWin);
    }
    socket.on(C.USER_INFO, this.handleUserInfo);
    this.requestUserInfo();
  }

  componentWillUnmount() {
    this._isMounted = false;
    const { engine } = this.state;
    
    if (engine) {
      engine.off();
    }
    socket.off(C.USER_INFO, this.handleUserInfo);
  }

  requestUserInfo = () => {
    socket.emit(C.USER_INFO, encode({
      id: storage.getKey('uid'),
      coin: storage.getKey('coin') || '',
      rate: null,
      game: 'all',
      first: true
    }));
  };

  handleUserInfo = (data) => {
    const decodedData = decode(data);
    // console.log("User info received:", decodedData);
    if (decodedData.status) {
      this.setState({ userData: decodedData });
    }
  };

  handlePlay = (data) => {
    if (this._isMounted && data) {
      // console.log("Play data received:", data);
      this.setState(prevState => ({
        wagered: forceSatoshiFormat(prevState.wagered + parseFloat(data.amount))
      }));
    }
  }

  handleNext = (data) => {
    if (this._isMounted && data) {
      // console.log("Next card data received:", data);
    }
  }

  handleHighLow = (data) => {
    if (this._isMounted && data) {
      // console.log("High/Low data received:", data);
    }
  }

  handleBusted = (data) => {
    if (this._isMounted && data) {
      const newProfit = parseFloat(data.profit || 0);
      
      this.setState(prevState => {
        const currentProfit = parseFloat(prevState.profit);
        const totalProfit = forceSatoshiFormat(currentProfit + newProfit);
        const wins = newProfit > 0 ? prevState.wins + 1 : prevState.wins;
        const losses = newProfit <= 0 ? prevState.losses + 1 : prevState.losses;
        
        const newDataPoint = { value: parseFloat(totalProfit) };
        const updatedData = [...prevState.data, newDataPoint].slice(-20);
        
        // console.log("Busted data received:", data);
        // console.log("New data point:", newDataPoint);
        // console.log("Updated data:", updatedData);
        this.requestUserInfo();
        return {
          profit: totalProfit,
          wins,
          losses,
          data: updatedData,
          roundWins: 0
        };
      });
    }
  }

  handleWin = () => {
    this.setState(prevState => ({
      roundWins: prevState.roundWins + 1
    }));
  }

  handleError = (error) => {
    console.error("Hilo game error:", error);
  }

  handleClose = (e) => {
    e.preventDefault();
    e.stopPropagation();
    this.props.onClose();
  };

  render() {
    const { profit, wins, losses, data, userData, wagered, roundWins } = this.state;
    const profitNum = parseFloat(profit) || 0;
    const isWinning = profitNum > 0;
    const graphColor = isWinning ? '#67B699' : '#d9534f';

    return (
      <Draggable bounds="parent">
        <PopupWrapper>
          <PopupContainer>
            <Header>
              <Title>Live Stats</Title>
              <CloseButton onClick={this.handleClose}>×</CloseButton>
            </Header>

            <InnerContainer>
              <BetInfo>
                <InfoColumn>
                  <Label>Wagered</Label>
                  <Value>{userData?.profit || '0.00000000'}</Value>
                </InfoColumn>
                <InfoColumn>
                  <Label>Profit</Label>
                  <Value style={{ color: graphColor }}>
                    {profitNum > 0 ? '+' : ''}{forceSatoshiFormat(profitNum)}
                  </Value>
                </InfoColumn>
              </BetInfo>
              
              <GraphContainer>
                <ResponsiveContainer width="100%" height="100%">
                  <AreaChart data={data}>
                    <Area
                      type="linear"
                      dataKey="value"
                      stroke={graphColor}
                      fill={graphColor}
                      fillOpacity={0.6}
                      isAnimationActive={false}
                    />
                  </AreaChart>
                </ResponsiveContainer>
              </GraphContainer>
            </InnerContainer>
            
            <WinLoseInfo>
              <InfoColumn>
                <Label>Win</Label>
                <Value style={{ color: isWinning ? '#67B699' : 'inherit' }}>{wins}</Value>
              </InfoColumn>
              <InfoColumn>
                <Label>Lose</Label>
                <Value style={{ color: !isWinning ? '#d9534f' : 'inherit' }}>{losses}</Value>
              </InfoColumn>
              <InfoColumn>
                <Label>Round Wins</Label>
                <Value style={{ color: '#67B699' }}>{roundWins}</Value>
              </InfoColumn>
            </WinLoseInfo>
          </PopupContainer>
        </PopupWrapper>
      </Draggable>
    );
  }
}

export default BetsPopHilo;