import React, { Component } from "react";
import styled from "styled-components";
import { Play } from "lucide-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faArrowLeft } from "@fortawesome/free-solid-svg-icons";
import { getUID } from "../../../Helper";

// Styled components remain unchanged, so I'll skip them for brevity
const GridWrapper = styled.div`
  display: grid;
  gap: 12px;
  padding: 12px;
  width: 100%;
  grid-template-columns: repeat(3, 1fr);
  @media (min-width: 640px) {
    grid-template-columns: repeat(4, 1fr);
  }
  @media (min-width: 1024px) {
    grid-template-columns: repeat(6, 1fr);
    gap: 16px;
    padding: 16px;
  }
  @media (min-width: 1280px) {
    grid-template-columns: repeat(8, 1fr);
    gap: 20px;
    padding: 20px;
  }
`;

const SkeletonLoader = styled.div`
  width: 100%;
  height: 100%;
  background: linear-gradient(
    90deg,
    rgba(190, 190, 190, 0.2) 25%,
    rgba(129, 129, 129, 0.24) 37%,
    rgba(190, 190, 190, 0.2) 63%
  );
  background-size: 400% 100%;
  animation: skeleton-loading 1.4s ease infinite;
  border-radius: 8px;
  @keyframes skeleton-loading {
    0% { background-position: 100% 50%; }
    100% { background-position: 0% 50%; }
  }
`;

const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.4);
  backdrop-filter: blur(5px);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(255, 255, 255, 0.3);
  display: flex;
  justify-content: center;
  align-items: center;
`;

const FullScreenContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  z-index: 999;
  background-color: #1b2336;
`;

const FullScreenIframe = styled.iframe`
  width: 100%;
  height: 100%;
  border: none;
  display: ${(props) => (props.loading ? "none" : "block")};
`;

const PlayIcon = styled(Play)`
  color: #fff;
  font-size: 24px;
`;

const BackButton = styled.button`
  position: absolute;
  top: 10px;
  left: 10px;
  z-index: 1000;
  background: none;
  border: none;
  color: #fff;
  font-size: 24px;
  cursor: pointer;
  padding: 8px;
  &:hover { opacity: 0.8; }
`;

const LoadMoreButton = styled.button`
  background-color: #03030F;
  color: white;
  border: none;
  padding: 12px 24px;
  border-radius: 8px;
  cursor: pointer;
  font-weight: bold;
  margin: 20px auto;
  display: block;
  transition: background-color 0.3s ease;
  &:hover { background-color: #4a5354; }
  &:disabled {
    background-color: #ccc;
    cursor: not-allowed;
  }
`;

const LoaderContainer = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 20px;
`;

const Spinner = styled.div`
  width: 40px;
  height: 40px;
  border: 4px solid #1c1e22;
  border-top: 4px solid #03030F;
  border-radius: 50%;
  animation: spin 1s linear infinite;
  @keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
  }
`;

const LotteryGameCard = styled.div`
  width: 100%;
  aspect-ratio: 1.32;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;
  &:hover {
    ${GameOverlay} { opacity: 1; }
    transform: translateY(-5px);
  }
`;

const LotteryGameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
`;

class LiveGames extends Component {
  constructor(props) {
    super(props);
    this.state = {
      isLogged: localStorage.getItem("logged") === "true",
      displayGamesEvolutionXgaming: [],
      message: null,
      iframeUrl: "",
      showIframe: false,
      iframeLoading: false,
      isLoadingMore: false,
      evolutionPage: 1,
      loading: true,
      loadedImages: new Set(),
    };
  }

  handleImageLoad = (gameId) => {
    this.setState((prevState) => ({
      loadedImages: new Set(prevState.loadedImages).add(gameId),
    }));
  };

  async componentDidMount() {
    console.log("componentDidMount triggered"); // Debug: Check if this runs
    await this.fetchInitialEvolutionGames();
  }

  fetchVendorData = async ({ vendorName, page = 1, perPage = 24 }) => {
    const baseUrl = "https://api.boss707.com/xGaming/by-vendor";
    const vendorCategories = {
      evolution: [
        "evolution", // Added "evolution" explicitly to test the actual vendor
        "ezugi",
        "redtiger",
        "sagaming",
        "spadegaming",
        "microgaming",
        "playngo",
        "xprogaming",
        "bgaming",
        "conceptgaming",
        "boldplay",
        "vibragaming",
        "fugaso",
        "rivalgames",
        "beterlive",
        "dragoonsoft",
        "boss707",
      ],
    };

    const createUrl = (vendor) =>
      `${baseUrl}?vendor=${vendor}&page=${page}&per_page=${perPage}`;

    const fetchSingleVendor = async (vendor) => {
      const url = createUrl(vendor);
      console.log(`Fetching data from: ${url}`); // Debug: Log the URL
      const token = localStorage.getItem('token');
      try {
        const response = await fetch(url, {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        });
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        console.log(`Response for ${vendor}:`, data); // Debug: Log the response
        return { vendor, data };
      } catch (error) {
        console.error(`Error fetching data for vendor ${vendor}:`, error);
        return { vendor, error: error.message };
      }
    };

    try {
      console.log(`Fetching vendor data for: ${vendorName}`); // Debug
      if (vendorCategories[vendorName]) {
        const vendors = vendorCategories[vendorName];
        const results = await Promise.all(
          vendors.map((vendor) => fetchSingleVendor(vendor))
        );
        console.log("Fetch results:", results); // Debug: Log all results
        return { category: vendorName, results };
      }
      const result = await fetchSingleVendor(vendorName);
      return { category: "single", results: [result] };
    } catch (error) {
      console.error("Error in fetchVendorData:", error);
      throw new Error(`Failed to fetch vendor data: ${error.message}`);
    }
  };

  fetchInitialEvolutionGames = async () => {
    console.log("Fetching initial Evolution games..."); // Debug
    try {
      const response = await this.fetchVendorData({
        vendorName: "evolution",
        page: 1,
        perPage: 24,
      });

      console.log("API Response:", response); // Debug: Log full response

      const evolutionGames = response?.results.reduce((games, vendorObj) => {
        if (
          vendorObj.data &&
          vendorObj.data.success === true &&
          Array.isArray(vendorObj.data.data)
        ) {
          console.log(`Games from ${vendorObj.vendor}:`, vendorObj.data.data); // Debug
          return [...games, ...vendorObj.data.data];
        } else {
          console.log(`No valid data from ${vendorObj.vendor}:`, vendorObj); // Debug
        }
        return games;
      }, []);

      console.log("Processed Evolution Games:", evolutionGames); // Debug

      this.setState({
        displayGamesEvolutionXgaming: evolutionGames,
        loading: false,
      });
    } catch (error) {
      console.error("Error fetching initial evolution games:", error);
      this.setState({
        message: { type: "error", text: "Failed to load games. Please try again later." },
        loading: false,
      });
    }
  };

  loadMoreEvolutionGames = async () => {
    this.setState({ isLoadingMore: true });
    const nextPage = this.state.evolutionPage + 1;

    try {
      const response = await this.fetchVendorData({
        vendorName: "evolution",
        page: nextPage,
        perPage: 24,
      });

      console.log("Load more response:", response); // Debug

      const newEvolutionGames = response?.results.reduce((games, vendorObj) => {
        if (
          vendorObj.data &&
          vendorObj.data.success === true &&
          Array.isArray(vendorObj.data.data)
        ) {
          return [...games, ...vendorObj.data.data];
        }
        return games;
      }, []);

      this.setState((prevState) => ({
        displayGamesEvolutionXgaming: [
          ...prevState.displayGamesEvolutionXgaming,
          ...newEvolutionGames,
        ],
        evolutionPage: nextPage,
        isLoadingMore: false,
      }));
    } catch (error) {
      console.error("Error loading more evolution games:", error);
      this.setState({
        isLoadingMore: false,
        message: {
          type: "error",
          text: "Failed to load more games. Please try again.",
        },
      });
    }
  };

  handleIframeLoad = () => {
    this.setState({ iframeLoading: false });
  };

  handleBackButton = () => {
    this.setState({ showIframe: false, iframeUrl: "", iframeLoading: true });
  };

  handleGameLaunchXGaming = async (game) => {
    if (!this.state.isLogged) {
      this.setState({
        message: { type: "error", text: "Please login to play live game" },
      });
      return;
    }
    const token = localStorage.getItem('token');
    try {
      const response = await fetch("https://api.boss707.com/api/casino/gamerun", {
        method: "POST",
        headers: { "Content-Type": "application/json", "Authorization": `Bearer ${token}` },
        body: JSON.stringify({
          game_id: game.id,
          currency: "USD",
          mode: "real",
          language: "en",
          home_url: "https://api.boss707.com",
          device: "desktop",
          user_id: getUID,
          vendor: game.vendor,
          title: game.title,
          coin: "usdt",
        }),
      });

      const result = await response.json();
      if (result.gameRunUrl) {
        this.setState({
          iframeUrl: result.gameRunUrl,
          showIframe: true,
          iframeLoading: true,
        });
      } else {
        throw new Error("Failed to launch game");
      }
    } catch (error) {
      this.setState({ message: { type: "error", text: error.message } });
    }
  };

  render() {
    const {
      displayGamesEvolutionXgaming,
      showIframe,
      iframeUrl,
      iframeLoading,
      isLoadingMore,
      loading,
      loadedImages,
      message,
    } = this.state;

    console.log("Current games in state:", displayGamesEvolutionXgaming); // Debug

    if (loading) {
      return (
        <LoaderContainer>
          <Spinner />
        </LoaderContainer>
      );
    }

    if (showIframe) {
      return (
        <FullScreenContainer>
          {iframeLoading && (
            <LoaderContainer>
              <Spinner />
            </LoaderContainer>
          )}
          <FullScreenIframe
            src={iframeUrl}
            title="Game Iframe"
            onLoad={this.handleIframeLoad}
            loading={iframeLoading}
          />
          <BackButton onClick={this.handleBackButton}>
            <FontAwesomeIcon icon={faArrowLeft} />
          </BackButton>
        </FullScreenContainer>
      );
    }

    return (
      <>
        {message && (
          <div style={{ color: message.type === "error" ? "red" : "green", textAlign: "center", marginBottom: "20px" }}>
            {message.text}
          </div>
        )}
        <GridWrapper>
          {displayGamesEvolutionXgaming.length > 0 ? (
            displayGamesEvolutionXgaming.map((game) => (
              <LotteryGameCard
                key={game.id}
                onClick={() => this.handleGameLaunchXGaming(game)}
              >
                {!loadedImages.has(game.id) && <SkeletonLoader />}
                <LotteryGameImage
                  src={game.details_thumbnails_300x300}
                  alt={game.title}
                  onLoad={() => this.handleImageLoad(game.id)}
                  style={{ display: loadedImages.has(game.id) ? "block" : "none" }}
                />
                <GameOverlay>
                  <PlayIconContainer>
                    <PlayIcon />
                  </PlayIconContainer>
                </GameOverlay>
              </LotteryGameCard>
            ))
          ) : (
            <div>No Evolution games available at the moment.</div>
          )}
        </GridWrapper>
        {displayGamesEvolutionXgaming.length > 0 && (
          isLoadingMore ? (
            <LoaderContainer>
              <Spinner />
            </LoaderContainer>
          ) : (
            <LoadMoreButton onClick={this.loadMoreEvolutionGames} disabled={isLoadingMore}>
              Load More
            </LoadMoreButton>
          )
        )}
      </>
    );
  }
}

export default LiveGames;