import React from 'react';
import PropTypes from "prop-types";
import {connect} from "react-redux";
import {Button, Col, Dropdown, Row} from "react-bootstrap";
import RangeCredit from "../../Components/Game/Addons";
import {__, Event, forceSatoshiFormat, isValidNumber, wait, sendNotfication, playAudio} from "../../../Helper";
import {gameCoin} from "../../../actions/gameCoin";
import storage from "../../../Storage";
import {setWallet} from "../../../actions/gameWallet";
import C from "../../../Constant";
import styled from "styled-components";
import { Info } from "lucide-react";

// Styled Components (same as Classic Dice)
const Container2 = styled.div`
 background:rgb(18, 20, 24);
  color: white;
  padding: 20px;
  width: 100%;  /* Full width on mobile */
  border-right: 1px solid #2a2a2a;
  border-bottom: 1px solid #2a2a2a;
  border-radius: 0;
  border-top-left-radius: 12px;
  border-bottom-left-radius: 12px;

  /* Desktop size (768px and above) */
  @media (min-width: 768px) {
    max-width: 380px;
  }
`;

const AmountSection = styled.div`
  margin-bottom: 16px;
`;

const Label = styled.div`
  display: flex;
  align-items: center;
  color: #808080;
  margin-bottom: 4px;
  
  svg {
    margin-left: 4px;
    width: 16px;
    height: 16px;
  }
`;

const BetDisplay = styled.div`
  display: flex;
  align-items: center;
  background-color: #292D2E;
  padding: 4px 12px;
  border-radius: 8px;
  margin-bottom: 16px;
  position:relative;
`;

const BetInput = styled.input`
  border: none;
  outline: none;
  color: white;
  background: none;
  flex: 1;
  
  &:focus {
    background: none;
  }
  
  /* Additional browser resets to ensure no background appears */
  &:-webkit-autofill,
  &:-webkit-autofill:hover,
  &:-webkit-autofill:focus,
  &:-webkit-autofill:active {
    -webkit-box-shadow: 0 0 0 30px #1a1a1a inset !important;
    -webkit-text-fill-color: white !important;
    transition: background-color 5000s ease-in-out 0s;
  }
`;

const MultiplierGroup = styled.div`
  margin-left: auto;
  display: flex;
  gap: 8px;
`;

const MultiplierButton = styled.button`
  background-color: #3A4142;
  padding: 4px 12px;
  border-radius: 6px;
  border: none;
  color: white;
  cursor: pointer;
`;

const WinAmountInput = styled.div`
  background-color: #2a2a2a;
  padding: 12px;
  border-radius: 8px;
  display: flex;
  align-items: center;
  margin-bottom: 16px;
`;

const RollButton = styled.div`
  width: 100%;
  padding: 12px;
  background: linear-gradient(90deg, rgb(36, 238, 137), rgb(159, 232, 113));
  border: none;
  border-radius: 8px;
  color: black !important;
  font-weight: bold;
  cursor: pointer;
  display: flex;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    transform: scale(1.02);
  }

  &:focus {
    transform: scale(1.02);
    outline: 2px solid rgb(36, 238, 137);
    outline-offset: 2px;
  }
`;

const WarningContainer = styled.div`
  background-color: rgba(52, 199, 89, 0.1);
  border-radius: 8px;
  padding: 2px 16px;
  display: flex;
  align-items: center;
  gap: 12px;
  margin: 0 auto;
  margin-bottom: 16px;
  margin-top: 8px;
`;

const DemoText = styled.div`
  color: #808080;
  font-size: 14px;
  text-align: center;
`;

const TypeSection = styled.div`
  margin-bottom: 16px;
`;

const TypeDisplay = styled.div`
  display: flex;
  align-items: center;
  background-color: #292D2E;
  padding: 4px 12px;
  border-radius: 8px;
  margin-bottom: 16px;
  position: relative;
`;

const TypeDropdown = styled.div`
  display: flex;
  align-items: center;
  justify-content: space-between;
  width: 100%;
  cursor: pointer;
  color: white;
  position: relative;
`;

const DropdownMenu = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  right: 0;
  background-color: #292D2E;
  border-radius: 8px;
  margin-top: 4px;
  z-index: 1000;
  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
  overflow: hidden;
`;

const DropdownOption = styled.div`
  padding: 12px;
  cursor: pointer;
  color: white;
  transition: background-color 0.2s ease;
  
  &:hover {
    background-color: #3A4142;
  }
  
  &:first-child {
    border-radius: 8px 8px 0 0;
  }
  
  &:last-child {
    border-radius: 0 0 8px 8px;
  }
`;

class Bet extends React.Component {
    _isMounted = false;
    
    constructor (props){
        super(props);
        this.state = {
            engine: this.props.engine,
            isLogged: storage.getKey("logged") !== null ? true : false,
            amount: forceSatoshiFormat(storage.getKey('lam') ? storage.getKey('lam') : .00000010),
            buttonText: 'BET',
            type: 1,
            typeName: 'DEFAULT',
            disabled: false,
            deal: false,
            cards: [],
            dealCards: [],
            winning: false,
            c: 0,
            // Add dropdown state
            showDropdown: false
        };
        this.handleInputChange = this.handleInputChange.bind(this);
        this.handleBet = this.handleBet.bind(this);
        this.dropdownRef = React.createRef();
    }

    componentDidMount(){
        let { engine } = this.state;
        this._isMounted = true;

        if(this._isMounted)
        {
            this.props.gameCoin();
            engine.on(C.PLAY_VIDEOPOKER, () => this.play());
            engine.on(C.BUSTED_VIDEOPOKER, () => this.busted());
            engine.on(C.ERROR_VIDEOPOKER, data => this.error(data));
            engine.on("card", (data) => this.setCards(data));
            engine.on("deal_card", (data) => this.setDealCards(data));
            engine.on("toggle_card_hold", (data) => this.toggleCard(data));
            engine.on("deal", (data) => this.deal(data));
            engine.on("stop_playing", () => this.stopManual());
            
            // Add click outside listener for dropdown
            document.addEventListener('click', this.handleClickOutside);
        }
    }

    componentWillUnmount() {
        let { engine } = this.state;
        engine.init = false;
        this._isMounted = false;
        
        // Remove click outside listener
        document.removeEventListener('click', this.handleClickOutside);
    }

    // Handle click outside dropdown
    handleClickOutside = (event) => {
        if (this.dropdownRef.current && !this.dropdownRef.current.contains(event.target)) {
            this.setState({ showDropdown: false });
        }
    }

    setCards(data) {
        this.setState(state => ({ cards: [data, ...state.cards] }));
    }

    setDealCards(data) {
        this.setState(state => ({ dealCards: [data, ...state.dealCards] }));
    }

    toggleCard(data) {
        let { cards } = this.state;

        for( var i = 0; i < cards.length; i++){
            if (cards[i].pos === data.pos){
                cards[i].hold = data.hold;
            }
        }
    }

    deal(data) {
        let { win, score } = data;
        let { engine, dealCards } = this.state;

        if(win !== 0)
            playAudio('win.mp3');
        else
            playAudio('lost.mp3');

        // Update Table
        engine.emit('update_table', score)

        engine.init = true;
        engine.dealCards = dealCards;
        engine.winning = win;
        engine.deal();
    }
    
    stopManual() {
        if(this._isMounted){
            this.setState({ started: false, buttonText: 'BET', disabled: false });
        }
    }

    error(data) {
        if(this._isMounted)
        {
            this.setState({disabled: false});
        
            sendNotfication(data.message, 'info', 'top-center');

            if(data.code === 'credit')
                this.props.setWallet(true, data.uid);
        }
    }

    play(){
        if(this._isMounted){
            this.setState({ disabled: false, buttonText: "Deal", deal: true });
        }
    }

    busted(){
        if(this._isMounted)
        {
            this.setState({ disabled: true, buttonText: "BET", deal: false, cards: [], dealCards: [] });
            wait(3000).then(() => {
                this.setState({ disabled: false  });
            })
        }
    }

    placeBet(){
        if(this._isMounted)
        {
            let { engine, amount, deal, cards } = this.state;
            wait(100)
                .then(() => {
                    engine.init = true;
                    engine.coin = this.props.coin;
                    engine.amount = amount;
                    engine.cards = cards;

                    if(deal)
                    {
                        engine.emit(C.DEAL_VIDEOPOKER)
                    }
                    else {
                        engine.start();
                    }
                });
        }
    }

    isCorrectBet(){
        if(this._isMounted)
        {
            let { amount, isLogged } = this.state;

           // Check User
            if(!isLogged){
               return Event.emit('showAuthModal', true);
            }

            if(amount.trim() === '')
                return false;

            if(!isValidNumber(amount))
                return false;
            
            
            if(parseFloat(amount) <= 0){
                this.setState({ amount: forceSatoshiFormat(0.00000010) });
            }
            else {
                amount = Math.min(parseFloat(storage.getKey('credit')), parseFloat(amount));
                this.setState({ amount: forceSatoshiFormat(amount) });
            }
            return true;
        }
    }

    handleBet(e){
        if(this._isMounted)
        {
            e.preventDefault();

            // Check is Correct Bet
            if(!this.isCorrectBet())
                return false;

            this.setState({ disabled: true });
            this.placeBet();
        }
    }

    handleInputChange(event){
        if(this._isMounted)
        {
            let target = event.target;
            let value  = target.type === 'checkbox' ? true : target.value;
            let name   = target.name;

            if(__.isEmpty(value)){
                if(name === 'amount')
                    this.setState({ bitError: 'red-border' });
            }
            else {
                if(name === 'amount')
                    this.setState({ bitError: false });
            }

            if(name === 'amount'){
                this.setState({amount:value})
                if(!isValidNumber(value)) return;
            }

            if(name === 'amount'){
                storage.setKey('lam', value);

                // Update Table
                this.state.engine.emit('update_table_amount', value);
            }
            
            this.setState({ [name]: value });
        }
    }

    handleMode = value => {
        if(this._isMounted)
        {
            this.setState({ 
                type: value,
                showDropdown: false // Close dropdown after selection
            });
            let name;
            if(value === '1') name = 'DEFAULT';
            if(value === '2') name = 'HARD';
            this.setState({ typeName: name });
        }
    };

    toggleDropdown = () => {
        this.setState(prevState => ({ showDropdown: !prevState.showDropdown }));
    };
    
    setAmount = (val) => {
        if(this._isMounted)
        {
            var am = forceSatoshiFormat(val);
            this.setState({ amount: am });
            storage.setKey('lam', am );

            // Update Table
            this.state.engine.emit('update_table_amount', am);
        }
    }

    handleMultiplier = (type) => {
        if (this._isMounted) {
            let currentAmount = parseFloat(this.state.amount);
            let newAmount;

            if (type === 'half') {
                newAmount = currentAmount / 2;
            } else if (type === 'double') {
                newAmount = currentAmount * 2;
            }

            // Ensure the amount doesn't go below the minimum bet
            if (this.props.coin === 'INR') {
                newAmount = Math.max(newAmount, 100);
            } else if (this.props.coin === 'USDT') {
                newAmount = Math.max(newAmount, 1);
            } else {
                newAmount = Math.max(newAmount, 0.0000001);
            }

            // Format and update the amount
            const formattedAmount = forceSatoshiFormat(newAmount);
            this.setState({ amount: formattedAmount });
            storage.setKey("lam", formattedAmount);
        }
    };

    render() {
        let {disabled, typeName, amount} = this.state;

        return(
            <Container2>
                <div className="w-100 mt-1">
                    <AmountSection>
                        <Label>
                            Amount
                            <Info />
                        </Label>
                        <BetDisplay>
                            <img
                                src={`/assets/images/${this.props.coin}.png`}
                                className="mini-coin-7"
                                alt=""
                            />
                            <BetInput
                                type="text"
                                disabled={disabled}
                                className="form-control"
                                id="amount" 
                                name="amount"
                                value={amount}
                                autoComplete={"off"}
                                onKeyUp={this.handleInputChange}
                                onChange={this.handleInputChange} 
                            />
                            <MultiplierGroup>
                                <MultiplierButton
                                    onClick={() => this.handleMultiplier('half')}
                                    disabled={disabled}
                                >
                                    1/2
                                </MultiplierButton>
                                <MultiplierButton
                                    onClick={() => this.handleMultiplier('double')}
                                    disabled={disabled}
                                >
                                    2x
                                </MultiplierButton>
                                <div className="input-group-append">
                                    <RangeCredit set={this.setAmount} />
                                </div>
                            </MultiplierGroup>
                        </BetDisplay>
                    </AmountSection>

                    <TypeSection>
                        <Label>
                            Game Type
                            <Info />
                        </Label>
                        <TypeDisplay ref={this.dropdownRef}>
                            <TypeDropdown onClick={this.toggleDropdown}>
                                <span>{typeName}</span>
                                <i className={`mdi mdi-chevron-${this.state.showDropdown ? 'up' : 'down'}`} />
                            </TypeDropdown>
                            {this.state.showDropdown && (
                                <DropdownMenu>
                                    <DropdownOption onClick={() => this.handleMode('1')}>
                                        DEFAULT
                                    </DropdownOption>
                                    <DropdownOption onClick={() => this.handleMode('2')}>
                                        HARD
                                    </DropdownOption>
                                </DropdownMenu>
                            )}
                        </TypeDisplay>
                    </TypeSection>

                    <Label>Win Amount</Label>
                    <WinAmountInput>
                        <img
                            src={`/assets/images/${this.props.coin}.png`}
                            className="mini-coin-7"
                            alt=""
                        />
                        <span style={{ marginLeft: "15px" }}>
                            {forceSatoshiFormat(amount * (this.state.type === '2' ? 2 : 1.5))} {/* Higher payout for HARD mode */}
                        </span>
                    </WinAmountInput>

                    <RollButton 
                        onClick={(e) => {
                            e.preventDefault();
                            if(this.props.coin==='INR'){
                                if(this.state.amount < 100){
                                    sendNotfication('Enter valid amount')
                                }else{
                                    this.handleBet(e);
                                }
                            }
                            else if (this.props.coin ==='USDT'){
                                if(this.state.amount < 1){
                                    sendNotfication('Enter valid amount')
                                }else{
                                    this.handleBet(e);
                                }
                            }else{
                                this.handleBet(e);
                            }
                        }}
                        disabled={disabled}
                    >
                        {this.state.deal ? "Deal" : "BET"}
                    </RollButton>

                    <WarningContainer>
                        <DemoText>
                            {this.props.coin ==='INR' ?
                                "Minimum Bet - 100 INR" :
                                this.props.coin === 'USDT' ?
                                "Minimum Bet - 1 USDT" :
                                "Minimum Bet - 0.0000001"
                            }
                        </DemoText>
                    </WarningContainer>
                </div>
            </Container2>
        );
    }
}

Bet.propTypes = {
    coin: PropTypes.string,
    credit: PropTypes.string
};

const mapStateToProps = state => ({
    coin: state.items.coin,
    credit: state.items.credit
});

export default connect(mapStateToProps, { gameCoin, setWallet })(Bet);
