import React, { Component } from 'react';
import io from 'socket.io-client';

class GameComponent extends Component {
  constructor(props) {
    super(props);
    this.state = {
      namespaces: ['30s', '1m', '2m'],
      currentNamespace: '30s',
      timers: { '30s': 0, '1m': 0, '2m': 0 },
      sessions: { '30s': null, '1m': null, '2m': null },
      bettingOpen: { '30s': true, '1m': true, '2m': true },
      sessionData: { '30s': [], '1m': [], '2m': [] },
      sockets: {},
      selectedValue: 0,
      selectedColor: 'red',
      betAmount: 0,
      betType: 'number'
    };
  }

  componentDidMount() {
    this.state.namespaces.forEach(namespace => {
      this.setupSocket(namespace);
    });
  }

  setupSocket = (namespace) => {
    const socket = io(`https://api.boss707.com/${namespace}`);
    this.setState(prevState => ({
      sockets: {
        ...prevState.sockets,
        [namespace]: socket
      }
    }));

    socket.on('sessionUpdate', (data) => this.handleSessionUpdate(namespace, data));
    socket.on('sessionTimer', (data) => this.handleSessionTimer(namespace, data));
    socket.on('bettingClosed', (data) => this.handleBettingClosed(namespace, data));
    socket.on('winningNumber', (data) => this.handleWinningNumber(namespace, data));
    socket.on('balanceUpdate', (data) => this.handleBalanceUpdate(namespace, data));
    socket.on('betOutcome', (data) => this.handleBetOutcome(namespace, data));
    socket.on('sessionData', (data) => this.handleSessionData(namespace, data));
    socket.on('betResult', (data) => this.handleBetResult(namespace, data));
    socket.on('betError', (data) => this.handleBetError(namespace, data));
  }

  handleSessionUpdate = (namespace, data) => {
    this.setState(prevState => ({
      sessions: {
        ...prevState.sessions,
        [namespace]: data.sessionId
      }
    }));
  }

  handleSessionTimer = (namespace, data) => {
    this.setState(prevState => ({
      timers: {
        ...prevState.timers,
        [namespace]: data.timeLeft
      }
    }));
  }

  handleBettingClosed = (namespace, data) => {
    this.setState(prevState => ({
      bettingOpen: {
        ...prevState.bettingOpen,
        [namespace]: false
      }
    }));
  }

  handleWinningNumber = (namespace, data) => {
    alert(`Winning number for session ${data.sessionId} in ${namespace}: ${data.number} (${data.color})`);
  }

  handleBalanceUpdate = (namespace, data) => {
    alert(`Balance update: ${data.message}, Win Amount: ${data.winAmount}, Coin Type: ${data.cointype}`);
  }

  handleBetOutcome = (namespace, data) => {
    alert(`Bet outcome for session ${data.sessionId} in ${namespace}: Bet Value: ${data.betValue}, Win Amount: ${data.winAmount}`);
  }

  handleSessionData = (namespace, data) => {
    this.setState(prevState => ({
      sessionData: {
        ...prevState.sessionData,
        [namespace]: data
      }
    }));
  }

  handleBetResult = (namespace, data) => {
    alert(`Bet placed successfully in ${namespace}: Bet Value: ${data.betValue}, Amount: ${data.amount}`);
  }

  handleBetError = (namespace, data) => {
    alert(`Error placing bet in ${namespace}: ${data.message}`);
  }

  placeBet = () => {
    const { betType, selectedValue, selectedColor, betAmount, currentNamespace, sockets } = this.state;
    const value = betType === 'number' ? selectedValue : selectedColor;

    if (!value || !betAmount) {
      alert('Please select a value and enter bet amount.');
      return;
    }

    const betData = {
      uid: 'user1', // This should be the actual user ID in a real application
      value,
      amount: parseFloat(betAmount),
      cointype: 'coins',
      type: betType,
    };

    sockets[currentNamespace].emit('placeBet', betData);
  }

  handleNamespaceChange = (namespace) => {
    this.setState({ currentNamespace: namespace });
  }

  handleBetTypeChange = (event) => {
    this.setState({ betType: event.target.value });
  }

  handleValueChange = (event) => {
    this.setState({ selectedValue: event.target.value });
  }

  handleColorChange = (event) => {
    this.setState({ selectedColor: event.target.value });
  }

  handleAmountChange = (event) => {
    this.setState({ betAmount: event.target.value });
  }

  render() {
    const { namespaces, currentNamespace, timers, sessions, bettingOpen, sessionData, betType } = this.state;

    return (
      <div>
        <div id="namespace-buttons">
          {namespaces.map(namespace => (
            <button key={namespace} onClick={() => this.handleNamespaceChange(namespace)}>
              {namespace}
            </button>
          ))}
        </div>

        <h2 id="namespace-timer-title">{`${currentNamespace} Timer`}</h2>
        <p id="session-id">{`Session ID: ${sessions[currentNamespace]}`}</p>
        <p id="time-left">{`Time Left: ${timers[currentNamespace]}s`}</p>
        <p id="betting-status">{`Betting Status: ${bettingOpen[currentNamespace] ? 'Open' : 'Closed'}`}</p>

        <div>
          <h3>Place Your Bet</h3>
          <label>
            Bet Type:
            <select id="bet-type" value={betType} onChange={this.handleBetTypeChange}>
              <option value="number">Number</option>
              <option value="color">Color</option>
            </select>
          </label>
          {betType === 'number' && (
            <label id="number-selection">
              Select Number:
              <select id="selected-value" onChange={this.handleValueChange}>
                {[...Array(10).keys()].map(i => (
                  <option key={i} value={i}>{i}</option>
                ))}
              </select>
            </label>
          )}
          {betType === 'color' && (
            <label id="color-selection">
              Select Color:
              <select id="selected-color" onChange={this.handleColorChange}>
                <option value="red">Red</option>
                <option value="green">Green</option>
                <option value="violet">Violet</option>
              </select>
            </label>
          )}
          <label>
            Bet Amount:
            <input type="number" id="bet-amount" onChange={this.handleAmountChange} />
          </label>
          <button id="place-bet" onClick={this.placeBet}>Place Bet</button>
        </div>

        <div>
          <h3>Previous Sessions Data</h3>
          <table>
            <thead>
              <tr>
                <th>Session ID</th>
                <th>Status</th>
                <th>Winning Number</th>
                <th>Winning Color</th>
              </tr>
            </thead>
            <tbody id="session-data">
              {sessionData[currentNamespace].map(session => (
                <tr key={session.sessionId}>
                  <td>{session.sessionId}</td>
                  <td>{session.status}</td>
                  <td>{session.betNumber}</td>
                  <td>{session.betColour}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    );
  }
}

export default GameComponent;
