import styled, { keyframes } from "styled-components";
import { Search, X, Play, ChevronDown, Trash2 } from "lucide-react";
import { useState, useEffect, useRef } from "react";
import { useHistory } from "react-router-dom";
import { GameProvider, useGames } from "../../../Context/GameContext";

const Container = styled.div`
  width: 100%;
  max-width: 1200px;
  max-height: 90vh;
  background: #03030F;
  color: #E0E6E9;
  border-radius: 12px;
  overflow-y: auto;
  position: relative;
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;

  @media (max-width: 768px) {
    max-height: 100vh;
    overflow-x: hidden;
    border-radius: 0;
  }
`;

const CloseButton = styled.button`
  background: rgba(30, 35, 40, 0.3);
  border: none;
  color: #E0E6E9;
  width: 25px;
  height: 25px;
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  margin: 4px 12px;
  z-index: 10;
  transition: all 0.2s ease;

  &:hover {
    background: rgba(30, 35, 40, 0.8);
    transform: rotate(90deg);
  }
`;

const GridContainer = styled.div`
  display: grid;
  grid-template-columns: repeat(3, 1fr);
  width: 100%;
  gap: 16px;
  padding: 0px;
`;

const Wrapper = styled.div`
  margin: 0 auto;
  padding: 1rem;
  height: 100vh;
`;

const Header = styled.div`
  background: rgba(30, 35, 40, 0.4);
  border-radius: 0.5rem;
  padding: 1px;
  margin-top: -12px;
  margin-bottom: 1rem;
`;

const SearchBar = styled.div`
  display: flex;
  align-items: center;
  gap: 0.5rem;
`;

const SearchInput = styled.div`
  flex: 1;
  position: relative;

  input {
    width: 100%;
    background: rgba(30, 35, 40, 0.4);
    padding: 9px 9px 9px 40px;
    border: none;
    border-radius: 8px;
    color: #E0E6E9;
    font-size: 14px;
    
    &::placeholder {
      color: rgba(224, 230, 233, 0.6);
    }
    
    &:focus {
      outline: none;
      background: rgba(30, 35, 40, 0.6);
    }
  }
`;

const SearchIconWrapper = styled.div`
  position: absolute;
  left: 12px;
  top: 50%;
  transform: translateY(-50%);
  color: #E0E6E9;
  display: flex;
  align-items: center;
  justify-content: center;
`;

const slideIn = keyframes`
  from {
    opacity: 0;
    transform: translateY(-10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
`;

const ProviderDropdown = styled.div`
  position: relative;
  width: 200px;
`;

const ProviderButton = styled.button`
  width: 100%;
  background: rgb(30, 35, 40);
  padding: 12px;
  border: none;
  border-radius: 8px;
  color: #E0E6E9;
  font-size: 14px;
  font-weight: 500;
  display: flex;
  align-items: center;
  justify-content: space-between;
  cursor: pointer;
  transition: all 0.3s ease;
  box-shadow: 0 2px 6px rgba(30, 35, 40, 0.3);

  &:hover {
    background: linear-gradient(145deg, #3A4552, #2B343F);
    transform: scale(1.02);
    box-shadow: 0 4px 12px rgba(30, 35, 40, 0.4);
  }

  &:focus {
    outline: none;
    box-shadow: 0 0 0 3px rgba(59, 70, 82, 0.5);
  }
`;

const ProviderMenu = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  width: 320px;
  background: #03030F;
  border-radius: 8px;
  box-shadow: 0 6px 16px rgba(0, 0, 0, 0.5);
  z-index: 1001;
  max-height: 400px;
  overflow-y: auto;
  display: ${(props) => (props.$isOpen ? "block" : "none")};
  margin-top: 4px;
  padding: 12px;
  animation: ${slideIn} 0.2s ease-out;
`;

const SearchInputContainer = styled.div`
  position: relative;
  margin-bottom: 12px;
  
  input {
    width: 100%;
    background: rgba(30, 35, 40, 0.4);
    padding: 10px 10px 10px 36px;
    border: none;
    border-radius: 8px;
    color: #E0E6E9;
    font-size: 14px;

    &::placeholder {
      color: rgba(224, 230, 233, 0.6);
    }

    &:focus {
      outline: none;
      background: rgba(30, 35, 40, 0.6);
    }
  }
`;

const ProviderList = styled.div`
  display: flex;
  flex-direction: column;
  gap: 14px;
  margin-bottom: 16px;
`;

const ProviderItem = styled.div`
  display: flex;
  align-items: center;
  gap: 10px;
  cursor: pointer;
`;

const CheckboxContainer = styled.div`
  width: 22px;
  height: 22px;
  border: 2px solid ${props => props.checked ? '#F89C03' : '#3A4552'};
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  background: ${props => props.checked ? '#F89C03' : 'transparent'};
  transition: all 0.2s ease;

  &::after {
    content: ${props => props.checked ? '" "' : 'none'};
    width: 12px;
    height: 12px;
    background: #03030F;
    border-radius: 50%;
    transition: all 0.2s ease;
  }
`;

const CheckboxInput = styled.input`
  opacity: 0;
  position: absolute;
  cursor: pointer;
`;

const ProviderLogo = styled.div`
  display: flex;
  align-items: center;
  justify-content: flex-start;
  flex: 1;
`;

const Logo = styled.div`
  color: #FFFFFF;
  font-size: 14px;
  font-weight: 500;
`;

const ClearAllButton = styled.button`
  width: 100%;
  padding: 10px;
  background-color: transparent;
  color: #F89C03;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 8px;
  font-size: 14px;

  &:hover {
    background-color: rgba(46, 231, 134, 0.1);
  }
`;

const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(30, 35, 40, 0.5);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const GridWrapper = styled.div`
  display: grid;
  grid-template-columns: repeat(7, 1fr);
  gap: 12px;
  min-height: 80vh;
  @media (max-width: 767px) {
    grid-template-columns: repeat(3, 1fr);
  }
`;

const GameCard = styled.div`
  min-width: 142px;
  height: 195px;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    ${GameOverlay} {
      opacity: 1;
    }
    transform: translateY(-5px);
  }

  @media (max-width: 768px) {
    min-width: 106px;
    height: 160px;
  }
`;

const GameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
  display: ${props => (props.$isLoading ? 'none' : 'block')};
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(224, 230, 233, 0.2);
  display: flex;
  justify-content: center;
  align-items: center;
`;

const PlayIcon = styled(Play)`
  color: #E0E6E9;
  font-size: 24px;
`;

const NoResultsWrapper = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 40px 20px;
  text-align: center;
  color: rgba(224, 230, 233, 0.8);
`;

const NoResultsText = styled.h3`
  font-size: 18px;
  font-weight: 500;
  margin-bottom: 8px;
`;

const NoResultsSubText = styled.p`
  font-size: 14px;
  color: rgba(224, 230, 233, 0.6);
`;

const spinAnimation = keyframes`
  0% { transform: rotate(0deg); }
  100% { transform: rotate(360deg); }
`;

const LoadingWrapper = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 40px 20px;
  text-align: center;
  min-height: 70vh;
`;

const Spinner = styled.div`
  width: 40px;
  height: 40px;
  border: 3px solid rgba(224, 230, 233, 0.1);
  border-radius: 50%;
  border-top-color: #E0E6E9;
  animation: ${spinAnimation} 0.8s linear infinite;
  margin-bottom: 16px;
`;

const LoadingText = styled.p`
  font-size: 16px;
  color: rgba(224, 230, 233, 0.8);
`;

const SkeletonLoader = styled.div`
  width: 100%;
  height: 100%;
  background: rgba(224, 230, 233, 0.1);
  border-radius: 8px;
`;

const LoadMoreButton = styled.button`
  width: 200px;
  padding: 10px;
  background-color: #3A4552;
  color: #E0E6E9;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  margin: 20px auto;
  display: ${props => (props.$show ? 'block' : 'none')};

  &:hover {
    background-color: #2B343F;
  }
`;

const GlobalSearchMobile = () => {
  const { fetchGamesForVendors, fetchGamesFromAPI, handleGameLaunchXGaming, isLoading, searchResults, vendorDisplayNames, searchGames, loadMoreGames, currentPage } = useGames();
  const [searchQuery, setSearchQuery] = useState('');
  const [displayGames, setDisplayGames] = useState([]);
  const [imageLoadingStates, setImageLoadingStates] = useState({});
  const [selectedVendors, setSelectedVendors] = useState([]);
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [providerSearchQuery, setProviderSearchQuery] = useState('');
  const history = useHistory();
  const dropdownRef = useRef(null);
  const prevSearchQuery = useRef('');
  const debounceTimeout = useRef(null);

  const vendorCategories = Object.keys(vendorDisplayNames).map(key => ({
    key,
    displayName: vendorDisplayNames[key],
  }));

  const allVendorKeys = vendorCategories.map(v => v.key);

  // Handle clicks outside dropdown
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
        setIsDropdownOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  // Sync displayGames with searchResults based on user interaction
  useEffect(() => {
    if (searchResults.length > 0) {
      setDisplayGames(searchResults);
    }
  }, [searchResults]);

  // Initialize with default games if no interaction
  useEffect(() => {
    if (searchResults.length === 0 && selectedVendors.length === 0 && !searchQuery.trim()) {
      setDisplayGames(searchResults); // Use initialSearchResults from GameProvider
    }
  }, [searchResults, selectedVendors, searchQuery]);

  const handleSearch = (e) => {
    const query = e.target.value;
    setSearchQuery(query);

    if (debounceTimeout.current) {
      clearTimeout(debounceTimeout.current);
    }

    debounceTimeout.current = setTimeout(() => {
      if (query !== prevSearchQuery.current) {
        prevSearchQuery.current = query;
        if (query.trim()) {
          searchGames(query, selectedVendors.length > 0 ? selectedVendors : allVendorKeys);
        } else if (selectedVendors.length > 0) {
          fetchGamesForVendors(selectedVendors, 1);
        } else {
          fetchGamesFromAPI('', allVendorKeys, 1);
        }
      }
    }, 500);
  };

  const handleVendorChange = (vendorKey) => {
    setSelectedVendors(prev => {
      const newVendors = prev.includes(vendorKey)
        ? prev.filter(v => v !== vendorKey)
        : [...prev, vendorKey];
      
      if (newVendors.length > 0) {
        fetchGamesForVendors(newVendors, 1);
      } else if (!searchQuery.trim()) {
        fetchGamesFromAPI('', allVendorKeys, 1); // Revert to all vendors if no query
      }
      return newVendors;
    });
  };

  const clearAllVendors = () => {
    setSelectedVendors([]);
    if (!searchQuery.trim()) {
      fetchGamesFromAPI('', allVendorKeys, 1); // Revert to all vendors if no query
    }
  };

  const handleImageLoad = (gameId) => {
    setImageLoadingStates(prev => ({
      ...prev,
      [gameId]: false,
    }));
  };

  const filteredProviders = vendorCategories.filter(
    provider => provider.displayName.toLowerCase().includes(providerSearchQuery.toLowerCase())
  );

  const renderContent = () => {
    if (isLoading && displayGames.length === 0) {
      return (
        <LoadingWrapper>
          <Spinner />
          <LoadingText>Loading games...</LoadingText>
        </LoadingWrapper>
      );
    }

    if (displayGames.length > 0) {
      return (
        <div>
          <GridWrapper>
            {displayGames.map((game, index) => {
              const gameId = game.id || game.title || index;
              const isImageLoading = imageLoadingStates[gameId] ?? true;

              return (
                <GameCard
                  key={gameId}
                  onClick={() => handleGameLaunchXGaming(game)}
                >
                  {isImageLoading && <SkeletonLoader />}
                  <GameImage 
                    src={game.game_icon} 
                    alt={`Game ${game.title || index}`} 
                    $isLoading={isImageLoading}
                    onLoad={() => handleImageLoad(gameId)}
                    onError={() => handleImageLoad(gameId)}
                  />
                  <GameOverlay>
                    <PlayIconContainer>
                      <PlayIcon />
                    </PlayIconContainer>
                  </GameOverlay>
                </GameCard>
              );
            })}
          </GridWrapper>
          <LoadMoreButton $show={true} onClick={loadMoreGames}>Load More</LoadMoreButton>
        </div>
      );
    }

    return (
      <NoResultsWrapper>
        <NoResultsText>No games found</NoResultsText>
        <NoResultsSubText>
          Select providers or enter a search term to find games
        </NoResultsSubText>
      </NoResultsWrapper>
    );
  };

  const dropdownLabel = selectedVendors.length === vendorCategories.length
    ? 'Providers: ALL'
    : selectedVendors.length > 0
    ? `Providers: +${selectedVendors.length}`
    : 'Select Providers';

  useEffect(() => {
    return () => {
      if (debounceTimeout.current) {
        clearTimeout(debounceTimeout.current);
      }
    };
  }, []);

  return (
    <GameProvider searchQuery={searchQuery}>
      <Container>
        <GridContainer>
          <div></div>
          <div style={{ textAlign: 'center', marginTop: '8px', fontSize: '20px', textWrap: 'nowrap' }}>Most Popular Games</div>
          <div style={{ marginTop: '6px', marginRight: '10px', width: '100%', display: 'flex', justifyContent: 'flex-end' }}>
            <CloseButton onClick={() => history.goBack()}>
              <X size={20} />
            </CloseButton>
          </div>
        </GridContainer>
        <Wrapper>
          <Header>
            <SearchBar>
              <ProviderDropdown ref={dropdownRef}>
                <ProviderButton onClick={() => setIsDropdownOpen(!isDropdownOpen)}>
                  {/* <span>{dropdownLabel}</span> */}
                  <span>Providers</span>
                  <ChevronDown size={18} />
                </ProviderButton>
                <ProviderMenu $isOpen={isDropdownOpen}>
                  <SearchInputContainer>
                    <SearchIconWrapper>
                      <Search size={18} color="#E0E6E9" opacity={0.7} />
                    </SearchIconWrapper>
                    <input
                      type="text"
                      placeholder="Search providers..."
                      value={providerSearchQuery}
                      onChange={(e) => setProviderSearchQuery(e.target.value)}
                    />
                  </SearchInputContainer>
                  <ProviderList>
                    {filteredProviders.map((vendor) => (
                      <ProviderItem key={vendor.key} onClick={() => handleVendorChange(vendor.key)}>
                        <CheckboxContainer checked={selectedVendors.includes(vendor.key)}>
                          <CheckboxInput 
                            type="checkbox" 
                            checked={selectedVendors.includes(vendor.key)}
                            onChange={() => {}}
                          />
                        </CheckboxContainer>
                        <ProviderLogo>
                          <Logo>{vendor.displayName}</Logo>
                        </ProviderLogo>
                      </ProviderItem>
                    ))}
                  </ProviderList>
                  <ClearAllButton onClick={clearAllVendors}>
                    <Trash2 size={16} color="#F89C03" />
                    Clear All
                  </ClearAllButton>
                </ProviderMenu>
              </ProviderDropdown>
              <SearchInput>
                <SearchIconWrapper>
                  <Search size={18} color="#E0E6E9" opacity={0.7} />
                </SearchIconWrapper>
                <input
                  type="text"
                  placeholder="Search games..."
                  value={searchQuery}
                  onChange={handleSearch}
                />
              </SearchInput>
            </SearchBar>
          </Header>
          {renderContent()}
        </Wrapper>
      </Container>
    </GameProvider>
  );
};

export default GlobalSearchMobile;