import React, { useState, useEffect, useLayoutEffect, useCallback, useContext } from "react";
import { Link, useHistory, useLocation } from 'react-router-dom';
import styled from "styled-components";
import Login from "../../Auth/Login";
import Register from "../../Auth/Register";
import SkylaMenu from "../../../../Icons/Menu";
import MenuPlayIcon from "../../../../Icons/Menu";
import SearchStarIcon from "../../../../Icons/Search";
import MessageIcon from "../../../../Icons/Chat";
import GlobeIcon from "../../../../Icons/Glob";
import SkylaSidebar from "../../../Components/SkylaSiebar";
import { useSidebar } from "../../../../context/SidebarContext";
import User from "../../Parts/User/index.js";
import SkylaProfile from "../SkylaHeaderComponents/SkylaUserProfile.js";
import ProfileMenu from "../SkylaHeaderComponents/SkylaUserProfile.js";
import Credit from './../Credit';
import SkylaCredit from "../SkylaCredit/index.js";
import NotificationsSkyla from "../../../../Icons/Notifications.js";
import GiftSkyla from "../../../../Icons/GiftSkyla.js";
import BottomNavigation from "../../../Components/SkylaBottomTab/index.js";
import { addNewStyle, CRISTP_ID, getUID, wait } from "../../../../Helper.js";
import { useSiteConfig } from "../../../../hooks/useSiteConfig.js";
import { cleanupNotifications, getToken, onMessageListener } from '../../../../services/firebase.js';
import axios from 'axios';
import { NotificationContext, useNotification } from "../../../../context/notificationcontext";
import NotificationDropdown from "../../../Components/NotificationDropdown/index.js";
import GlobalSearchModal from "../GlobalSearch/index.js";
import { useEvent } from "../../../../contexts/EventContext.js";

const DEFAULT_AVATAR_PATH = "//img2.distributedresourcestorage.com/avatar/57268122/s?t=0";

const HeaderWrapper = styled.div`
  width: 100%;
  background: #03030F;
  border-bottom: 1px solid #2C2F36;
  height: 63px;
  padding: 12px 16px;
  position: sticky;
  top: 0;
  z-index: 1000;
  box-shadow: 
    0 10px 15px -3px rgba(0, 0, 0, 0.15), 
    0 4px 6px -2px rgba(0, 0, 0, 0.1),
    0 2px 4px -1px rgba(0, 0, 0, 0.05);
`;

const HeaderContainer = styled.header`
  display: flex;
  align-items: center;
  justify-content: space-between;
`;

const LeftSection = styled.div`
  display: flex;
  align-items: center;
  gap: 8px;
`;

const LogoContainer = styled(Link)`
  display: flex;
  align-items: center;
`;

const Logo = styled.img`
  width: ${props => props.showFull ? '100px' : '96px'};
  height: auto;
  transition: width 0.3s ease;
  object-fit: contain;
`;

const Logo1 = styled.img`
  width: 40px;
  height: auto;
  transition: width 0.3s ease;
`;

const SearchBarContainer = styled.div`
  display: none;
  flex: 1;
  max-width: 576px;
  margin: 0 16px;
  
  @media (min-width: 768px) {
    display: flex;
  }
`;

const SearchBarWrapper = styled.div`
  position: relative;
  width: 100%;
`;

const SearchInputContainer = styled.div`
  display: flex;
  align-items: center;
  background-color: #1B1E22;
  border-radius: 9999px;
  padding: 8px 16px;
  transition: all 0.2s ease;

  &:focus-within {
    border: 1px solid #4CAF50;
  }
`;

const SearchIcon = styled.img`
  width: 16px;
  height: 16px;
  opacity: 0.6;
`;

const SearchInput = styled.input`
  width: 100%;
  background: transparent;
  border: none;
  color: white;
  font-size: 14px;
  margin-left: 8px;
  outline: none;

  &::placeholder {
    color: #9CA3AF;
  }
`;

const RightSection = styled.div`
  display: flex;
  align-items: center;
  gap: 12px;
`;

const IconButton = styled.button`
  padding: 8px;
  background: transparent;
  border: none;
  border-radius: 9999px;
  cursor: pointer;
  transition: background-color 0.2s ease;

  &:hover {
    background-color: #2C2F36;
  }
`;

const Icon = styled.img`
  width: 20px;
  height: 20px;
`;

const AuthButtons = styled.div`
  display: flex;
  align-items: center;
  gap: 8px;
`;

const SignInButton = styled.button`
  padding: 6px 16px;
  background: transparent;
  color: white;
  font-size: 14px;
  font-weight: 500;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  transition: background-color 0.2s ease;

  &:hover {
    background-color: #2C2F36;
  }
`;

const SignUpButton = styled.button`
  padding: 6px 16px;
  background-color: #10B981;
  color: white;
  font-size: 14px;
  font-weight: 500;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  transition: background-color 0.2s ease;

  &:hover {
    background-color: #059669;
  }
`;

const IconBg = styled.div`
  padding: 6px 8px;
  background-color: rgb(18, 20, 24);
  border-radius: 10px;
`;

const IconBg2 = styled.div`
  display: none;

  @media (max-width: 768px) {
    display: block;
    padding: 6px 8px;
    background-color: rgb(18, 20, 24);
    border-radius: 10px;
  }
`;

const IconBg1 = styled.div`
  padding: 6px 8px;
  background-color: rgb(18, 20, 24);
  border-radius: 10px;
  display: flex;
  align-items: center;
  gap: 4px;
  cursor: pointer;
`;

const MessageIconWrapper = styled.div`
  cursor: pointer;
  display: flex;
  align-items: center;
`;

const ProfileButton = styled(Link)`
  background: transparent;
  border: none;
  cursor: pointer;
  padding: 8px;
`;

const ProfileImage = styled.img`
  height: 30px;
  width: 30px;
  border-radius: 50%;
  object-fit: contain;
`;

const ResponsiveSidebarWrapper = styled.div`
  @media (max-width: 768px) {
    display: none;
  }
`;

const Header = () => {
  // Removed debug console.log to clean up console output
  
  const [show, setShow] = useState(true);
  const [isMobile, setIsMobile] = useState(window.innerWidth <= 768);
  const [showLogo, setShowLogo] = useState(window.innerWidth > 768);
  const { isCollapsed, setIsCollapsed } = useSidebar();
  const [loaded, setLoaded] = useState(false);
  const [isBonusEnabled, setIsBonusEnabled] = useState(false);
  const { siteConfig } = useSiteConfig();
  const { showNotification } = useContext(NotificationContext);
  const [notifications, setNotifications] = useState({ unread: [], read: [] });
  const [isLoading, setIsLoading] = useState(false);
  const [isOpenSearch, setIsOpenSearch] = useState(false);
  const [unreadCount, setUnreadCount] = useState(0);
  const { avatarUrl } = useEvent();
  const history = useHistory();

  const location = useLocation();
  const isSportsRoute = () => {
    const path = location.pathname;
    return path.startsWith('/sports/') || path.startsWith('/match/');
  };
const getAuthToken = () => {
  return localStorage.getItem('token');
};
const fetchUnreadCount = async () => {
  const token = getAuthToken();
  if (!token) return; // Don't proceed if token is null

  try {
    const response = await fetch(`https://api.boss707.com/firebase/unread-count/${getUID()}`, {
      method: 'GET',
      headers: {
        Authorization: `Bearer ${token}`,
        'Content-Type': 'application/json',
      },
    });

    const data = await response.json();
    if (data.success && typeof setUnreadCount === 'function') {
      setUnreadCount(data.unreadCount);
    }
  } catch (error) {
    console.error('Failed to fetch unread count:', error);
  }
};

const fetchNotifications = async () => {
  const token = getAuthToken();
  if (!token) return; // Don't proceed if token is null

  setIsLoading(true);
  try {
    const response = await fetch(`https://api.boss707.com/firebase/history/${getUID()}`, {
      method: 'GET',
      headers: {
        Authorization: `Bearer ${token}`,
        'Content-Type': 'application/json',
      },
    });

    const data = await response.json();
    if (data.success) {
      setNotifications({
        unread: data.notifications.unread || [],
        read: data.notifications.read || []
      });
    }

    await fetchUnreadCount(); // This function also checks for token
  } catch (error) {
    console.error('Failed to fetch notifications:', error);
  } finally {
    setIsLoading(false);
  }
};


  const markAsRead = async (notificationId) => {
    const token = localStorage.getItem('token');
    try {
      const response = await fetch('https://api.boss707.com/firebase/mark-as-read', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify({
          userId: getUID(),
          notificationIds: [notificationId]
        })
      });

      if (response.ok) {
        setNotifications(prev => {
          const updatedUnread = prev.unread.filter(n => n.id !== notificationId);
          const movedNotification = prev.unread.find(n => n.id === notificationId);
          return {
            unread: updatedUnread,
            read: movedNotification ? [...prev.read, { ...movedNotification, is_read: true }] : prev.read
          };
        });
        await fetchNotifications();
      }
    } catch (error) {
      console.error('Failed to mark notification as read:', error);
    }
  };

  const markAllAsRead = async () => {
    try {
      const unreadIds = notifications.unread.map(notification => notification.id);
      const token = localStorage.getItem('token');
      const response = await fetch('https://api.boss707.com/firebase/mark-as-read', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify({
          userId: getUID(),
          notificationIds: unreadIds
        })
      });

      if (response.ok) {
        setNotifications(prev => ({
          unread: [],
          read: [...prev.read, ...prev.unread.map(n => ({ ...n, is_read: true }))]
        }));
        await fetchNotifications();
      }
    } catch (error) {
      console.error('Failed to mark all notifications as read:', error);
    }
  };

  useEffect(() => {
    fetchNotifications();
    fetchUnreadCount();
  }, []);

  useEffect(() => {
    // console.log('🏠 Header component mounted');
    
    // Check login state and credit data
    const isLoggedIn = localStorage.getItem('logged') === 'true';
    const storedCredit = localStorage.getItem('credit');
    const token = localStorage.getItem('token');
    
    // console.log('🔍 Header mount check:', {
    //   isLoggedIn,
    //   storedCredit,
    //   hasToken: !!token
    // });
    
    if (isLoggedIn && storedCredit) {
      // console.log('💰 Header found stored credit:', storedCredit);
    }
  }, []);

  let notificationCounter = 0;
  const generateUniqueId = () => {
    notificationCounter += 1;
    return `${Date.now()}-${notificationCounter}`;
  };

  const setupNotifications = async () => {
    let unsubscribeFromMessages = null;

    try {
      if (!('Notification' in window)) {
        console.error('This browser does not support notifications');
        return () => {};
      }

      if ('serviceWorker' in navigator) {
        try {
          const registration = await navigator.serviceWorker.register('/firebase-messaging-sw.js');
          // console.log('Service Worker registered:', registration);
        } catch (error) {
          console.error('Service Worker registration failed:', error);
        }
      }

      const token = await getToken();
      // console.log('fcm', token);
      const bearerToken = getAuthToken();
      await axios.post(
  'https://api.boss707.com/firebase/register',
  {
    userId: getUID(),
    token,
    deviceType: navigator.userAgent,
    platform: 'web',
  },
  {
    headers: {
      Authorization: `Bearer ${bearerToken}`,
      'Content-Type': 'application/json',
    },
  }
);

      unsubscribeFromMessages = onMessageListener((payload) => {
        // console.log('Received message payload:', payload);

        try {
          if (Notification.permission === 'granted') {
            const browserNotification = new Notification(payload.notification.title, {
              body: payload.notification.body,
              icon: './assets/boss707.png"',
              tag: 'notification-' + generateUniqueId(),


            });

            browserNotification.onclick = (event) => {
              event.preventDefault();
              window.focus();
              browserNotification.close();
            };
          }

          showNotification({
            title: payload.notification.title || 'New Notification',
            body: payload.notification.body || '',
          });
        } catch (error) {
          console.error('Notification creation failed:', error);
        }
      });

    } catch (error) {
      console.error('Notification setup failed:', error);
      console.error('Error details:', error.message);
    }

    return () => {
      if (unsubscribeFromMessages) {
        unsubscribeFromMessages();
      }
    };
  };

  useEffect(() => {
    let unsubscribe;
    const setup = async () => {
      unsubscribe = await setupNotifications();
    };
    setup();

    return () => {
      if (unsubscribe) unsubscribe();
      cleanupNotifications();
    };
  }, []);

  useEffect(() => {
    const handleResize = () => {
      const currentWidth = window.innerWidth;
      setIsMobile(currentWidth <= 768);
      setShowLogo(currentWidth > 768);
    };

    window.addEventListener('resize', handleResize);
    return () => {
      window.removeEventListener('resize', handleResize);
    };
  }, []);

  useEffect(() => {
    const checkBonusStatus = async () => {
      const response = await new Promise(resolve =>
        setTimeout(() => resolve({ isEnabled: true }), 1000)
      );
      setIsBonusEnabled(response.isEnabled);
    };

    checkBonusStatus();
  }, []);

  const handleCollapse = useCallback(() => {
    setIsCollapsed(prev => !prev);
  }, [setIsCollapsed]);

  // const handleSupportClick = useCallback((e) => {
  //   e.preventDefault();
  //   e.stopPropagation();
  //   console.log("Support clicked");
  //   window.$crisp = [];
  //   window.CRISP_WEBSITE_ID = CRISTP_ID;

  //   if (!loaded) {
  //     console.log("Loading support script");
  //     setLoaded(true);
  //     const script = document.createElement('script');
  //     script.src = "https://client.crisp.chat/l.js";
  //     script.id = "chtt";
  //     script.async = true;
  //     document.head.appendChild(script);
  //   }

  //   addNewStyle("#crisp-chatbox {display:block !important;}");
  //   addNewStyle(
  //     ".crisp-client .cc-kv6t .cc-1xry .cc-ge4v .cc-qqzz .cc-5b24 .cc-19cg {display:none !important;}"
  //   );

  //   wait(2000).then(() => {
  //     const close = document.querySelector(".cc-wdhl");
  //     if (close === null) return;
  //     close.addEventListener("click", function (event) {
  //       addNewStyle("#crisp-chatbox {display:none !important;}");
  //       document.getElementById("chtt").src = "";
  //     });
  //   });
  // }, [loaded]);
  const handleSupportClick = useCallback((e) => {
    e.preventDefault();
    e.stopPropagation();
    console.log("Support clicked");
    window.$crisp = [];
    window.CRISP_WEBSITE_ID = CRISTP_ID;
    
    if (!loaded) {
      console.log("Loading support script");
      setLoaded(true);
      const script = document.createElement('script');
      script.src = "https://client.crisp.chat/l.js";
      script.id = "chtt";
      script.async = true;
      document.head.appendChild(script);
      
      // Add this to open the chat window directly
      window.$crisp.push(["do", "chat:open"]);
    } else {
      // If already loaded, just open the chat window
      window.$crisp.push(["do", "chat:open"]);
    }
    
    addNewStyle("#crisp-chatbox {display:block !important;}");
    addNewStyle(
      ".crisp-client .cc-kv6t .cc-1xry .cc-ge4v .cc-qqzz .cc-5b24 .cc-19cg {display:none !important;}"
    );
    
    wait(2000).then(() => {
      const close = document.querySelector(".cc-wdhl");
      if (close === null) return;
      close.addEventListener("click", function (event) {
        addNewStyle("#crisp-chatbox {display:none !important;}");
        document.getElementById("chtt").src = "";
      });
    });
  }, [loaded]);
  return (
    <div style={{ display: "flex" }}>
        {
          !isSportsRoute() && (
            <ResponsiveSidebarWrapper>
              <SkylaSidebar isCollapsed={isCollapsed} />
            </ResponsiveSidebarWrapper>
          )
        }

      <HeaderWrapper>
        <HeaderContainer>
          <LeftSection>
            {!isMobile && (
              <IconBg2 onClick={handleCollapse}>
                <MenuPlayIcon />
              </IconBg2>
            )}
            {isMobile ? (
              <LogoContainer to="/">
                <Logo1
                  src="./assets/boss707small.png"
                  alt="Logo"
                  $showFull={showLogo}
                />
              </LogoContainer>
            ) : (
              <LogoContainer to="/">
                <Logo
                  src="./assets/boss707.png"
                  alt="Logo"
                  $showFull={showLogo}
                />
              </LogoContainer>
            )}
          </LeftSection>
          <RightSection>
            {!isMobile && (
              <IconBg onClick={() => setIsOpenSearch(!isOpenSearch)}>
                <SearchStarIcon />
              </IconBg>
            )}
            <GlobalSearchModal isOpen={isOpenSearch} onClose={() => setIsOpenSearch(false)} />
            <SkylaCredit />
            <IconBg onClick={() => history.push("/bonus")}>
              <GiftSkyla />
            </IconBg>
            {!isMobile && (
              <IconBg1>
                <MessageIconWrapper onClick={handleSupportClick}>
                  <MessageIcon />
                </MessageIconWrapper>
                <NotificationDropdown
                  notifications={notifications}
                  onMarkAsRead={markAsRead}
                  onMarkAllAsRead={markAllAsRead}
                  unreadCount={unreadCount}
                  isLoading={isLoading}
                />
              </IconBg1>
            )}
            {!isMobile ? (
              <ProfileMenu />
            ) : (
              <ProfileButton to="/mobile-user">
                <ProfileImage
                  src={avatarUrl || DEFAULT_AVATAR_PATH}
                  alt="Profile"
                  onError={(e) => {
                    e.target.onerror = null;
                    e.target.src = DEFAULT_AVATAR_PATH;
                  }}
                />
              </ProfileButton>
            )}
          </RightSection>
        </HeaderContainer>
      </HeaderWrapper>
    </div>
  );
};

export default Header;