import React, { Component } from 'react';
import { withRouter } from 'react-router-dom';
import styled from 'styled-components';
import { ChevronLeft, ChevronRight, Play } from "lucide-react";
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faArrowLeft } from '@fortawesome/free-solid-svg-icons';
import { getUID, sendNotfication } from '../../../../../Helper';

// Styled Components (unchanged, omitted for brevity)
// ... [Keep all your styled components as they are]
const HeaderContainer = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-bottom: 16px;
  padding: 0 10px;
`;

const TopTitle = styled.h3`
  color: white;
  font-size: 16px;
  font-weight: bold;
  margin-left: 4px;
`;


const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(0, 0, 0, 0.4);
  backdrop-filter: blur(5px);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const GameCard = styled.div`
  min-width: 142px;
  height: 195px;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    ${GameOverlay} {
      opacity: 1;
    }
    transform: translateY(-5px);
  }

  @media (max-width: 768px) {
    min-width: 115px;
    height: 160px;
  }
`;

const GameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(255, 255, 255, 0.3);
  display: flex;
  justify-content: center;
  align-items: center;
`;
const Logo = styled.div`
  img {
    height: 40px;
  }
`
const GridWrapper = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(142px, 1fr));
  gap: 16px;
  padding: 0 10px;
  
  @media (max-width: 768px) {
    grid-template-columns: repeat(auto-fill, minmax(115px, 1fr));
  }
`;

const SkeletonLoader = styled.div`
  width: 100%;
  height: 100%;
  background: linear-gradient(90deg, #333 25%, #444 50%, #333 75%);
  background-size: 200% 100%;
  animation: loading 1.5s infinite;
  border-radius: 8px;
  
  @keyframes loading {
    0% {
      background-position: 200% 0;
    }
    100% {
      background-position: -200% 0;
    }
  }
`;

const PaginationContainer = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  margin-top: 20px;
  gap: 15px;
`;

const PaginationButton = styled.button`
  display: flex;
  align-items: center;
  gap: 5px;
  padding: 8px 15px;
  background-color: rgba(255, 255, 255, 0.1);
  border: none;
  border-radius: 4px;
  color: white;
  cursor: pointer;
  transition: all 0.3s ease;
  
  &:hover:not(:disabled) {
    background-color: rgba(255, 255, 255, 0.2);
  }
  
  &:disabled {
    opacity: 0.5;
    cursor: not-allowed;
  }
`;

const PaginationInfo = styled.div`
  font-size: 14px;
  color: rgba(255, 255, 255, 0.7);
`;

const Spinner = styled.div`
  width: ${props => props.size === "sm" ? "16px" : "24px"};
  height: ${props => props.size === "sm" ? "16px" : "24px"};
  border: 2px solid rgba(255, 255, 255, 0.3);
  border-top: 2px solid white;
  border-radius: 50%;
  animation: spin 1s linear infinite;
  
  @keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
  }
`;

const showNotification = (message, type = 'error') => {
  if (window.showToast) {
    window.showToast(message, type);
  } else {
    console[type === 'error' ? 'error' : 'log'](message);
  }
};

// Add these new styled components for the iframe
const FullScreenContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background: #000;
  z-index: 9999;
  display: flex;
  justify-content: center;
  align-items: center;
`;

const FullScreenIframe = styled.iframe`
  width: 100%;
  height: 100%;
  border: none;
  display: ${props => props.loading ? 'none' : 'block'};
`;

const BackButton = styled.button`
  position: absolute;
  top: 20px;
  left: 20px;
  background: rgba(0, 0, 0, 0.5);
  color: white;
  border: none;
  border-radius: 50%;
  width: 40px;
  height: 40px;
  display: flex;
  justify-content: center;
  align-items: center;
  cursor: pointer;
  z-index: 10000;
  
  &:hover {
    background: rgba(0, 0, 0, 0.8);
  }
`;
class EvolutionGamingByProvider extends Component {
  constructor(props) {
    super(props);
    this.state = {
      loading: true,
      games: [],
      loadedImages: new Set(),
      pagination: {
        currentPage: 1,
        totalPages: 1,
        loading: false
      },
      showIframe: false,
      iframeUrl: '',
      iframeLoading: false, // Changed initial value to false
      message: null
    };
  }

  componentDidMount() {
    this.fetchEvolutionGames();
  }

  fetchEvolutionGames = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('https://api.boss707.com/jsGames/games?vendor=evolution&page=1&per_page=20', {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });

      if (!response.ok) {
        throw new Error(`Network response was not ok: ${response.status}`);
      }

      const result = await response.json();

      if (result.success && result.payload && result.payload.games) {
        this.setState({
          games: result.payload.games,
          pagination: {
            currentPage: result.payload.pagination?.current_page || 1,
            totalPages: result.payload.pagination?.total_pages || 1,
            loading: false
          },
          loading: false
        });
      } else {
        throw new Error('Invalid API response structure');
      }
    } catch (error) {
      console.error("Error fetching evolution games:", error);
      this.setState({
        loading: false,
        message: { type: 'error', text: error.message }
      });
    }
  };

  handleGamePlay = async (game) => {
    const isLogged = localStorage.getItem('logged') === 'true';

    if (!isLogged) {
      sendNotfication('Please login to play', 'error');
      return;
    }

    this.setState({ loading: true, message: null });

    this.setState({ loading: true, message: null });

    try {
      // Get credits and coin type from localStorage
      const credits = localStorage.getItem('credit') || '0';
      const coins = localStorage.getItem('coin') || 'USDT';
      if (coins !== 'USDT' && coins !== 'INR') {
        sendNotfication("Please Swap Into USDT To Play", "error");
        this.setState({ loading: false });
        return;
      }
      // Prepare the launch request payload
      const requestPayload = {
        game_uid: game.game_uid,
        user_id: getUID(),
        credit_amount: credits,
        currency_code: coins,
        language: "en",
      };
      const token = localStorage.getItem('token');
      // Make the API call to launch the game
      console.log(token);
      const response = await fetch('https://api.boss707.com/jsGamesv2/launch', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify(requestPayload),
      });

      // Check if the response is successful
      if (!response.ok) {
        // throw new Error("Network response was not ok");
        sendNotfication("Game under maintenance", "error")
      }

      // Parse the response
      const result = await response.json();

      // Check if game launch was successful
      if (result.success && result.data.game_launch_url) {
        this.setState({
          iframeUrl: result.data.game_launch_url,
          showIframe: true,
          iframeLoading: true,
        });
      } else {
        // Throw an error if game launch failed
        // throw new Error("Failed to launch game: " + (result.message || "Unknown error"));
      }
    } catch (error) {
      // Handle any errors during the game launch process
      this.setState({
        message: {
          type: 'error',
          text: error.message
        }
      });
      sendNotfication(error.message, "error")
    } finally {
      // Reset loading state
      this.setState({ loading: false });

      // Clear message after 5 seconds
      setTimeout(() => {
        this.setState({ message: null });
      }, 5000);
    }
  };

  handleBackButton = () => {
    this.setState({
      showIframe: false,
      iframeUrl: '',
      iframeLoading: false
    });
  };

  handleIframeLoad = () => {
    this.setState({ iframeLoading: false });
  };

  handleImageLoad = (gameId) => {
    this.setState(prevState => ({
      loadedImages: new Set([...prevState.loadedImages]).add(gameId)
    }));
  };

  handlePagination = async (direction) => {
    const { pagination } = this.state;
    if (pagination.loading) return;

    const targetPage = direction === 'next'
      ? pagination.currentPage + 1
      : pagination.currentPage - 1;

    if (targetPage < 1 || targetPage > pagination.totalPages) return;

    try {
      this.setState({
        pagination: { ...pagination, loading: true }
      });

      const token = localStorage.getItem('token');
      const response = await fetch(`https://api.boss707.com/jsGames/games?vendor=evolution&page=${targetPage}&per_page=20`, {
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });

      if (!response.ok) {
        throw new Error(`Network response was not ok: ${response.status}`);
      }

      const result = await response.json();

      if (result.success && result.payload && result.payload.games) {
        this.setState({
          games: result.payload.games,
          pagination: {
            currentPage: result.payload.pagination?.current_page || targetPage,
            totalPages: result.payload.pagination?.total_pages || pagination.totalPages,
            loading: false
          }
        });
      }
    } catch (error) {
      console.error("Error fetching evolution games:", error);
      this.setState({
        pagination: { ...pagination, loading: false },
        message: { type: 'error', text: error.message }
      });
    }
  };

  showNotification = (message, type = 'error') => {
    if (window.showToast) {
      window.showToast(message, type);
    } else {
      console[type === 'error' ? 'error' : 'log'](message);
    }
  };

  render() {
    const { loading, games, loadedImages, pagination, showIframe, iframeUrl, iframeLoading, message } = this.state;

    const imageStyle = {
      position: 'absolute',
      top: '50%',
      left: '50%',
      transform: 'translate(-50%, -50%)',
      width: '250px',
      animation: 'fadeinout 2s infinite',
    };

    const animationStyle = `
      @keyframes fadeinout {
        0%, 100% { opacity: 0; }
        50% { opacity: 1; }
      }
    `;

    if (showIframe) {
      return (
        <FullScreenContainer>
          <style>{animationStyle}</style>
          {iframeLoading && (
            <img src="./assets/boss707.png" alt="Loading" style={imageStyle} />
          )}
          <FullScreenIframe
            src={iframeUrl}
            title="Evolution Game"
            onLoad={this.handleIframeLoad}
            loading={iframeLoading}
            allowFullScreen
          />
          <BackButton onClick={this.handleBackButton}>
            <FontAwesomeIcon icon={faArrowLeft} />
          </BackButton>
        </FullScreenContainer>
      );
    }

    const renderGames = () => {
      if (loading) {
        return Array(8).fill().map((_, index) => (
          <GameCard key={`skeleton-${index}`}>
            <SkeletonLoader />
          </GameCard>
        ));
      }

      if (games.length === 0) {
        return (
          <div style={{ width: '100%', textAlign: 'center', padding: '30px', color: 'white' }}>
            No games found
          </div>
        );
      }

      return games.map((game, index) => (
        <GameCard
          key={`game-${game.id || index}`}
          onClick={() => this.handleGamePlay(game)}
        >
          {!loadedImages.has(game.id) && <SkeletonLoader />}
          <GameImage
            onLoad={() => this.handleImageLoad(game.id)}
            style={{
              display: loadedImages.has(game.id) ? 'block' : 'none'
            }}
            src={game.game_icon || game.image}
            alt={game.name || `Game ${index}`}
          />
          <GameOverlay>
            <PlayIconContainer>
              <Play size={24} color="#fff" />
            </PlayIconContainer>
          </GameOverlay>
        </GameCard>
      ));
    };

    return (
      <div className="evolution-gaming-section">



        <TopTitle>Evolution Gaming</TopTitle>


        <GridWrapper>
          {renderGames()}
        </GridWrapper>

        <PaginationContainer>
          <PaginationButton
            onClick={() => this.handlePagination('prev')}
            disabled={pagination.currentPage <= 1 || pagination.loading}
          >
            <ChevronLeft size={16} /> Previous
          </PaginationButton>

          <PaginationInfo>
            Page {pagination.currentPage} of {pagination.totalPages}
          </PaginationInfo>

          <PaginationButton
            onClick={() => this.handlePagination('next')}
            disabled={pagination.currentPage >= pagination.totalPages || pagination.loading}
          >
            Next <ChevronRight size={16} />
          </PaginationButton>

          {pagination.loading && <Spinner size="sm" />}
        </PaginationContainer>

        {message && (
          <div style={{
            position: 'fixed',
            bottom: '20px',
            left: '50%',
            transform: 'translateX(-50%)',
            padding: '10px 20px',
            background: message.type === 'error' ? '#dc3545' : '#28a745',
            color: 'white',
            borderRadius: '4px',
            zIndex: 10000
          }}>
            {message.text}
          </div>
        )}
      </div>
    );
  }
}

export default withRouter(EvolutionGamingByProvider);