// BetHistoryModal.js
import React, { useState, useEffect } from 'react';
import styled from 'styled-components';
import { Modal } from 'react-bootstrap';
import { X, Search, Calendar, Filter, FileDown } from 'lucide-react';
import { getUID } from '../../../../Helper';


const BetHistoryModal = ({ show, onHide }) => {
  const [betHistory, setBetHistory] = useState({ transactions: [], pagination: {} });
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [filters, setFilters] = useState({
    gameTitle: '',
    vendor: '',
    dateFrom: '',
    dateTo: '',
    status: '',
    transactionType: ''
  });
  const [showFilters, setShowFilters] = useState(false);

  const fetchBetHistory = async (page = 1, limit = 10) => {
    setLoading(true);
    try {
      // Construct the query parameters
      const queryParams = new URLSearchParams({
        page,
        limit,
        ...Object.fromEntries(
          Object.entries({
            game_title: filters.gameTitle,
            game_vendor: filters.vendor,
            date_from: filters.dateFrom,
            date_to: filters.dateTo,
            transaction_status: filters.status,
            transaction_type: filters.transactionType
          }).filter(([_, value]) => value !== '')
        )
      });
      
      if (searchTerm) {
        queryParams.append('search', searchTerm);
      }
      let id = getUID();
      const token = localStorage.getItem('token');
      const response = await fetch(
        `https://api.boss707.com/betHistory/transactions/user/${id}?${queryParams}`,
        {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`,
            // Add any authentication headers if needed
          }
        }
      );
      
      if (!response.ok) {
        throw new Error('Failed to fetch bet history');
      }
      
      const data = await response.json();
      setBetHistory(data);
    } catch (error) {
      console.error('Error fetching bet history:', error);
    } finally {
      setLoading(false);
    }
  };

// Add debounce hook
const useDebounce = (value, delay) => {
  const [debouncedValue, setDebouncedValue] = useState(value);
  useEffect(() => {
    const handler = setTimeout(() => {
      setDebouncedValue(value);
    }, delay);
    return () => clearTimeout(handler);
  }, [value, delay]);
  return debouncedValue;
};

// In BetHistoryModal
const debouncedSearchTerm = useDebounce(searchTerm, 500);

useEffect(() => {
  if (show) {
    fetchBetHistory(currentPage);
  }
}, [show, currentPage, debouncedSearchTerm, filters]);

  const handlePageChange = (newPage) => {
    setCurrentPage(newPage);
  };

  const handleFilterChange = (e) => {
    const { name, value } = e.target;
    setFilters(prev => ({ ...prev, [name]: value }));
  };

  const handleSearch = (e) => {
    e.preventDefault();
    setSearchTerm(searchTerm.trim());
    setCurrentPage(1);
  };
  
  const applyFilters = () => {
    setCurrentPage(1);
  };

  const resetFilters = () => {
    setFilters({
      gameTitle: '',
      vendor: '',
      dateFrom: '',
      dateTo: '',
      status: '',
      transactionType: ''
    });
    setSearchTerm('');
    setCurrentPage(1);
    fetchBetHistory(1);
  };

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleString();
  };

  const getStatusColor = (status) => {
    switch(status) {
      case 'ERROR': return '#FF5252';
      case 'SUCCESS': return '#4CAF50';
      default: return '#FFC107';
    }
  };

  return (
    <StyledModal show={show} onHide={onHide} centered>
      <ModalHeader>
        <div className="header-title">Bet History</div>
        <button className="close-button" onClick={onHide}>
          <X size={20} color="white" />
        </button>
      </ModalHeader>
      <ModalBody>
        <SearchContainer>
          <SearchForm onSubmit={handleSearch}>
            <SearchInput
              type="text"
              placeholder="Search by game title, transaction ID..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <SearchButton type="submit">
              <Search size={16} color="white" />
            </SearchButton>
          </SearchForm>
          <FilterButton onClick={() => setShowFilters(!showFilters)}>
            <Filter size={16} color="white" />
          </FilterButton>
        </SearchContainer>

        {showFilters && (
          <FiltersContainer>
            {/* <FilterRow>
              <FilterColumn>
                <Label>Game Title</Label>
                <FilterInput
                  type="text"
                  name="gameTitle"
                  value={filters.gameTitle}
                  onChange={handleFilterChange}
                  placeholder="Game title"
                />
              </FilterColumn>
              <FilterColumn>
                <Label>Game Type</Label>
                <FilterInput
                  type="text"
                  name="type"
                  value={filters.vendor}
                  onChange={handleFilterChange}
                  placeholder="Type"
                />
              </FilterColumn>
            </FilterRow> */}
            <FilterRow>
              <FilterColumn>
                <Label>From Date</Label>
                <FilterInput
                  type="date"
                  name="dateFrom"
                  value={filters.dateFrom}
                  onChange={handleFilterChange}
                />
              </FilterColumn>
              <FilterColumn>
                <Label>To Date</Label>
                <FilterInput
                  type="date"
                  name="dateTo"
                  value={filters.dateTo}
                  onChange={handleFilterChange}
                />
              </FilterColumn>
            </FilterRow>
            <FilterRow>
              <FilterColumn>
                <Label>Status</Label>
                <FilterSelect
                  name="status"
                  value={filters.status}
                  onChange={handleFilterChange}
                >
                  <option value="">All</option>
                  <option value="SUCCESS">Success</option>
                  <option value="ERROR">Error</option>
                  <option value="PENDING">Pending</option>
                </FilterSelect>
              </FilterColumn>
              <FilterColumn>
                <Label>Transaction Type</Label>
                <FilterSelect
                  name="transactionType"
                  value={filters.transactionType}
                  onChange={handleFilterChange}
                >
                  <option value="">All</option>
                  <option value="BET">Bet</option>
                  <option value="WIN">Win</option>
                  <option value="REFUND">Refund</option>
                </FilterSelect>
              </FilterColumn>
            </FilterRow>
            <FilterButtonRow>
              <FilterActionButton onClick={applyFilters}>Apply Filters</FilterActionButton>
              <FilterActionButton className="reset" onClick={resetFilters}>Reset</FilterActionButton>
            </FilterButtonRow>
          </FiltersContainer>
        )}

        {loading ? (
          <LoadingContainer>
            <LoadingSpinner />
            <p>Loading bet history...</p>
          </LoadingContainer>
        ) : betHistory.transactions.length === 0 ? (
          <EmptyState>
            <p>No bet history found</p>
          </EmptyState>
        ) : (
          <>
            <TableContainer>
              <ResponsiveTable>
                <thead>
                  <tr>
                    <th>Game</th>
                    <th>Amount</th>
                    <th>Type</th>
                    <th>Date</th>
                  </tr>
                </thead>
                <tbody>
                  {betHistory.transactions.map((bet) => (
                    <tr key={bet.id}>
                      <td>
                        <GameInfo>
                          <GameTitle>{bet.game_title}</GameTitle>
                          <VendorName>{bet.game_vendor}</VendorName>
                        </GameInfo>
                      </td>
                      <td>
  <Amount type={bet.transaction_type}>
    {bet.transaction_type === 'WIN' ? '+' : ''}
    {bet.transaction_type === 'LOSS' ? '-' : ''}
    {bet.amount} {bet.currency_code}
  </Amount>
</td>
                   <td>
                        <TransactionType>{bet.transaction_type}</TransactionType>
                      </td>
                      <td>
                        <TransactionDate>
                          {formatDate(bet.transaction_timestamp)}
                        </TransactionDate>
                      </td>
                    
                    </tr>
                  ))}
                </tbody>
              </ResponsiveTable>
            </TableContainer>

            <PaginationContainer>
              <PaginationInfo>
                Showing {betHistory.transactions.length} of {betHistory.pagination.totalCount} entries
              </PaginationInfo>
              <PaginationControls>
                <PaginationButton 
                  disabled={currentPage === 1}
                  onClick={() => handlePageChange(currentPage - 1)}
                >
                  Previous
                </PaginationButton>
                <PageInfo>
                  Page {currentPage} of {betHistory.pagination.totalPages}
                </PageInfo>
                <PaginationButton 
                  disabled={currentPage === betHistory.pagination.totalPages}
                  onClick={() => handlePageChange(currentPage + 1)}
                >
                  Next
                </PaginationButton>
              </PaginationControls>
            </PaginationContainer>
          </>
        )}
      </ModalBody>
    </StyledModal>
  );
};

// Styled Components


const Amount = styled.div`
  font-weight: 600;
  color: ${props => 
    props.type === 'LOSS' ? '#FF5252' : 
    props.type === 'WIN' ? '#4CAF50' : 
    'inherit'
  };
`;


const StyledModal = styled(Modal)`
  .modal-content {
    background-color: #232626;
    border-radius: 15px;
    border: none;
    color: white;
    max-width: 900px;
    width: 95%;
    margin: 0 auto;
    max-height: 85vh;
  }
  
  @media (max-width: 768px) {
    .modal-dialog {
      margin: 0.5rem;
      max-width: 100%;
    }
  }
`;

const ModalHeader = styled.div`
  border-bottom: none;
  padding: 15px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  flex-shrink: 0;
  background: rgb(30, 35, 40);
  position: relative;
  
  .close-button {
    background: none;
    border: none;
    color: white;
    cursor: pointer;
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
  }
  
  .header-title {
    width: 100%;
    text-align: center;
    font-size: 18px;
    font-weight: bold;
    color: white;
  }
`;

const ModalBody = styled(Modal.Body)`
  padding: 15px;
  background-color: #03030F;
  max-height: calc(85vh - 60px);
  overflow-y: auto;
`;

const SearchContainer = styled.div`
  display: flex;
  margin-bottom: 15px;
  gap: 10px;
`;

const SearchForm = styled.form`
  display: flex;
  flex: 1;
  position: relative;
`;

const SearchInput = styled.input`
  width: 100%;
  padding: 10px 40px 10px 15px;
  background: rgba(0, 0, 0, 0.2);
  border: 1px solid #3a3a3a;
  border-radius: 20px;
  color: #ffffff;
  font-size: 14px;
  
  &:focus {
    outline: none;
    border-color: #f09c06;
  }
`;

const SearchButton = styled.button`
  position: absolute;
  right: 10px;
  top: 50%;
  transform: translateY(-50%);
  background: none;
  border: none;
  cursor: pointer;
`;

const FilterButton = styled.button`
  background: rgba(0, 0, 0, 0.2);
  border: 1px solid #3a3a3a;
  border-radius: 50%;
  width: 40px;
  height: 40px;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  
  &:hover {
    background: rgba(0, 0, 0, 0.3);
  }
`;

const FiltersContainer = styled.div`
  background: rgba(0, 0, 0, 0.15);
  border-radius: 10px;
  padding: 15px;
  margin-bottom: 20px;
`;

const FilterRow = styled.div`
  display: flex;
  gap: 15px;
  margin-bottom: 15px;
  
  @media (max-width: 600px) {
    flex-direction: column;
    gap: 10px;
  }
`;

const FilterColumn = styled.div`
  flex: 1;
`;

const Label = styled.label`
  display: block;
  margin-bottom: 5px;
  font-size: 13px;
  color: #a0a0a0;
`;

const FilterInput = styled.input`
  width: 100%;
  padding: 8px 12px;
  background: rgba(0, 0, 0, 0.2);
  border: 1px solid #3a3a3a;
  border-radius: 10px;
  color: #ffffff;
  font-size: 13px;
  
  &:focus {
    outline: none;
    border-color: #f09c06;
  }
`;

const FilterSelect = styled.select`
  width: 100%;
  padding: 8px 12px;
  background: rgba(0, 0, 0, 0.2);
  border: 1px solid #3a3a3a;
  border-radius: 10px;
  color: #ffffff;
  font-size: 13px;
  
  option {
    background: #232626;
  }
  
  &:focus {
    outline: none;
    border-color: #f09c06;
  }
`;

const FilterButtonRow = styled.div`
  display: flex;
  justify-content: flex-end;
  gap: 10px;
  margin-top: 5px;
`;

const FilterActionButton = styled.button`
  background: rgb(30, 35, 40);
  color: white;
  font-weight: bold;
  padding: 8px 15px;
  border-radius: 20px;
  border: none;
  cursor: pointer;
  font-size: 13px;
  
  &.reset {
    background: #373E3F;
  }
  
  &:hover {
    opacity: 0.9;
  }
`;

const TableContainer = styled.div`
  margin-bottom: 20px;
  overflow-x: auto;
`;

const ResponsiveTable = styled.table`
  width: 100%;
  border-collapse: collapse;
  
  th, td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
  }
  
  th {
    background-color: rgba(0, 0, 0, 0.2);
    font-size: 13px;
    font-weight: 600;
    color: #a0a0a0;
  }
  
  tr:hover td {
    background-color: rgba(0, 0, 0, 0.1);
  }
  
  @media (max-width: 600px) {
    display: block;
    
    thead, tbody, th, td, tr {
      display: block;
    }
    
    thead tr {
      position: absolute;
      top: -9999px;
      left: -9999px;
    }
    
    tr {
      margin-bottom: 15px;
      border: 1px solid rgba(255, 255, 255, 0.1);
      border-radius: 8px;
      background-color: rgba(0, 0, 0, 0.1);
    }
    
    td {
      border: none;
      border-bottom: 1px solid rgba(255, 255, 255, 0.05);
      position: relative;
      padding-left: 50%;
      
      &:before {
        position: absolute;
        top: 12px;
        left: 12px;
        width: 45%;
        white-space: nowrap;
        font-weight: 600;
        color: #a0a0a0;
      }
      
      &:nth-of-type(1):before { content: "Game"; }
      &:nth-of-type(2):before { content: "Amount"; }
      &:nth-of-type(3):before { content: "Type"; }
      &:nth-of-type(4):before { content: "Date"; }
      &:nth-of-type(5):before { content: "Status"; }
    }
  }
`;

const GameInfo = styled.div`
  display: flex;
  flex-direction: column;
`;

const GameTitle = styled.div`
  font-weight: 600;
  font-size: 14px;
`;

const VendorName = styled.div`
  font-size: 12px;
  color: #a0a0a0;
  text-transform: capitalize;
`;


const TransactionType = styled.div`
  background-color: rgba(0, 0, 0, 0.2);
  display: inline-block;
  padding: 3px 8px;
  border-radius: 4px;
  font-size: 12px;
`;

const TransactionDate = styled.div`
  font-size: 13px;
`;

const StatusBadge = styled.div`
  display: inline-block;
  padding: 3px 8px;
  border-radius: 4px;
  font-size: 12px;
  font-weight: 600;
  background-color: ${props => props.color || '#FFC107'};
  color: white;
`;

const PaginationContainer = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-top: 20px;
  
  @media (max-width: 600px) {
    flex-direction: column;
    gap: 15px;
  }
`;

const PaginationInfo = styled.div`
  font-size: 13px;
  color: #a0a0a0;
`;

const PaginationControls = styled.div`
  display: flex;
  align-items: center;
  gap: 10px;
`;

const PaginationButton = styled.button`
  background: rgba(0, 0, 0, 0.2);
  border: 1px solid #3a3a3a;
  color: white;
  padding: 5px 12px;
  border-radius: 5px;
  cursor: pointer;
  font-size: 13px;
  
  &:disabled {
    opacity: 0.5;
    cursor: not-allowed;
  }
  
  &:hover:not(:disabled) {
    background: rgba(0, 0, 0, 0.3);
  }
`;

const PageInfo = styled.div`
  font-size: 13px;
`;

const LoadingContainer = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 40px;
  
  p {
    margin-top: 15px;
    color: #a0a0a0;
  }
`;

const LoadingSpinner = styled.div`
  border: 3px solid rgba(255, 255, 255, 0.1);
  border-top: 3px solid #f09c06;
  border-radius: 50%;
  width: 30px;
  height: 30px;
  animation: spin 1s linear infinite;
  
  @keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
  }
`;

const EmptyState = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 40px;
  font-size: 16px;
  color: #a0a0a0;
`;

export default BetHistoryModal;