"use client";

import React, { useEffect, useState } from "react";
import styled from "styled-components";
import { format, isValid } from "date-fns";
import { ArrowDown, ArrowUp, Minus } from "lucide-react";
import { getUID } from "../../../../../Helper";

// --- Styled components (unchanged) ---
const Table = styled.table`
  width: 100%;
  border-collapse: collapse;
  font-family: 'Montserrat Alternates', sans-serif;
  background: #000000;
`;

const TableHeader = styled.thead`
  background-color: rgb(50, 55, 56);
  color: rgb(240, 156, 6);
`;

const TableHeaderCell = styled.th`
  padding: 12px 16px;
  text-align: left;
  font-weight: 500;
  font-size: 14px;
  font-family: 'Montserrat Alternates', sans-serif;
`;

const TableRow = styled.tr`
  border-bottom: 1px solid rgb(50, 55, 56);
  background-color: ${(props) => {
    switch (props.status) {
      case "won":
        return "#1a2a1a";
      case "lost":
        return "#2a1a1a";
      default:
        return "#000000";
    }
  }};

  &:hover {
    background-color: ${(props) => {
      switch (props.status) {
        case "won":
          return "#2a3a2a";
        case "lost":
          return "#3a2a2a";
        default:
          return "rgb(50, 55, 56)";
      }
    }};
  }
`;

const TableCell = styled.td`
  padding: 12px 16px;
  font-size: 14px;
  color: #ffffff;
  font-family: 'Montserrat Alternates', sans-serif;
`;

const BetType = styled.span`
  padding: 4px 8px;
  border-radius: 4px;
  font-size: 12px;
  font-weight: 500;
  background-color: ${(props) => {
    switch (props.type) {
      case "back":
        return "#e3f2fd";
      case "lay":
        return "#fce4ec";
      default:
        return "#f3e5f5";
    }
  }};
  color: ${(props) => {
    switch (props.type) {
      case "back":
        return "#1565c0";
      case "lay":
        return "#c2185b";
      default:
        return "#7b1fa2";
    }
  }};
`;

const Status = styled.span`
  padding: 4px 8px;
  border-radius: 4px;
  font-size: 12px;
  font-weight: 500;
  background-color: ${(props) => {
    switch (props.status) {
      case "won":
        return "#1a2a1a";
      case "lost":
        return "#2a1a1a";
      default:
        return "rgb(50, 55, 56)";
    }
  }};
  color: ${(props) => {
    switch (props.status) {
      case "won":
        return "#4caf50";
      case "lost":
        return "#f44336";
      default:
        return "#cccccc";
    }
  }};
`;

const GameType = styled.div`
  display: flex;
  align-items: center;
  gap: 8px;
  text-transform: capitalize;
  color: #ffffff;
`;

const BetDirection = styled.span`
  color: ${(props) => (props.type === "back" ? "#4caf50" : "#f44336")};
  display: inline-flex;
  align-items: center;
  gap: 4px;
`;

// --- Main Component ---
export default function BettingHistoryTable() {
  const [bets, setBets] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchBets = async () => {
      try {
        const uid = getUID();
        const token = localStorage.getItem("token");

        const res = await fetch(
          `https://api.boss707.com/sportsbetting/history/${uid}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              "Content-Type": "application/json",
            },
          }
        );

        const data = await res.json();
        if (data.success) {
          setBets(data.bets || []);
        } else {
          setBets([]);
        }
      } catch (error) {
        console.error("Error fetching bet history:", error);
        setBets([]);
      } finally {
        setLoading(false);
      }
    };

    fetchBets();
  }, []);

  if (loading) {
    return <div style={{ color: "#fff", padding: "20px" }}>Loading...</div>;
  }

  return (
    <Table>
      <TableHeader>
        <tr>
          <TableHeaderCell>Game</TableHeaderCell>
          <TableHeaderCell>Match</TableHeaderCell>
          <TableHeaderCell>Selection</TableHeaderCell>
          <TableHeaderCell>Type</TableHeaderCell>
          <TableHeaderCell>Odds</TableHeaderCell>
          <TableHeaderCell>Stake</TableHeaderCell>
          <TableHeaderCell>Liability</TableHeaderCell>
          <TableHeaderCell>Status</TableHeaderCell>
          <TableHeaderCell>Date</TableHeaderCell>
        </tr>
      </TableHeader>
      <tbody>
        {bets.length > 0 ? (
          bets.map((bet) => {
            // Fallbacks for missing fields
            const gameType = bet.game_type || "Unknown"; // Use provided game_type or fallback
            const matchTitle = bet.match_title || bet.match_id || "Unknown Match";
            const betType = bet.bet_type || "fancy"; // Use provided bet_type or fallback
            const matchStartTime = bet.match_start_time || bet.created_at; // Fallback to created_at
            const formattedDate = matchStartTime
              ? isValid(new Date(matchStartTime))
                ? format(new Date(matchStartTime), "MMM d, yyyy")
                : "N/A"
              : "N/A";

            return (
              <TableRow key={bet.id} status={bet.status}>
                <TableCell>
                  <GameType>
                    {matchTitle} ({gameType})
                  </GameType>
                </TableCell>
                <TableCell>{matchTitle}</TableCell>
                <TableCell>{bet.selection_name || "N/A"}</TableCell>
                <TableCell>
                  <BetType type={betType}>
                    {betType === "back" && (
                      <BetDirection type="back">Back</BetDirection>
                    )}
                    {betType === "lay" && (
                      <BetDirection type="lay">Lay</BetDirection>
                    )}
                    {betType === "fancy" && <span>Fancy</span>}
                  </BetType>
                </TableCell>
                <TableCell>{bet.odds || "N/A"}</TableCell>
                <TableCell>${bet.stake || "N/A"}</TableCell>
                <TableCell>${bet.liability || "N/A"}</TableCell>
                <TableCell>
                  <Status status={bet.status}>
                    {bet.status
                      ? bet.status.charAt(0).toUpperCase() + bet.status.slice(1)
                      : "Unknown"}
                  </Status>
                </TableCell>
                <TableCell>{formattedDate}</TableCell>
              </TableRow>
            );
          })
        ) : (
          <TableRow>
            <TableCell colSpan="9" style={{ textAlign: "center", color: "#ccc" }}>
              No bets placed
            </TableCell>
          </TableRow>
        )}
      </tbody>
    </Table>
  );
}