import React, { createContext, useContext, useState, useCallback, useEffect } from 'react';
import styled from 'styled-components';
import { faArrowLeft } from '@fortawesome/free-solid-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import storage from '../Storage';
import { getUID, sendNotfication } from '../Helper';
import { gamesService } from '../utils/Services';

const BackButton = styled.button`
  position: absolute;
  top: 10px;
  left: 10px;
  z-index: 1000;
  background: none;
  border: none;
  color: #fff;
  font-size: 24px;
  cursor: pointer;
  padding: 8px;
  
  &:hover {
    opacity: 0.8;
  }
`;

const FullScreenContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  z-index: 999;
  background-color: #1b2336;
`;

const FullScreenIframe = styled.iframe`
  width: 100%;
  height: 100%;
  border: none;
  display: ${(props) => (props.loading ? 'none' : 'block')};
`;

const GameContext = createContext();

const vendorDisplayNames = {
  'evolution': 'Evolution Gaming',
  'pragmatic': 'Pragmatic Slots',
  'pragmaticlive': 'Pragmatic Live',
  'spribe': 'Spribe',
  // 'ezugi': 'Ezugi',
  'ideal': 'Ideal',
  'microgaming': 'Microgaming',
  'pgsoft': 'PGSoft',
  'jili': 'Jili',
};

export const GameProvider = ({ children }) => {
  const [initialSearchResults, setInitialSearchResults] = useState([]); // Store default evolution games
  const [actualSearchResults, setActualSearchResults] = useState([]); // Store user-driven results
  const [searchResults, setSearchResults] = useState([]); // Final results to display
  const [iframeUrl, setIframeUrl] = useState('');
  const [logged, setLogged] = useState(localStorage.getItem('logged') === 'true');
  const [showIframe, setShowIframe] = useState(false);
  const [showIframeLoading, setShowIframeLoading] = useState(false);
  const [message, setMessage] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [hasUserInteracted, setHasUserInteracted] = useState(false); // Track user interaction

  // Fetch 10 pages of 'evolution' games on initial render
 useEffect(() => {
    const initializeDefaultGames = async () => {
      setIsLoading(true);
      try {
        const evolutionVendor = ['evolution'];
        const games = await fetchGamesFromAPI('', evolutionVendor, 1, 15);
        setInitialSearchResults(games);
        if (!hasUserInteracted) {
          setSearchResults(games); // Set initial results if no user interaction
        }
        setCurrentPage(1);
      } catch (error) {
        console.error('Error fetching default evolution games:', error);
        setInitialSearchResults([]);
        setSearchResults([]);
      } finally {
        setIsLoading(false);
      }
    };
    initializeDefaultGames();
  }, []);// Empty dependency array ensures this runs only once on mount

  const fetchGamesForVendors = async (vendorNames, page = 1, perPage = 15) => {
    if (vendorNames.length === 0) {
      setActualSearchResults([]);
      setSearchResults(initialSearchResults); // Revert to initial if no vendors
      setHasUserInteracted(false);
      return;
    }

    setIsLoading(true);
    try {
      const results = await Promise.all(
        vendorNames.map(vendor =>
          gamesService.getGamesByVendor(vendor, page, perPage)
        )
      );

      const games = results
        .filter(result => result.success && result.payload.games)
        .flatMap(result => result.payload.games)
        .map(game => ({
          id: game.id,
          title: game.game_name,
          vendor: game.vendor,
          game_icon: game.game_icon,
          game_type: game.game_type,
          game_uid: game.game_uid,
        }));

      setActualSearchResults(games);
      setSearchResults(games); // Update final results with user-driven results
      setCurrentPage(page);
      setHasUserInteracted(true); // Mark that user has interacted
    } catch (error) {
      console.error('Error fetching games for vendors:', error);
      setActualSearchResults([]);
      setSearchResults(initialSearchResults); // Revert to initial on error
    } finally {
      setIsLoading(false);
    }
  };

  const fetchGamesFromAPI = async (query, vendors, page = 1, perPage = 15) => {
    setIsLoading(true);
    try {
      const results = await Promise.all(
        vendors.map(vendor =>
          gamesService.getGamesByVendor(vendor, page, perPage)
        )
      );

      const games = results
        .filter(result => result.success && result.payload.games)
        .flatMap(result => result.payload.games)
        .map(game => ({
          id: game.id,
          title: game.game_name,
          vendor: game.vendor,
          game_icon: game.game_icon,
          game_type: game.game_type,
          game_uid: game.game_uid,
        }));

      setActualSearchResults(games);
      setSearchResults(games); // Update final results with user-driven results
      setCurrentPage(page);
      setHasUserInteracted(true); // Mark that user has interacted
      return games;
    } catch (error) {
      console.error('Error fetching games:', error);
      setActualSearchResults([]);
      setSearchResults(initialSearchResults); // Revert to initial on error
      return [];
    } finally {
      setIsLoading(false);
    }
  };

  const handleGameLaunchXGaming = async (game) => {
    const isLogged = localStorage.getItem('logged') === 'true';
    if (!isLogged) {
      sendNotfication('Please login to play', 'error');
      return;
    }



    try {
      const credits = localStorage.getItem('credit') || '0';
      const coins = localStorage.getItem('coin') || 'INR';
      const userId = getUID();

      if (coins !== 'USDT' && coins !== 'INR') {
        sendNotfication("Please Swap Into USDT To Play", "error");
        setShowIframe(false);
        setIframeUrl('');
        setShowIframeLoading(false);
        setIsLoading(false);
        return;
      }
      setIsLoading(true);
      setShowIframeLoading(true);
      setShowIframe(true);
      const requestPayload = {
        game_uid: game.game_uid,
        user_id: userId,
        credit_amount: credits,
        currency_code: coins,
        language: "en",
      };

      const result = await gamesService.launchGame(requestPayload);

      if (!result.success) {
        sendNotfication("Game under maintenance", "error");
        throw new Error("Network response was not ok");
      }
      if (result.success && result.data?.game_launch_url) {
        setIframeUrl(result.data.game_launch_url);
      } else {
        throw new Error(result.message || "Failed to launch game");
      }
    } catch (error) {
      console.error("Game launch error:", error);
      setMessage({ type: 'error', text: error.message || "Failed to launch game" });
      setShowIframe(false);
      setIframeUrl('');
      setShowIframeLoading(false);
      setTimeout(() => setMessage(null), 3000);
    } finally {
      setIsLoading(false);
    }
  };

  const handleBackButton = () => {
    setIframeUrl("");
    setShowIframe(false);
    setShowIframeLoading(false);
    window.location.reload();
  };

  const handleIframeLoaded = () => {
    setShowIframeLoading(false);
  };

  const searchGames = useCallback((query, vendors) => {
    fetchGamesFromAPI(query, vendors, 1);
  }, []);

  const loadMoreGames = () => {
    const nextPage = currentPage + 1;
    fetchGamesFromAPI('', Object.keys(vendorDisplayNames), nextPage);
  };

  if (showIframe) {
    return (
      <FullScreenContainer>
        {showIframeLoading && (
          <img
            src="assets/images/./assets/boss707.png"
            alt="Loading"
            style={{ position: 'absolute', top: '50%', left: '50%', transform: 'translate(-50%, -50%)', width: '250px' }}
          />
        )}
        <FullScreenIframe
          src={iframeUrl}
          title="Game Iframe"
          onLoad={handleIframeLoaded}
          loading={showIframeLoading}
        />
        <BackButton onClick={handleBackButton}>
          <FontAwesomeIcon icon={faArrowLeft} />
        </BackButton>
      </FullScreenContainer>
    );
  }

  return (
    <GameContext.Provider
      value={{
        searchResults,
        setSearchResults,
        isLoading,
        handleGameLaunchXGaming,
        fetchGamesForVendors,
        fetchGamesFromAPI,
        vendorDisplayNames,
        message,
        showIframe,
        handleBackButton,
        searchGames,
        loadMoreGames,
        currentPage,
      }}
    >
      {children}
    </GameContext.Provider>
  );
};

export const useGames = () => {
  const context = useContext(GameContext);
  if (!context) {
    throw new Error('useGames must be used within a GameProvider');
  }
  return context;
};