

import React, { createContext, useState, useCallback, useRef, useEffect } from 'react';
import { X } from 'lucide-react';

export const NotificationContext = createContext();

export const NotificationProvider = ({ children }) => {
  const [notifications, setNotifications] = useState([]);
  const notificationCounter = useRef(0);

  const generateUniqueId = useCallback(() => {
    notificationCounter.current += 1;
    return `${Date.now()}-${notificationCounter.current}`;
  }, []);

  const showNotification = useCallback((notification) => {
  
    
    const id = generateUniqueId();
    setNotifications(prev => [...prev, { ...notification, id }]);

    // Auto remove after 5 seconds
    setTimeout(() => {
      removeNotification(id);
    }, 5000);
  }, [generateUniqueId]);

  const removeNotification = useCallback((id) => {
    setNotifications(prev => prev.filter(notif => notif.id !== id));
  }, []);

  // Play sound when notifications count increases
  useEffect(() => {
    if (notifications.length > 0) {
      const audio = new Audio('/sounds/popup.mp3');
      audio.play().catch(error => console.error("Error playing audio:", error));
    }
  }, [notifications.length]);



  return (
    <NotificationContext.Provider 
      value={{
        showNotification,
        removeNotification
      }}
    >
      {children}
      {/* Toast Container - Make sure it's always on top */}
      <div 
        style={{
          position: 'fixed',
          top: '20px',
          right: '20px',
          zIndex: 9999,
          display: 'flex',
          flexDirection: 'column',
          gap: '10px'
        }}
      >
        {notifications.map(({ id, title, body }) => (
          <div
            key={id}
            style={{
              backgroundColor: 'white',
              borderRadius: '8px',
              boxShadow: '0 4px 6px rgba(0, 0, 0, 0.1)',
              padding: '16px',
              minWidth: '300px',
              maxWidth: '400px',
              animation: 'slideIn 0.3s ease-out',
              position: 'relative'
            }}
          >
            <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start' }}>
              <div>
                <h4 style={{ margin: 0, fontWeight: '600', color: '#1a1a1a' }}>{title}</h4>
                <p style={{ margin: '8px 0 0 0', color: '#666' }}>{body}</p>
              </div>
              <button
                onClick={() => removeNotification(id)}
                style={{
                  background: 'none',
                  border: 'none',
                  cursor: 'pointer',
                  padding: '4px'
                }}
              >
                <X size={16} color="#999" />
              </button>
            </div>
          </div>
        ))}
      </div>

      <style>
        {`
          @keyframes slideIn {
            from {
              transform: translateX(100%);
              opacity: 0;
            }
            to {
              transform: translateX(0);
              opacity: 1;
            }
          }
        `}
      </style>
    </NotificationContext.Provider>
  );
};