# API Services Documentation

This directory contains organized API services that handle all HTTP requests with automatic token authentication.

## Structure

```
Services/
├── apiService.js          # Base API service with token authentication
├── walletService.js       # Wallet, deposits, withdrawals, 2FA
├── notificationService.js # Firebase notifications
├── sportsService.js       # Sports betting APIs
├── gamesService.js        # Games and casino APIs
├── clubService.js         # Club membership APIs
├── affiliateService.js    # Affiliate system APIs
├── swapService.js         # Swap/exchange APIs
├── uiService.js           # UI and banner APIs
├── index.js              # Export all services
└── README.md             # This file
```

## Features

- **Automatic Token Authentication**: All requests automatically include the Bearer token from localStorage
- **Error Handling**: Centralized error handling with 401 redirect to login
- **Request/Response Interceptors**: Automatic token injection and error handling
- **Organized by Feature**: Each service handles related API endpoints

## Usage

### Import Services

```javascript
// Import specific service
import { walletService } from '../utils/Services';

// Or import all services
import Services from '../utils/Services';
```

### Example Usage

```javascript
import { walletService, notificationService } from '../utils/Services';

// Wallet operations
const coinDetails = await walletService.getCoinDetails(data);
const deposit = await walletService.createDeposit(payload);

// Notification operations
const unreadCount = await notificationService.getUnreadCount(userId);
await notificationService.markAsRead(data);

// Sports operations
import { sportsService } from '../utils/Services';
const matches = await sportsService.getAllMatches();
const odds = await sportsService.getMarketOdds(marketId);
```

## Token Management

The token is automatically:
- Retrieved from `localStorage.getItem('token')`
- Added to all requests as `Authorization: Bearer <token>`
- Handled for 401 errors (redirects to login)

## Migration Guide

### Before (Old way)
```javascript
const response = await fetch('https://api.boss707.com/getCoinDetails', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${localStorage.getItem('token')}`
  },
  body: JSON.stringify(data)
});
```

### After (New way)
```javascript
import { walletService } from '../utils/Services';
const response = await walletService.getCoinDetails(data);
```

## Benefits

1. **Cleaner Code**: No need to manually add headers and tokens
2. **Consistent Error Handling**: Centralized error management
3. **Better Organization**: Related APIs grouped together
4. **Easy Maintenance**: Single place to update API endpoints
5. **Type Safety**: Better IDE support and error catching
6. **Reusability**: Services can be used across components

## Adding New Services

1. Create a new service file (e.g., `newFeatureService.js`)
2. Import `apiService` from `./apiService`
3. Export methods using `apiService.get()`, `apiService.post()`, etc.
4. Add to `index.js` exports
5. Update this README

Example:
```javascript
import apiService from './apiService';

export const newFeatureService = {
  getData: async (id) => {
    return await apiService.get(`/newfeature/${id}`);
  },
  
  createData: async (data) => {
    return await apiService.post('/newfeature', data);
  }
};

export default newFeatureService;
``` 