// src/components/Layout.tsx
import React, { useEffect, useState } from "react";
import { Outlet, useLocation } from "react-router-dom";
import Sidebar from "./Sidebar";
import { Wallet } from "lucide-react";
import { apiFetch } from "../utils/api";

interface UserBar {
  name: string;
  role: string;
  balance: number;
}

const Layout = () => {
  const location                = useLocation();
  const [isSidebarOpen, setSB]  = useState(true);
  const [isMobile, setMobile]   = useState(window.innerWidth < 1024);
  const [user, setUser]         = useState<UserBar>({ name: "", role: "", balance: 0 });

  /* ──────────────────────────────  handle resize  */
  useEffect(() => {
    const onResize = () => setMobile(window.innerWidth < 1024);
    window.addEventListener("resize", onResize);
    return () => window.removeEventListener("resize", onResize);
  }, []);

  useEffect(() => { if (isMobile) setSB(false); }, [isMobile]);

  /* ──────────────────────────────  fetch /api/staff/:id  */
const fetchMe = async () => {
 const id = localStorage.getItem("currentUserId");
 if (!id) return;

 /* two small calls in parallel: ① who-am-I, ② my roll-up */
 const [info, roll] = await Promise.all([
   apiFetch<{ name:string; role:string }>(`/api/staff/${id}`),
   apiFetch<any>(`/api/staff/rollup/${id}`)
 ]);

 setUser({
   name   : info.name,
   role   : info.role,
   balance: Number(roll.data?.own_balance || 0)
 });
};
useEffect(() => { fetchMe(); }, []);
useEffect(() => {
  window.addEventListener("balance-changed", fetchMe);
  return () => window.removeEventListener("balance-changed", fetchMe);
}, []);

  /* ──────────────────────────────  helpers  */
  const pageTitle = (() => {
    const seg = location.pathname.split("/").filter(Boolean).pop() || "dashboard";
    return seg
      .replace(/-/g, " ")
      .replace(/([A-Z])/g, " $1")
      .replace(/^./, s => s.toUpperCase());
  })();

  /* ──────────────────────────────  render  */
  return (
    <div className="flex h-screen bg-[#0F1525] overflow-hidden">
      <Sidebar isOpen={isSidebarOpen} toggleSidebar={() => setSB(!isSidebarOpen)} />

      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Top bar */}
        <header className="bg-[#1A2033] border-b border-[#2D334A] p-4 flex justify-between items-center sticky top-0 z-10">
          <div className="flex items-center">
            <button
              className="lg:hidden text-gray-300 mr-4"
              onClick={() => setSB(!isSidebarOpen)}
            >
              <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" fill="none"
                   viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2}
                      d="M4 6h16M4 12h16M4 18h16"/>
              </svg>
            </button>
            <h1 className="text-white text-xl font-medium">{pageTitle}</h1>
                 
          </div>

          <div className="flex items-center space-x-4">
            {/* Balance */}
            <div className="flex items-center space-x-2 text-white bg-[#0F1525] px-3 py-1.5 rounded-lg">
              <Wallet className="w-5 h-5 text-yellow-400" />
              <span className="text-sm font-semibold">Wallet {user.balance.toFixed(2)}</span>
            </div>

            {/* User pill */}
            <div className="flex items-center space-x-3">
              <div className="h-8 w-8 rounded-full bg-[#646ECD] flex items-center justify-center text-white">
                <span className="text-sm font-medium">
                  {user.name ? user.name.charAt(0).toUpperCase() : "?"}
                </span>
              </div>
              <div>
                <h3 className="text-white text-sm font-semibold">{user.name || "..."}</h3>
                <p className="text-gray-400 text-xs">{user.role}</p>
              </div>
            </div>
          </div>
        </header>

        {/* Main area */}
        <main className="flex-1 overflow-auto bg-[#161B2B]">
          <div className="container mx-auto max-w-12xl">
            <Outlet />
          </div>
        </main>
      </div>
    </div>
  );
};

export default Layout;
